#include <solomon/services/dataproxy/lib/datasource/merge_sts_lts/find/merger.h>

#include <solomon/libs/cpp/string_pool/string_pool.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/builder.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace NMonitoring;
using namespace NMerger;

class TResultBuilder {
public:
    void AddMetric(EMetricType type) {
        Res_.Metrics.emplace_back();
        Res_.Metrics.back().Type = type;
    }

    void AddLabel(const char* key, const char* value) {
        Res_.Metrics.back().Labels.emplace_back(Builder_.Put(key), Builder_.Put(value));
    }

    void AddLabels(const std::vector<std::pair<const char*, const char*>>& labels) {
        for (auto [key, value]: labels) {
            AddLabel(key, value);
        }
    }

    void SetLtsId(TReplicaMap<TStockpileId> id) {
        Res_.Metrics.back().StockpileIds = id;
    }

    void Truncate() {
        Res_.Truncated = true;
    }

    void SetTotalCount(ui32 total) {
        Res_.TotalCount = total;
    }

    TFindResult Finish() {
        Res_.Strings = std::move(Builder_);
        if (!Res_.TotalCount) {
            Res_.TotalCount = Res_.Metrics.size();
        }
        return std::move(Res_);
    }

private:
    NStringPool::TStringPoolBuilder Builder_;

    TFindResult Res_;
};

TEST(TFindMerger, Common) {
    TFindMerger merger(10);

    TResultBuilder sts;
    sts.AddMetric(EMetricType::GAUGE);

    sts.AddLabels({{"sensor", "cpu_usage"}, {"service", "fetcher"}});

    sts.AddMetric(EMetricType::RATE);
    sts.AddLabels({{"sensor", "memory_usage"}, {"service", "fetcher"}});

    sts.SetTotalCount(5u);

    auto stsResp = sts.Finish();

    TResultBuilder lts;
    lts.AddMetric(EMetricType::GAUGE);
    lts.AddLabels({{"service", "fetcher"}, {"sensor", "cpu_usage"}});
    {
        TReplicaMap<TStockpileId> replicaMap;
        replicaMap[EReplica::R0] = TStockpileId{1, 1};
        replicaMap[EReplica::R1] = TStockpileId{2, 2};
        lts.SetLtsId(replicaMap);
    }
    lts.AddMetric(EMetricType::RATE);
    lts.AddLabels({{"sensor", "disk_usage"}, {"service", "fetcher"}});
    {
        TReplicaMap<TStockpileId> replicaMap;
        replicaMap[EReplica::R0] = TStockpileId{3, 3};
        replicaMap[EReplica::R1] = TStockpileId{4, 4};
        lts.SetLtsId(replicaMap);
    }
    lts.SetTotalCount(6u);

    auto ltsResp = lts.Finish();

    merger.AddResponse(std::move(stsResp), EStorageType::STS);
    merger.AddResponse(std::move(ltsResp), EStorageType::LTS);

    auto res = merger.Finish();

    ASSERT_EQ(3u, res->Metrics.size());
    EXPECT_EQ(6u, res->Strings.Size());
    EXPECT_EQ(6u, res->TotalCount);

    auto resStrings = res->Strings.Build();

    std::sort(res->Metrics.begin(), res->Metrics.end(), [&resStrings](const auto& lhs, const auto& rhs) {
        return resStrings[lhs.Labels[0].Value] < resStrings[rhs.Labels[0].Value];
    });

    {
        auto& m = res->Metrics[0];
        ASSERT_EQ(2u, m.Labels.size());
        EXPECT_EQ("sensor", resStrings[m.Labels[0].Key]);
        EXPECT_EQ("cpu_usage", resStrings[m.Labels[0].Value]);
        EXPECT_EQ("service", resStrings[m.Labels[1].Key]);
        EXPECT_EQ("fetcher", resStrings[m.Labels[1].Value]);

        EXPECT_EQ((TStockpileId{1, 1}), m.StockpileIds[EReplica::R0]);
        EXPECT_EQ((TStockpileId{2, 2}), m.StockpileIds[EReplica::R1]);
    }

    {
        auto& m = res->Metrics[1];
        ASSERT_EQ(2u, m.Labels.size());
        EXPECT_EQ("sensor", resStrings[m.Labels[0].Key]);
        EXPECT_EQ("disk_usage", resStrings[m.Labels[0].Value]);
        EXPECT_EQ("service", resStrings[m.Labels[1].Key]);
        EXPECT_EQ("fetcher", resStrings[m.Labels[1].Value]);

        EXPECT_EQ((TStockpileId{3, 3}), m.StockpileIds[EReplica::R0]);
        EXPECT_EQ((TStockpileId{4, 4}), m.StockpileIds[EReplica::R1]);
    }

    {
        auto& m = res->Metrics[2];
        ASSERT_EQ(2u, m.Labels.size());
        EXPECT_EQ("sensor", resStrings[m.Labels[0].Key]);
        EXPECT_EQ("memory_usage", resStrings[m.Labels[0].Value]);
        EXPECT_EQ("service", resStrings[m.Labels[1].Key]);
        EXPECT_EQ("fetcher", resStrings[m.Labels[1].Value]);

        auto id = TStockpileId{0, 0};

        EXPECT_EQ(id, m.StockpileIds[EReplica::R0]);
        EXPECT_EQ(id, m.StockpileIds[EReplica::R1]);
    }
}
