#include "merger.h"

namespace NSolomon::NDataProxy::NMerger {
TLabelValuesMerger::TLabelValuesMerger(ui32 valuesLimit)
    : ValuesLimit_(valuesLimit)
{
}

void TLabelValuesMerger::AddError(TDataSourceError error) {
    Errors_.emplace_back(std::move(error));
}

void TLabelValuesMerger::AddResponse(TLabelValuesResult&& res, EStorageType type) {
    if (!Project_) {
        Project_ = std::move(res.Project);
    }

    MetricCount_[ToUnderlying(type)] += res.MetricCount;
    auto resStrings = res.Strings.Build();

    for (auto& labelState: res.Labels) {
        auto [it, _] = State_.emplace(Strings_.Put(resStrings[labelState.Key]), TState());

        auto& state = it->second;

        state.Truncated |= labelState.Truncated;
        state.TotalCount[ToUnderlying(type)] += labelState.MetricCount;

        for (auto& value: labelState.Values) {
            if (state.Values.size() >= ValuesLimit_) {
                state.Truncated = true;
                break;
            }

            ui32 v = Strings_.Put(resStrings[value]);
            state.Values.emplace(v);
        }
    }

    Errors_.insert(Errors_.end(), res.Errors.begin(), res.Errors.end());
}

std::unique_ptr<TLabelValuesResult> TLabelValuesMerger::Finish() {
    auto res = std::make_unique<TLabelValuesResult>();
    res->Labels.reserve(State_.size());

    res->Project = std::move(Project_);
    res->Strings = std::move(Strings_);
    res->MetricCount = Max(
            MetricCount_[ToUnderlying(EStorageType::STS)],
            MetricCount_[ToUnderlying(EStorageType::LTS)],
            (ui32) State_.size());

    for (auto it = State_.begin(); it != State_.end(); ) {
        auto& state = res->Labels.emplace_back();
        auto node = State_.extract(it++);

        auto& labelState = node.mapped();

        state.MetricCount = Max(
                labelState.TotalCount[ToUnderlying(EStorageType::STS)],
                labelState.TotalCount[ToUnderlying(EStorageType::LTS)],
                (ui32) labelState.Values.size());

        state.Truncated = labelState.Truncated;
        state.Key = node.key();
        state.Values = TVector<ui32>(labelState.Values.begin(), labelState.Values.end());
    }

    res->Errors = std::move(Errors_);

    return res;
}

} // namespace NSolomon::NDataProxy::NMerger
