#include <solomon/services/dataproxy/lib/datasource/merge_sts_lts/label_values/merger.h>

#include <solomon/libs/cpp/string_pool/string_pool.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/builder.h>

#include <algorithm>

using namespace NSolomon;
using namespace NDataProxy;
using namespace NMonitoring;
using namespace NMerger;

TLabelValuesResult::TLabelState MakeLabelState(TStringBuf key, const TVector<TStringBuf>& values, NStringPool::TStringPoolBuilder& sb) {
    TLabelValuesResult::TLabelState state;
    state.Key = sb.Put(key);
    for (const auto& value: values) {
        state.Values.emplace_back(sb.Put(value));
    }

    state.Truncated = false;
    state.MetricCount = values.size();

    return state;
}

TEST(TLabelKeysMerger, Common) {
    TLabelValuesMerger merger(3u);

    TLabelValuesResult stsRes;
    TLabelValuesResult ltsRes;

    {
        NStringPool::TStringPoolBuilder sb;
        stsRes.MetricCount = 3u;
        stsRes.Labels.emplace_back(MakeLabelState("sensor", {"cpu_usage", "memory_usage"}, sb));
        stsRes.Labels.emplace_back(MakeLabelState("service", {"memstore", "fetcher"}, sb));

        stsRes.Strings = std::move(sb);
    }

    {
        NStringPool::TStringPoolBuilder sb;
        ltsRes.MetricCount = 3u;
        ltsRes.Labels.emplace_back(MakeLabelState("service", {"ingestor", "fetcher"}, sb));
        ltsRes.Labels.emplace_back(MakeLabelState("ctype", {"none"}, sb));

        ltsRes.Strings = std::move(sb);
    }

    merger.AddResponse(std::move(stsRes), EStorageType::STS);
    merger.AddResponse(std::move(ltsRes), EStorageType::LTS);

    auto res = merger.Finish();

    EXPECT_EQ(res->Strings.Size(), 9u);
    EXPECT_EQ(res->MetricCount, 3u);
    ASSERT_EQ(res->Labels.size(), 3u);

    auto resStrings = res->Strings.Build();

    std::sort(res->Labels.begin(), res->Labels.end(), [&resStrings](const auto& lhs, const auto& rhs) {
       return resStrings[lhs.Key] < resStrings[rhs.Key];
    });

    {
        auto& state = res->Labels[1];

        EXPECT_EQ(state.MetricCount, 2u);
        EXPECT_EQ(resStrings[state.Key], "sensor");

        ASSERT_EQ(state.Values.size(), 2u);

        std::sort(state.Values.begin(), state.Values.end(), [&resStrings](const auto& lhs, const auto& rhs) {
            return resStrings[lhs] < resStrings[rhs];
        });

        EXPECT_EQ(resStrings[state.Values[0]], "cpu_usage");
        EXPECT_EQ(resStrings[state.Values[1]], "memory_usage");
    }

    {
        auto& state = res->Labels[2];

        EXPECT_EQ(resStrings[state.Key], "service");

        ASSERT_EQ(state.Values.size(), 3u);

        std::sort(state.Values.begin(), state.Values.end(), [&resStrings](const auto& lhs, const auto& rhs) {
            return resStrings[lhs] < resStrings[rhs];
        });

        EXPECT_EQ(resStrings[state.Values[2]], "memstore");
        EXPECT_EQ(resStrings[state.Values[1]], "ingestor");
        EXPECT_EQ(resStrings[state.Values[0]], "fetcher");
    }

    {
        auto& state = res->Labels[0];

        EXPECT_EQ(resStrings[state.Key], "ctype");

        std::sort(state.Values.begin(), state.Values.end(), [&resStrings](const auto& lhs, const auto& rhs) {
            return resStrings[lhs] < resStrings[rhs];
        });

        ASSERT_EQ(state.Values.size(), 1u);
        EXPECT_EQ(resStrings[state.Values[0]], "none");
    }
}
