#include "merge_sts_lts.h"
#include "storage_type.h"

#include <solomon/services/dataproxy/lib/datasource/merge_sts_lts/find/merger.h>
#include <solomon/services/dataproxy/lib/datasource/merge_sts_lts/read_many/merger.h>
#include <solomon/services/dataproxy/lib/datasource/merge_sts_lts/unique_labels/merger.h>
#include <solomon/services/dataproxy/lib/datasource/merge_sts_lts/label_keys/merger.h>
#include <solomon/services/dataproxy/lib/datasource/merge_sts_lts/label_values/merger.h>

#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>
#include <solomon/libs/cpp/actors/events/events.h>

#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/event_local.h>

#include <array>

using namespace NActors;
using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy::NMerger {
namespace {

struct TPrivateEvent: private TPrivateEvents {
    enum {
        Response = SpaceBegin,
        Error,
        End
    };

    static_assert(End < SpaceEnd, "too many event types");

    template <class T>
    struct TSuccess: public TEventLocal<TSuccess<T>, Response> {
        TSuccess(std::unique_ptr<T> response, EStorageType type)
            : Response(std::move(response))
            , StorageType(type)
        {
        }

        std::unique_ptr<T> Response;
        EStorageType StorageType;
    };

    struct TError: public TEventLocal<TError, Error> {
        TError(TString&& project, EDataSourceStatus status, TString&& message, EStorageType type)
            : Project(std::move(project))
            , Status(status)
            , Message(std::move(message))
            , Type(type)
        {
        }

        TString Project;
        EDataSourceStatus Status;
        TString Message;
        EStorageType Type;
    };
};

template <typename T>
class THandlerActor : public TActor<THandlerActor<T>> {
    using TBase = TActor<THandlerActor<T>>;
public:
    THandlerActor(IResultHandlerPtr<T> handler)
        : TBase(&THandlerActor<T>::Handle)
        , Handler_(std::move(handler))
    {
    }

    STATEFN(Handle) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TPrivateEvent::TSuccess<T>, OnSuccess);
            hFunc(TPrivateEvent::TError, OnError);
        }
    }

    void OnSuccess(typename TPrivateEvent::TSuccess<T>::TPtr& ev) {
        TRACING_SPAN_END_EV(ev);
        Handler_->OnSuccess(std::move(ev->Get()->Response));
        TBase::PassAway();
    }

    void OnError(TPrivateEvent::TError::TPtr& ev) {
        TRACING_SPAN_END_EV(ev);
        auto& message = ev->Get()->Message;
        auto status = ev->Get()->Status;
        auto& project = ev->Get()->Project;
        Handler_->OnError(std::move(project), status, std::move(message));
        TBase::PassAway();
    }

    static constexpr char ActorName[] = "MergeStsLts/HandlerActor";

private:
    IResultHandlerPtr<T> Handler_;
};

template <typename T, typename TMerger>
class TRequester: public TActor<TRequester<T, TMerger>> {
    using TBase = TActor<TRequester<T, TMerger>>;

public:
    template <class... TArgs>
    TRequester(TActorRuntime& runtime, IResultHandlerPtr<T> handler, TArgs&&... args)
        : TBase(&TRequester<T, TMerger>::StateWork)
        , HandlerId_(runtime.Register(new THandlerActor<T>(std::move(handler))))
        , Merger_(std::forward<TArgs>(args)...)
    {
    }

    static constexpr char ActorName[] = "MergeStsLts";

private:
    void OnSuccess(typename TPrivateEvent::TSuccess<T>::TPtr& ev) {
        auto& r = *(ev->Get()->Response);
        auto t = ev->Get()->StorageType;

        Merger_.AddResponse(std::move(r), t);
        ++SuccessCount_;

        if (SuccessCount_ + ErrorCount_ == ToUnderlying(EStorageType::COUNT)) {
            TBase::Send(HandlerId_, new TPrivateEvent::TSuccess(Merger_.Finish(), t), 0, 0, std::move(ev->TraceId));
            TBase::PassAway();
        }
    }

    void OnError(TPrivateEvent::TError::TPtr& ev) {
        if (ev->Get()->Type == EStorageType::LTS) {
            ReportErrorAndPassAway(ev);
            return;
        }

        ++ErrorCount_;
        Merger_.AddError(TDataSourceError(ev->Get()->Status, std::move(ev->Get()->Message)));

        if (SuccessCount_) {
            TBase::Send(HandlerId_, new TPrivateEvent::TSuccess(Merger_.Finish(), ev->Get()->Type), 0, 0, std::move(ev->TraceId));
            TBase::PassAway();
        } else if (ErrorCount_ == ToUnderlying(EStorageType::COUNT)) {
            ReportErrorAndPassAway(ev);
        }
    }

    void ReportErrorAndPassAway(TPrivateEvent::TError::TPtr& ev) {
        TBase::Send(HandlerId_, ev->Release().Release(), 0, 0, std::move(ev->TraceId));
        TBase::PassAway();
    }

    STATEFN(StateWork) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TPrivateEvent::TSuccess<T>, OnSuccess);
            hFunc(TPrivateEvent::TError, OnError);
        }
    }

    size_t SuccessCount_ = 0;
    size_t ErrorCount_ = 0;

    TActorId HandlerId_;
    TMerger Merger_;
};

class TMergeShortLongTermsSource: public IDataSource {
public:
    TMergeShortLongTermsSource(TActorRuntime& runtime, IDataSourcePtr shortTerm, IDataSourcePtr longTerm)
        : Runtime_{runtime}
        , ShortTerm_{std::move(shortTerm)}
        , LongTerm_{std::move(longTerm)}
    {
    }

    bool CanHandle(const TQuery& /*query*/) const override {
//        if (query.Time.To < (TInstant::Now() - STS_STORAGE_PERIOD)) {
//            // time range outside of STS storage
//            return false;
//        }
        // TODO: return false for TResolveOneQuery and TResolveManyQuery
        return true;
    }

    void Find(TFindQuery query, IResultHandlerPtr<TFindResult> handler, TSpanId span) const override {
        using TResult = TFindResult;
        using TMerger = TFindMerger;

        auto actor = Runtime_.Register(new TRequester<TResult, TMerger>(Runtime_, std::move(handler), query.Limit));

        auto stsSpan = TRACING_NEW_SPAN_START(span, "STS-Find");
        auto stsHandler = CreateHandler<TResult>(actor, EStorageType::STS, TSpanId(span));
        ShortTerm_->Find(query, std::move(stsHandler), std::move(stsSpan));
        auto ltsSpan = TRACING_NEW_SPAN_START(span, "LTS-Find");
        auto ltsHandler = CreateHandler<TResult>(actor, EStorageType::LTS, TSpanId(span));
        LongTerm_->Find(std::move(query), std::move(ltsHandler), std::move(ltsSpan));
    }

    void ResolveOne(TResolveOneQuery query, IResultHandlerPtr<TResolveOneResult> handler, TSpanId) const override {
        // TODO: better error status
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "ResolveOne is not supported by merge STS-LTS datasource");
    }

    void ResolveMany(TResolveManyQuery query, IResultHandlerPtr<TResolveManyResult> handler, TSpanId) const override {
        // TODO: better error status
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "ResolveOne is not supported by merge STS-LTS datasource");
    }

    void MetricNames(TMetricNamesQuery query, IResultHandlerPtr<TMetricNamesResult> handler, TSpanId) const override {
        // TODO: better error status
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "MetricNames is not yet implemented in merge STS-LTS datasource");
    }

    void LabelKeys(TLabelKeysQuery query, IResultHandlerPtr<TLabelKeysResult> handler, TSpanId span) const override {
        using TResult = TLabelKeysResult;
        using TMerger = TLabelKeysMerger;

        auto actor = Runtime_.Register(new TRequester<TResult, TMerger>(Runtime_, std::move(handler)));

        auto stsSpan = TRACING_NEW_SPAN_START(span, "STS-LabelKeys");
        auto stsHandler = CreateHandler<TResult>(actor, EStorageType::STS, TSpanId(span));
        ShortTerm_->LabelKeys(query, std::move(stsHandler), std::move(stsSpan));
        auto ltsSpan = TRACING_NEW_SPAN_START(span, "LTS-LabelKeys");
        auto ltsHandler = CreateHandler<TResult>(actor, EStorageType::LTS, TSpanId(span));
        LongTerm_->LabelKeys(std::move(query), std::move(ltsHandler), std::move(ltsSpan));
    }

    void LabelValues(TLabelValuesQuery query, IResultHandlerPtr<TLabelValuesResult> handler, TSpanId span) const override {
        using TResult = TLabelValuesResult;
        using TMerger = TLabelValuesMerger;

        auto actor = Runtime_.Register(new TRequester<TResult, TMerger>(Runtime_, std::move(handler), query.Limit));

        auto stsSpan = TRACING_NEW_SPAN_START(span, "STS-LabelValues");
        auto stsHandler = CreateHandler<TResult>(actor, EStorageType::STS, TSpanId(span));
        ShortTerm_->LabelValues(query, std::move(stsHandler), std::move(stsSpan));
        auto ltsSpan = TRACING_NEW_SPAN_START(span, "LTS-LabelValues");
        auto ltsHandler = CreateHandler<TResult>(actor, EStorageType::LTS, TSpanId(span));
        LongTerm_->LabelValues(std::move(query), std::move(ltsHandler), std::move(ltsSpan));
    }

    void UniqueLabels(TUniqueLabelsQuery query, IResultHandlerPtr<TUniqueLabelsResult> handler, TSpanId span) const override {
        using TResult = TUniqueLabelsResult;
        using TMerger = TUniqueLabelsMerger;

        auto actor = Runtime_.Register(new TRequester<TResult, TMerger>(Runtime_, std::move(handler)));

        auto stsSpan = TRACING_NEW_SPAN_START(span, "STS-UniqueLabels");
        auto stsHandler = CreateHandler<TResult>(actor, EStorageType::STS, TSpanId(span));
        ShortTerm_->UniqueLabels(query, std::move(stsHandler), std::move(stsSpan));
        auto ltsSpan = TRACING_NEW_SPAN_START(span, "LTS-UniqueLabels");
        auto ltsHandler = CreateHandler<TResult>(actor, EStorageType::LTS, TSpanId(span));
        LongTerm_->UniqueLabels(std::move(query), std::move(ltsHandler), std::move(ltsSpan));
    }

    void ReadOne(TReadOneQuery query, IResultHandlerPtr<TReadOneResult> handler, TSpanId) const override {
        // TODO: better error status
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "ReadOne is not supported by merge STS-LTS datasource");
    }

    void ReadMany(TReadManyQuery query, IResultHandlerPtr<TReadManyResult> handler, TSpanId span) const override {
        ui32 limit = query.Lookup ? static_cast<ui32>(query.Lookup->Limit) : Max<ui32>();

        using TResult = TReadManyResult;
        using TMerger = TReadManyMerger;

        auto actor = Runtime_.Register(new TRequester<TResult, TMerger>(Runtime_, std::move(handler), limit));

        // TODO: think about better way to pass moveonly query
        TReadManyQuery queryCopy;
        *(TQuery*) (&queryCopy) = *(TQuery*) (&query);
        queryCopy.MaxTimeSeriesFormat = query.MaxTimeSeriesFormat;
        queryCopy.Operations = query.Operations;

        if (query.Lookup) {
            queryCopy.Lookup = std::make_unique<TReadManyQuery::TLookup>(
                    query.Lookup->Selectors,
                    query.Lookup->Limit,
                    query.Lookup->GroupBy);
        }

        if (query.ResolvedKeys) {
            queryCopy.ResolvedKeys = std::make_unique<TReadManyQuery::TResolvedKeys>(
                    query.ResolvedKeys->Strings.Copy(),
                    query.ResolvedKeys->CommonLabels,
                    query.ResolvedKeys->MetricKeys);
        }

        auto stsSpan = TRACING_NEW_SPAN_START(span, "STS-ReadMany");
        auto stsHandler = CreateHandler<TResult>(actor, EStorageType::STS, TSpanId(span));
        ShortTerm_->ReadMany(std::move(query), std::move(stsHandler), std::move(stsSpan));
        auto ltsSpan = TRACING_NEW_SPAN_START(span, "LTS-ReadMany");
        auto ltsHandler = CreateHandler<TResult>(actor, EStorageType::LTS, TSpanId(span));
        LongTerm_->ReadMany(std::move(queryCopy), std::move(ltsHandler), std::move(ltsSpan));
    }

    void WaitUntilInitialized() const override {
        LongTerm_->WaitUntilInitialized();
        ShortTerm_->WaitUntilInitialized();
    }

private:
    TActorRuntime& Runtime_;
    IDataSourcePtr ShortTerm_;
    IDataSourcePtr LongTerm_;

private:
    template <typename T>
    IResultHandlerPtr<T> CreateHandler(TActorId actor, EStorageType type, TSpanId span) const {
        struct THandler: public IResultHandler<T> {
            THandler(TActorSystem& sys, TActorId actor, EStorageType type, TSpanId span)
                    : Sys(sys)
                    , Actor(actor)
                    , Type(type)
                    , Span(std::move(span))
            {
            }

            void OnSuccess(std::unique_ptr<T> response) override {
                Sys.Send(new IEventHandle{
                        Actor, TActorId{}, new TPrivateEvent::TSuccess(std::move(response), Type),
                        0, 0, nullptr, TSpanId(Span)
                });
            }

            void OnError(TString&& project, EDataSourceStatus status, TString&& message) override {
                Sys.Send(new IEventHandle{
                        Actor, TActorId{}, new TPrivateEvent::TError(std::move(project), status, std::move(message), Type),
                        0, 0, nullptr, TSpanId(Span)
                });
            }

            TActorSystem& Sys;
            TActorId Actor;
            EStorageType Type;
            TSpanId Span;
        };

        return MakeIntrusive<THandler>(Runtime_.ActorSystem(), actor, type, std::move(span));
    }
};

} // namespace

IDataSourcePtr MergeShortLongTermsSource(TActorRuntime& runtime, IDataSourcePtr shortTerm, IDataSourcePtr longTerm) {
    return MakeIntrusive<TMergeShortLongTermsSource>(runtime, std::move(shortTerm), std::move(longTerm));
}

} // namespace NSolomon::NDataProxy::NMerger
