#include "merger.h"

#include <solomon/services/dataproxy/lib/timeseries/merger.h>
#include <solomon/services/dataproxy/lib/timeseries/compressed.h>

namespace NSolomon::NDataProxy::NMerger {

TReadManyMerger::TReadManyMerger(size_t metricsLimit)
    : MetricsLimit_(metricsLimit)
{
}

void TReadManyMerger::AddResponse(TReadManyResult&& response, EStorageType type) {
    auto responseStrings = response.Strings.Build();

    for (auto&& metric: response.Metrics) {
        TMetricKey<ui32> metricKey;
        metricKey.Labels.reserve(metric.Meta.Labels.size());
        for (auto [key, value]: metric.Meta.Labels) {
            ui32 k = Strings_.Put(responseStrings[key]);
            ui32 v = Strings_.Put(responseStrings[value]);
            metricKey.Labels.emplace_back(k, v);
        }

        std::sort(metricKey.Labels.begin(), metricKey.Labels.end(), [](auto lhs, auto rhs) {
            return lhs.Key < rhs.Key;
        });

        auto metricName = responseStrings[metric.Meta.Name];
        if (metricName) {
            metricKey.Name = Strings_.Put(metricName);
        }

        TMetricValue* value = nullptr;
        if (Metrics_.size() < MetricsLimit_) {
            auto[it, _] = Metrics_.emplace(std::move(metricKey), TMetricValue());
            value = &(it->second);
        } else if (auto it = Metrics_.find(metricKey); it != Metrics_.end()) {
            value = &(it->second);
        } else {
            continue;
        }

        value->Timeseries[ToUnderlying(type)] = std::move(metric.TimeSeries);
    }

    Errors_.insert(Errors_.end(), response.Errors.begin(), response.Errors.end());
}

void TReadManyMerger::AddError(TDataSourceError error) {
    Errors_.emplace_back(std::move(error));
}

std::unique_ptr<TReadManyResult> TReadManyMerger::Finish() {
    auto res = std::make_unique<TReadManyResult>();
    res->Metrics.reserve(Metrics_.size());
    res->Strings = std::move(Strings_);

    for (auto it = Metrics_.begin(); it != Metrics_.end();) {
        auto node = Metrics_.extract(it++);
        std::unique_ptr<ITimeSeries> tsMerged;
        try {
            tsMerged = Merge(std::move(node.mapped().Timeseries));
        } catch (yexception e) {
            AddError(TDataSourceError{EDataSourceStatus::BACKEND_ERROR, e.what()});
            continue;
        }
        if (!tsMerged) {
            continue;
        }

        auto& m = res->Metrics.emplace_back();
        m.Meta.Type = tsMerged->Type();
        m.Meta.Name = node.key().Name;
        m.Meta.Labels = std::move(node.key().Labels);
        m.TimeSeries = std::move(tsMerged);
    }

    res->Errors = std::move(Errors_);
    return res;
}

std::unique_ptr<ITimeSeries> TReadManyMerger::Merge(
        std::array<std::unique_ptr<ITimeSeries>, ToUnderlying(EStorageType::COUNT)>&& data)
{
    auto& sts = data[ToUnderlying(EStorageType::STS)];
    auto& lts = data[ToUnderlying(EStorageType::LTS)];

    if (!sts || sts->PointCount() == 0) {
        return std::move(lts);
    }

    if (!lts || lts->PointCount() == 0) {
        return std::move(sts);
    }

    auto type = sts->Type();

    TMergingIterator<TMaxMerger> mergeIt;
    mergeIt.AddTimeSeries(*sts);
    mergeIt.AddTimeSeries(*lts);

    return std::make_unique<TCompressedTimeSeries>(type, mergeIt.Columns(), mergeIt);
}


} // namespace NSolomon::NDataProxy::NMerger
