#include <solomon/services/dataproxy/lib/datasource/merge_sts_lts/read_many/merger.h>
#include <solomon/services/dataproxy/lib/timeseries/vector.h>

#include <solomon/libs/cpp/string_pool/string_pool.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/builder.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace NMonitoring;
using namespace NMerger;

class TResultBuilder {
public:
    void AddMetric(EMetricType type) {
        Res_.Metrics.emplace_back();
        Res_.Metrics.back().Meta.Type = type;
    }

    void AddLabel(const char* key, const char* value) {
        Res_.Metrics.back().Meta.Labels.emplace_back(Builder_.Put(key), Builder_.Put(value));
    }

    void AddLabels(const std::vector<std::pair<const char*, const char*>>& labels) {
        for (auto [key, value]: labels) {
            AddLabel(key, value);
        }
    }

    template <class TPoint>
    TVectorTimeSeries<TPoint>& AddTimeSeries() {
        Res_.Metrics.back().TimeSeries = std::make_unique<TVectorTimeSeries<TPoint>>(Res_.Metrics.back().Meta.Type);
        return *(dynamic_cast<TVectorTimeSeries<TPoint>*>(Res_.Metrics.back().TimeSeries.get()));
    }

    TReadManyResult Finish() {
        Res_.Strings = std::move(Builder_);

        return std::move(Res_);
    }

private:
    NStringPool::TStringPoolBuilder Builder_;

    TReadManyResult Res_;
};

TEST(TReadManyMerger, Common) {
    TReadManyResult stsRes;
    TReadManyResult ltsRes;

    {
        TResultBuilder sts;

        sts.AddMetric(EMetricType::IGAUGE);
        sts.AddLabels({{"sensor",  "cpu_usage"},
                       {"service", "fetcher"}});


        auto& ts = sts.AddTimeSeries<NTs::TLongPoint>();
        for (size_t i = 0; i < 10; ++i) {
            auto p = NTs::TLongPoint();
            p.Count = 1u;
            p.Value = i;
            p.Time = TInstant::Seconds(i + 1);

            ts.PushBack(std::move(p));
        }

        stsRes = sts.Finish();
    }

    {
        TResultBuilder lts;
        lts.AddMetric(EMetricType::IGAUGE);
        lts.AddLabels({{"sensor",  "cpu_usage"},
                       {"service", "fetcher"}});

        auto& ts = lts.AddTimeSeries<NTs::TLongPoint>();
        for (size_t i = 5; i < 10; ++i) {
            auto p = NTs::TLongPoint();
            p.Count = 1u;
            p.Value = i + 1;
            p.Time = TInstant::Seconds(i + 1);

            ts.PushBack(p);
        }

        for (size_t i = 10; i < 20; ++i) {
            auto p = NTs::TLongPoint();
            p.Count = 1u;
            p.Value = i;
            p.Time = TInstant::Seconds(i + 1);

            ts.PushBack(p);
        }

        ltsRes = lts.Finish();
    }

    TReadManyMerger merger(1u);
    merger.AddResponse(std::move(stsRes), EStorageType::STS);
    merger.AddResponse(std::move(ltsRes), EStorageType::LTS);

    auto res = merger.Finish();

    ASSERT_EQ(res->Metrics.size(), 1u);

    EXPECT_EQ(res->Metrics.back().Meta.Type, EMetricType::IGAUGE);

    ASSERT_EQ(res->Metrics.back().Meta.Labels.size(), 2u);
    EXPECT_EQ(res->Strings.Size(), 4u);

    auto resStrings = res->Strings.Build();

    EXPECT_EQ(resStrings[res->Metrics.back().Meta.Labels[0].Key], "sensor");
    EXPECT_EQ(resStrings[res->Metrics.back().Meta.Labels[0].Value], "cpu_usage");
    EXPECT_EQ(resStrings[res->Metrics.back().Meta.Labels[1].Key], "service");
    EXPECT_EQ(resStrings[res->Metrics.back().Meta.Labels[1].Value], "fetcher");

    ASSERT_TRUE(res->Metrics.back().TimeSeries);
    auto it = res->Metrics.back().TimeSeries->Iterator();
    NTs::TVariantPoint vp;
    for (size_t i = 0; i < 20u; ++i) {
        ASSERT_TRUE(it->Next(&vp));
        auto& p = vp.Get<NTs::NValue::TLong>();

        EXPECT_EQ(vp.Time, TInstant::Seconds(i + 1));
        EXPECT_EQ(p.Value, (i64) i);
    }


    ASSERT_FALSE(it->Next(&vp));
}
