#include "monitor.h"

#include <solomon/services/dataproxy/lib/datasource/top/top.h>
#include <solomon/libs/cpp/trace/trace.h>

#include <util/string/cast.h>

using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

class TDataSourceMetrics {
public:
    TDataSourceMetrics(NMonitoring::TMetricRegistry& registry, TStringBuf method) {
        using NMonitoring::TLabels;

        Started_ = registry.Rate(TLabels{{"sensor", "datasource.started"}, {"method", method}});
        Ok_ = registry.Rate(TLabels{{"sensor", "datasource.ok"}, {"method", method}});
        Error_ = registry.Rate(TLabels{{"sensor", "datasource.error"}, {"method", method}});
        InFlight_ = registry.LazyIntGauge(
                TLabels{{"sensor", "datasource.inFlight"}, {"method", method}},
                [this]() { return Started_->Get() - Ok_->Get() - Error_->Get(); });

        ElapsedTimeMillis_ = registry.HistogramRate(
                TLabels{{"sensor", "datasource.elapsedTimeMillis"}, {"method", method}},
                NMonitoring::ExponentialHistogram(18, 2, 1));

        for (size_t i = 0; i < std::size(DataSourceStatuses); i++) {
            auto status = ToString(static_cast<EDataSourceStatus>(i));
            Status_[i] = registry.Rate(TLabels{{"sensor", "datasource.status"}, {"method", method}, {"status", status}});
        }
    }

    void Start() {
        Started_->Inc();
    }

    void FinishOk(TDuration elapsed) {
        Ok_->Inc();
        ElapsedTimeMillis_->Record(elapsed.MilliSeconds());
        Status_[static_cast<size_t>(EDataSourceStatus::OK)]->Inc();
    }

    void FinishError(EDataSourceStatus status, TDuration elapsed) {
        Y_VERIFY_DEBUG(static_cast<size_t>(status) < std::size(DataSourceStatuses));
        Error_->Inc();
        ElapsedTimeMillis_->Record(elapsed.MilliSeconds());
        Status_[static_cast<size_t>(status)]->Inc();
    }

private:
    NMonitoring::TRate* Started_;
    NMonitoring::TRate* Ok_;
    NMonitoring::TRate* Error_;
    NMonitoring::TLazyIntGauge* InFlight_;
    NMonitoring::THistogram* ElapsedTimeMillis_;
    std::array<NMonitoring::TRate*, std::size(DataSourceStatuses)> Status_;
};

template <typename T>
class TMonitorHandler: public IResultHandler<T> {
public:
    TMonitorHandler(IResultHandlerPtr<T> handler, TDataSourceMetrics* metrics, TDataSourceMetrics* total)
        : Handler_{std::move(handler)}
        , Metrics_{metrics}
        , Total_{total}
        , StartedAt_{TInstant::Now()}
    {
        Metrics_->Start();
        Total_->Start();
    }

private:
    void OnSuccess(std::unique_ptr<T> result) override {
        auto elapsed = TInstant::Now() - StartedAt_;
        Metrics_->FinishOk(elapsed);
        Total_->FinishOk(elapsed);
        Handler_->OnSuccess(std::move(result));
    }

    void OnError(TString&& project, EDataSourceStatus status, TString&& message) override {
        auto elapsed = TInstant::Now() - StartedAt_;
        Metrics_->FinishError(status, elapsed);
        Total_->FinishError(status, elapsed);
        Handler_->OnError(std::move(project), status, std::move(message));
    }

private:
    IResultHandlerPtr<T> Handler_;
    TDataSourceMetrics* Metrics_;
    TDataSourceMetrics* Total_;
    TInstant StartedAt_;
};

class TMonitorDataSource: public IDataSource {
    enum class EMethod {
        Total = 0,
        Find,
        ResolveOne,
        ResolveMany,
        MetricNames,
        LabelKeys,
        LabelValues,
        UniqueLabels,
        ReadOne,
        ReadMany,
        Top,
    };
public:
    TMonitorDataSource(IDataSourcePtr dataSource, NMonitoring::TMetricRegistry& registry)
        : DataSource_{std::move(dataSource)}
    {
        std::array methods = {
                TStringBuf("total"),
                TStringBuf("Find"),
                TStringBuf("ResolveOne"),
                TStringBuf("ResolveMany"),
                TStringBuf("MetricNames"),
                TStringBuf("LabelKeys"),
                TStringBuf("LabelValues"),
                TStringBuf("UniqueLabels"),
                TStringBuf("ReadOne"),
                TStringBuf("ReadMany"),
                TStringBuf("Top"),
        };

        for (TStringBuf method: methods) {
            Metrics_.push_back(std::make_unique<TDataSourceMetrics>(registry, method));
        }
    }

private:
    bool CanHandle(const TQuery& query) const override {
        return DataSource_->CanHandle(query);
    }

    void Find(TFindQuery query, IResultHandlerPtr<TFindResult> handler, TSpanId traceCtx) const override {
        DataSource_->Find(std::move(query), MakeHandler<TFindResult, EMethod::Find>(std::move(handler)), std::move(traceCtx));
    }

    void ResolveOne(TResolveOneQuery query, IResultHandlerPtr<TResolveOneResult> handler, TSpanId traceCtx) const override {
        DataSource_->ResolveOne(std::move(query), MakeHandler<TResolveOneResult, EMethod::ResolveOne>(std::move(handler)), std::move(traceCtx));
    }

    void ResolveMany(TResolveManyQuery query, IResultHandlerPtr<TResolveManyResult> handler, TSpanId traceCtx) const override {
        DataSource_->ResolveMany(std::move(query), MakeHandler<TResolveManyResult, EMethod::ResolveMany>(std::move(handler)), std::move(traceCtx));
    }

    void MetricNames(TMetricNamesQuery query, IResultHandlerPtr<TMetricNamesResult> handler, TSpanId traceCtx) const override {
        DataSource_->MetricNames(std::move(query), MakeHandler<TMetricNamesResult, EMethod::MetricNames>(std::move(handler)), std::move(traceCtx));
    }

    void LabelKeys(TLabelKeysQuery query, IResultHandlerPtr<TLabelKeysResult> handler, TSpanId traceCtx) const override {
        DataSource_->LabelKeys(std::move(query), MakeHandler<TLabelKeysResult, EMethod::LabelKeys>(std::move(handler)), std::move(traceCtx));
    }

    void LabelValues(TLabelValuesQuery query, IResultHandlerPtr<TLabelValuesResult> handler, TSpanId traceCtx) const override {
        DataSource_->LabelValues(std::move(query), MakeHandler<TLabelValuesResult, EMethod::LabelValues>(std::move(handler)), std::move(traceCtx));
    }

    void UniqueLabels(TUniqueLabelsQuery query, IResultHandlerPtr<TUniqueLabelsResult> handler, TSpanId traceCtx) const override {
        DataSource_->UniqueLabels(std::move(query), MakeHandler<TUniqueLabelsResult, EMethod::UniqueLabels>(std::move(handler)), std::move(traceCtx));
    }

    void ReadOne(TReadOneQuery query, IResultHandlerPtr<TReadOneResult> handler, TSpanId traceCtx) const override {
        DataSource_->ReadOne(std::move(query), MakeHandler<TReadOneResult, EMethod::ReadOne>(std::move(handler)), std::move(traceCtx));
    }

    void ReadMany(TReadManyQuery query, IResultHandlerPtr<TReadManyResult> handler, TSpanId traceCtx) const override {
        if (GetTopOperation(query)) {
            DataSource_->ReadMany(std::move(query), MakeHandler<TReadManyResult, EMethod::Top>(std::move(handler)), std::move(traceCtx));
        } else {
            DataSource_->ReadMany(std::move(query), MakeHandler<TReadManyResult, EMethod::ReadMany>(std::move(handler)), std::move(traceCtx));
        }
    }

    void WaitUntilInitialized() const override {
        DataSource_->WaitUntilInitialized();
    }

    template <typename T, EMethod Method>
    IResultHandlerPtr<T> MakeHandler(IResultHandlerPtr<T> handler) const {
        auto* metrics = Metrics_[static_cast<size_t>(Method)].get();
        auto* total = Metrics_[static_cast<size_t>(EMethod::Total)].get();
        return new TMonitorHandler<T>(std::move(handler), metrics, total);
    }

private:
    IDataSourcePtr DataSource_;
    std::vector<std::unique_ptr<TDataSourceMetrics>> Metrics_;
};

} // namespace

IDataSourcePtr MonitorDataSource(IDataSourcePtr dataSource, NMonitoring::TMetricRegistry& registry) {
    return new TMonitorDataSource{std::move(dataSource), registry};
}

} // namespace NSolomon::NDataProxy
