#pragma once

#include "status_code.h"

#include <solomon/services/dataproxy/api/dataproxy_service.pb.h>

#include <solomon/services/dataproxy/lib/metric/metric.h>
#include <solomon/services/dataproxy/lib/timeseries/aggregates.h>
#include <solomon/services/dataproxy/lib/timeseries/compressed.h>
#include <solomon/services/dataproxy/lib/timeseries/timeseries.h>

#include <solomon/libs/cpp/selectors/selectors.h>
#include <solomon/libs/cpp/string_pool/string_pool.h>
#include <solomon/libs/cpp/yasm/constants/project.h>
#include <solomon/protos/common/request_producer.pb.h>
#include <solomon/protos/math/operation.pb.h>

#include <util/datetime/base.h>

#include <grpc++/support/status_code_enum.h>

namespace NSolomon::NDataProxy {

// -- common ------------------------------------------------------------------

struct TTimeRange {
    TInstant From; // inclusive
    TInstant To;   // exclusive
};

struct TDataSourceError {
    TDataSourceError(grpc::StatusCode status, TString message) noexcept
        : Message(std::move(message))
    {
        switch (status) {
            case grpc::StatusCode::INTERNAL: {
                Status = EDataSourceStatus::BACKEND_ERROR;
                break;
            }
            case grpc::StatusCode::NOT_FOUND: {
                Status = EDataSourceStatus::NOT_FOUND;
                break;
            }
            case grpc::StatusCode::OK: {
                Status = EDataSourceStatus::OK;
                break;
            }
            case grpc::StatusCode::DEADLINE_EXCEEDED: {
                Status = EDataSourceStatus::BACKEND_TIMEOUT;
                break;
            }
            case grpc::StatusCode::INVALID_ARGUMENT: {
                Status = EDataSourceStatus::BAD_REQUEST;
                break;
            }
            case grpc::StatusCode::UNAVAILABLE: {
                Status = EDataSourceStatus::BACKEND_UNAVAILABLE;
                break;
            }
            default: {
                Status = EDataSourceStatus::UNKNOWN;
            }
        }
    }

    TDataSourceError(EDataSourceStatus status, TString message) noexcept
        : Status(status)
        , Message(std::move(message))
    {
    }

    EDataSourceStatus Status;
    TString Message;
};

using TDataSourceErrors = TVector<TDataSourceError>;

struct TQuery {
    TString Project;
    TTimeRange Time;
    TInstant Deadline;
    std::optional<TInstant> SoftDeadline;
    TString ForceReplicaRead;
    yandex::solomon::common::RequestProducer Producer{yandex::solomon::common::RequestProducer::REQUEST_PRODUCER_UNSPECIFIED};
    yandex::monitoring::dataproxy::ShortTermStorage ShortTermStorage{yandex::monitoring::dataproxy::ShortTermStorage::UNSPECIFIED};
    TStringBuf ClientId;

    bool IsYasm() const {
        return NYasm::IsYasmProject(Project);
    }

    bool IsYasmSts() const {
        return NYasm::IsYasmStsProject(Project);
    }
};

// -- find --------------------------------------------------------------------

struct TFindQuery: public TQuery {
    TSelectors Selectors;
    ui32 Limit{0};
    bool FillMetricName{false};
};

struct TFindResult {
    TFindResult Clone() const {
        return {Strings.Copy(), Metrics, Errors, TotalCount, Truncated, Dcs};
    }

    NStringPool::TStringPoolBuilder Strings;
    TVector<TMetric<ui32>> Metrics;
    TDataSourceErrors Errors;
    ui32 TotalCount{0};
    bool Truncated{false};
    TReplicaMap<EDc> Dcs;
};

// -- resolve one -------------------------------------------------------------

struct TResolveOneQuery: public TQuery {
    TMetricKey<TString> Key;
};

struct TResolveOneResult {
    TMetric<TString> Metric;
    TDataSourceErrors Errors;
    TReplicaMap<EDc> Dcs;
};

// -- resolve many -------------------------------------------------------------

struct TResolveManyQuery: public TQuery {
    NStringPool::TStringPool Strings;
    TLabels<ui32> CommonLabels;
    TVector<TMetricKey<ui32>> MetricKeys;
};

struct TResolveManyResult {
    NStringPool::TStringPoolBuilder Strings;
    TVector<TMetric<ui32>> Metrics;
    TDataSourceErrors Errors;
    TReplicaMap<EDc> Dcs;
};

// -- metric names ------------------------------------------------------------

struct TMetricNamesQuery: public TQuery {
    TSelectors Selectors;
    TString TextFilter;
    ui32 Limit{0};
};

struct TMetricNamesResult {
    NStringPool::TStringPoolBuilder Strings;
    TVector<ui32> Names;
    TDataSourceErrors Errors;
    ui32 TotalCount{0};
    bool Truncated{false};
};

// -- label keys --------------------------------------------------------------

struct TLabelKeysQuery: public TQuery {
    TSelectors Selectors;
};

struct TLabelKeysResult {
    NStringPool::TStringPoolBuilder Strings;
    TVector<ui32> Keys;
    TDataSourceErrors Errors;
};

// -- label values ------------------------------------------------------------

struct TLabelValuesQuery: public TQuery {
    TSelectors Selectors;
    TVector<TString> Keys;
    TString TextFilter;
    ui32 Limit{0};
};

struct TLabelValuesResult {
    struct TLabelState {
        TVector<ui32> Values;
        TDataSourceErrors Errors;
        ui32 Key{0};
        ui32 MetricCount{0};
        bool Truncated{false};
    };

    TLabelValuesResult Clone() const {
        return {Project, Strings.Copy(), Labels, Errors, MetricCount};
    }

    TString Project;
    NStringPool::TStringPoolBuilder Strings;
    TVector<TLabelState> Labels;
    TDataSourceErrors Errors;
    ui32 MetricCount{0};
};

// -- unique labels -----------------------------------------------------------

struct TUniqueLabelsQuery: public TQuery {
    TSelectors Selectors;
    TVector<TString> Keys;
};

struct TUniqueLabelsResult {
    NStringPool::TStringPoolBuilder Strings;
    TVector<TLabels<ui32>> Labels;
    TDataSourceErrors Errors;
};

// -- read one ----------------------------------------------------------------

struct TReadOneQuery: public TQuery {
    TString Name;
    TLabels<TString> Labels;
    ui32 MaxTimeseriesFormat{0};
    // TODO: replace protobuf with c++ structure
    TVector<yandex::solomon::math::Operation> Operations;
};

struct TReadOneResult {
    TMetric<TString> Metric;
    std::unique_ptr<ITimeSeries> TimeSeries;
    TDataSourceErrors Errors;
};

// -- read many ---------------------------------------------------------------

struct TReadManyQuery: public TQuery {
    struct TResolvedKeys {
        TResolvedKeys() = default;
        TResolvedKeys(
                NStringPool::TStringPool strings,
                TLabels<ui32> commonLabels,
                TVector<TMetricKey<ui32>> keys)
           : Strings(std::move(strings))
           , CommonLabels(std::move(commonLabels))
           , MetricKeys(std::move(keys))
        {
        }

        NStringPool::TStringPool Strings;
        TLabels<ui32> CommonLabels;
        TVector<TMetricKey<ui32>> MetricKeys;
    };

    struct TLookup {
        TLookup() = default;
        TLookup(TSelectors selectors, ui32 limit, TVector<TString> groupBy = {})
            : Selectors(std::move(selectors))
            , Limit(limit)
            , GroupBy(std::move(groupBy))
        {
        }

        TSelectors Selectors;
        ui32 Limit{0};
        TVector<TString> GroupBy;
    };

    TReadManyQuery Clone() {
        TReadManyQuery queryCopy;
        *(TQuery*)(&queryCopy) = *(TQuery*)(this);
        queryCopy.MaxTimeSeriesFormat = MaxTimeSeriesFormat;
        queryCopy.Operations = Operations;

        if (Lookup) {
            queryCopy.Lookup = std::make_unique<TReadManyQuery::TLookup>(Lookup->Selectors, Lookup->Limit, Lookup->GroupBy);
        }

        if (ResolvedKeys) {
            queryCopy.ResolvedKeys = std::make_unique<TReadManyQuery::TResolvedKeys>(
                    ResolvedKeys->Strings.Copy(),
                    ResolvedKeys->CommonLabels,
                    ResolvedKeys->MetricKeys);
        }
        return queryCopy;
    }

    std::unique_ptr<TResolvedKeys> ResolvedKeys;
    std::unique_ptr<TLookup> Lookup;
    ui32 MaxTimeSeriesFormat{0};
    // TODO: replace protobuf with c++ structure
    TVector<yandex::solomon::math::Operation> Operations;
};

struct TReadManyResult {
    struct TMetricData {
        TMetric<ui32> Meta;
        std::unique_ptr<ITimeSeries> TimeSeries;
        std::unique_ptr<TAggregateVariant> Aggregate;
    };

    TReadManyResult Clone() const {
        TReadManyResult copy;
        copy.Errors = Errors;
        copy.Strings = Strings.Copy();

        copy.Metrics.reserve(Metrics.size());
        for (auto&& metric: Metrics) {
            auto& metricCopy = copy.Metrics.emplace_back();
            metricCopy.Meta = metric.Meta;

            if (metric.TimeSeries) {
                metricCopy.TimeSeries = std::make_unique<TCompressedTimeSeries>(*metric.TimeSeries);
            }
            if (metric.Aggregate) {
                metricCopy.Aggregate = std::make_unique<TAggregateVariant>(*metric.Aggregate);
            }
        }
        return copy;
    }

    NStringPool::TStringPoolBuilder Strings;
    TVector<TMetricData> Metrics;
    TDataSourceErrors Errors;
};

} // namespace NSolomon::NDataProxy
