#pragma once

#include "result_handler.h"
#include "query.h"

#include <solomon/services/dataproxy/lib/event_slots.h>

#include <solomon/libs/cpp/actors/events/events.h>

#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/core/actorsystem.h>

namespace NSolomon::NDataProxy {

class TDataSourceEvents: private TEventSlot<EEventSpace::DataProxy, ES_DATA_SOURCE> {
    enum {
        FindQuery = SpaceBegin,
        ResolveOneQuery,
        ResolveManyQuery,
        MetricNamesQuery,
        LabelKeysQuery,
        LabelValuesQuery,
        UniqueLabelsQuery,
        ReadOneQuery,
        ReadManyQuery,
        FindResult,
        ResolveOneResult,
        ResolveManyResult,
        MetricNamesResult,
        LabelKeysResult,
        LabelValuesResult,
        UniqueLabelsResult,
        ReadOneResult,
        ReadManyResult,
        Error,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

    template <typename T>
    static constexpr auto EventType() noexcept {
        if constexpr (std::is_same_v<T, TFindQuery>) return FindQuery;
        else if constexpr (std::is_same_v<T, TResolveOneQuery>) return ResolveOneQuery;
        else if constexpr (std::is_same_v<T, TResolveManyQuery>) return ResolveManyQuery;
        else if constexpr (std::is_same_v<T, TMetricNamesQuery>) return MetricNamesQuery;
        else if constexpr (std::is_same_v<T, TLabelKeysQuery>) return LabelKeysQuery;
        else if constexpr (std::is_same_v<T, TLabelValuesQuery>) return LabelValuesQuery;
        else if constexpr (std::is_same_v<T, TUniqueLabelsQuery>) return UniqueLabelsQuery;
        else if constexpr (std::is_same_v<T, TReadOneQuery>) return ReadOneQuery;
        else if constexpr (std::is_same_v<T, TReadManyQuery>) return ReadManyQuery;
        else if constexpr (std::is_same_v<T, TFindResult>) return FindResult;
        else if constexpr (std::is_same_v<T, TResolveOneResult>) return ResolveOneResult;
        else if constexpr (std::is_same_v<T, TResolveManyResult>) return ResolveManyResult;
        else if constexpr (std::is_same_v<T, TMetricNamesResult>) return MetricNamesResult;
        else if constexpr (std::is_same_v<T, TLabelKeysResult>) return LabelKeysResult;
        else if constexpr (std::is_same_v<T, TLabelValuesResult>) return LabelValuesResult;
        else if constexpr (std::is_same_v<T, TUniqueLabelsResult>) return UniqueLabelsResult;
        else if constexpr (std::is_same_v<T, TReadOneResult>) return ReadOneResult;
        else if constexpr (std::is_same_v<T, TReadManyResult>) return ReadManyResult;
        else static_assert(TDependentFalse<T>, "unsupported type");
    }

public:
    template <typename T>
    struct TQuery: public NActors::TEventLocal<TQuery<T>, EventType<T>()> {
        T Query;

        TQuery(T query) noexcept
            : Query(std::move(query))
        {
        }
    };

    template <typename T>
    struct TSuccess: public NActors::TEventLocal<TSuccess<T>, EventType<T>()> {
        std::unique_ptr<T> Result;

        TSuccess(std::unique_ptr<T> result) noexcept
            : Result(std::move(result))
        {
        }
    };

    struct TError: public NActors::TEventLocal<TError, Error> {
        TString Project;
        EDataSourceStatus Status;
        TString Message;

        TError(TString project, EDataSourceStatus status, TString message) noexcept
            : Project(std::move(project))
            , Status(status)
            , Message(std::move(message))
        {
        }
    };
};

template <typename T>
class TReplyToHandler final: public IResultHandler<T> {
public:
    TReplyToHandler(NActors::TActorSystem* actorSystem, NActors::TActorId replyTo)
        : ActorSystem_(actorSystem)
        , ReplyTo_(replyTo)
    {
    }

private:
    void OnSuccess(std::unique_ptr<T> result) override {
        ActorSystem_->Send(ReplyTo_, new TDataSourceEvents::TSuccess<T>(std::move(result)));
    }

    void OnError(TString&& project, EDataSourceStatus status, TString&& message) override {
        ActorSystem_->Send(ReplyTo_, new TDataSourceEvents::TError(std::move(project), status, std::move(message)));
    }

private:
    NActors::TActorSystem* ActorSystem_;
    NActors::TActorId ReplyTo_;
};

template <typename T>
IResultHandlerPtr<T> MakeReplyToHandler(NActors::TActorId replyTo) {
    Y_VERIFY(NActors::TlsActivationContext, "trying to create ReplyToHandler outside of actor context");
    auto* actorSystem = NActors::TActorContext::ActorSystem();
    return new TReplyToHandler<T>(actorSystem, replyTo);
}

} // namespace NSolomon::NDataProxy
