#pragma once

#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

#include <util/string/builder.h>

namespace NSolomon::NDataProxy {

using TMetricKeys = TVector<const TMetricKey<ui32>*>;
const TDuration DEFAULT_TIMEOUT = TDuration::Seconds(3);

struct TMetricKeyPtrHash {
    size_t operator()(const TMetricKey<ui32>* mk) const {
        return mk->Hash();
    }
};

struct TMetricKeyPtrEq {
    bool operator()(const TMetricKey<ui32>* lhs, const TMetricKey<ui32>* rhs) const {
        return *lhs == *rhs;
    }
};

class TRequestsManager {
    struct TReqCtx {
        TVector<const TMetricKey<ui32>*> MetricKeys;
        EReplica Replica{EReplica::Unknown};
    };

public:
    TRequestsManager(NSolomon::ELogComponent logComponent, const NActors::TActorIdentity& actorId)
        : LogComponent_(logComponent)
        , ActorId_(actorId)
    {
    }

    ui64 NewRequest(EReplica replica, const TMetricKeys& metricKeys) {
        ui64 id = ++NextReqId_;

        for (const auto& key: metricKeys) {
            ++InflightPerMetric_[key];
        }

        IdToCtx_[id] = {metricKeys, replica};

        return id;
    }

    EReplica GetReplica(ui64 reqId) const {
        return IdToCtx_.at(reqId).Replica;
    }

    template <class F>
    void ReportResponse(ui64 reqId, std::optional<TInstant> deadline, F&& schedule) {
        auto it = IdToCtx_.find(reqId);
        if (it == IdToCtx_.end()) {
            LOG_INFO_S(NActors::TActorContext::AsActorContext(), LogComponent_, "Reporting response to UNKNOWN reqId: " << reqId << ". Actor id: " << ActorId_);
            return;
        }
        auto ctx = it->second;

        if (!deadline.has_value()) {
            deadline = NActors::TActivationContext::Now() + DEFAULT_TIMEOUT;
        }

        for (const auto& key: ctx.MetricKeys) {
            auto& inflight = InflightPerMetric_.at(key);
            if (inflight > 0) {
                if (--inflight == 0) {
                    ++MetricsCompleted_;
                    LOG_INFO_S(NActors::TActorContext::AsActorContext(), LogComponent_, "Reporting response to reqId: " << reqId << ", " << MetricsCompleted_ << "/" << InflightPerMetric_.size() << " metrics is completed. Actor id: " << ActorId_);
                } else {
                    schedule(deadline.value(), key);
                }
            }
        }

        IdToCtx_.erase(it);
    }

    template <class F>
    void ReportResponse(ui64 reqId, const TMetricKey<ui32>* metricKey, std::optional<TInstant> deadline, F&& schedule) {
        if (!deadline.has_value()) {
            deadline = NActors::TActivationContext::Now() + DEFAULT_TIMEOUT;
        }

        auto& inflight = InflightPerMetric_.at(metricKey);
        if (inflight > 0) {
            if (--inflight == 0) {
                ++MetricsCompleted_;
                LOG_INFO_S(NActors::TActorContext::AsActorContext(), LogComponent_, "Reporting response: " << reqId << ", " << MetricsCompleted_ << "/" << InflightPerMetric_.size() << " metrics is completed. Actor id: " << ActorId_);
            } else {
                schedule(deadline.value(), metricKey);
            }
        }
    }

    void ReportError(ui64 reqId, const TMetricKey<ui32>* metricKey) {
        auto& inflight = InflightPerMetric_[metricKey];
        if (inflight > 0 && --inflight == 0) {
            ++MetricsCompleted_;
            LOG_INFO_S(NActors::TActorContext::AsActorContext(), LogComponent_, "Reporting error to reqId: " << reqId << ", " << MetricsCompleted_ << "/" << InflightPerMetric_.size() << " metrics is completed. Actor id: " << ActorId_);
        }
    }

    void ReportError(ui64 reqId) {
        auto it = IdToCtx_.find(reqId);
        if (it == IdToCtx_.end()) {
            LOG_INFO_S(NActors::TActorContext::AsActorContext(), LogComponent_, "Reporting error to UNKNOWN reqId: " << reqId << ". Actor id: " << ActorId_);
            return;
        }
        auto ctx = it->second;

        for (const auto& key: ctx.MetricKeys) {
            auto& inflight = InflightPerMetric_.at(key);
            if (inflight > 0 && --inflight == 0) {
                ++MetricsCompleted_;
                LOG_INFO_S(NActors::TActorContext::AsActorContext(), LogComponent_, "Reporting error to reqId: " << reqId << ", " << MetricsCompleted_ << "/" << InflightPerMetric_.size() << " metrics is completed. Actor id: " << ActorId_);
            }
        }

        IdToCtx_.erase(it);
    }

    void ReportTimeout(const TMetricKey<ui32>* metricKey) {
        auto it = InflightPerMetric_.find(metricKey);
        if (it == InflightPerMetric_.end()) {
            LOG_INFO_S(NActors::TActorContext::AsActorContext(), LogComponent_, "Reporting timeout to UNKNOWN metric key. Actor id: " << ActorId_);
            return;
        }
        auto& inflight = it->second;
        if (inflight > 0 && --inflight == 0) {
            ++MetricsCompleted_;
            LOG_INFO_S(NActors::TActorContext::AsActorContext(), LogComponent_, "Reporting timeout, " << MetricsCompleted_ << "/" << InflightPerMetric_.size() << " metrics is completed. Actor id: " << ActorId_);
        }
    }

    bool IsCompleted() const {
        return MetricsCompleted_ == InflightPerMetric_.size();
    }

    bool HasInflightRequests() const {
        return !IdToCtx_.empty();
    }

private:
    ui64 NextReqId_{0};
    ui64 MetricsCompleted_{0};
    absl::flat_hash_map<const TMetricKey<ui32>*, ui8, TMetricKeyPtrHash, TMetricKeyPtrEq> InflightPerMetric_;
    absl::flat_hash_map<ui64, TReqCtx> IdToCtx_;
    NSolomon::ELogComponent LogComponent_;
    NActors::TActorIdentity ActorId_;
};

}
