#include "merger.h"

#include <solomon/services/memstore/api/memstore_service.pb.h>

#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>

namespace NSolomon::NDataProxy {

using yandex::monitoring::memstore::FindResponse;

void TFindMerger::AddResponse(TClusterId clusterId, const TShardSubKey& shardKey, const FindResponse& resp) {
    if (resp.metrics_size() == 0) {
        return;
    }

    if (Metrics_.empty()) {
        Metrics_.reserve(Min(static_cast<size_t>(resp.metrics_size()), Limit_));
    }

    // TODO: make it once
    ui32 projectKey = Strings_.Put(NLabels::LABEL_PROJECT);
    ui32 projectValue = Strings_.Put(Project_);

    ui32 clusterKey = Strings_.Put(NLabels::LABEL_CLUSTER);
    ui32 clusterValue = Strings_.Put(shardKey.Cluster);

    ui32 serviceKey = Strings_.Put(NLabels::LABEL_SERVICE);
    ui32 serviceValue = Strings_.Put(shardKey.Service);

    NStringPool::TStringPool respPool{resp.string_pool()};

    for (const auto& metric: resp.metrics()) {
        if (Metrics_.size() >= Limit_) {
            // do not add more metrics than defined limit
            break;
        }

        TMetricKey<ui32> metricKey;
        metricKey.Labels.emplace_back(projectKey, projectValue);
        metricKey.Labels.emplace_back(clusterKey, clusterValue);
        metricKey.Labels.emplace_back(serviceKey, serviceValue);

        for (int i = 0, size = metric.labels_idx_size(); i < size; ) {
            // TODO: labels must be sorted by the key string
            ui32 keyId = Strings_.Put(respPool.At(metric.labels_idx(i++)));
            ui32 valueId = Strings_.Put(respPool.At(metric.labels_idx(i++)));
            metricKey.Labels.emplace_back(keyId, valueId);
        }

        auto metricType = NSolomon::FromProto(metric.type());
        Metrics_.emplace(std::move(metricKey), metricType);
    }

    Truncated_[clusterId.Replica()] |= resp.truncated();
    TotalCount_[clusterId.Replica()] += resp.total_count();
}

void TFindMerger::AddError(TClusterId clusterId, TShardId shardId, grpc::StatusCode statusCode, TString message) {
    Y_UNUSED(clusterId, shardId, statusCode, message);
    // TODO: aggregate per shard errors
    Errors_.emplace_back(statusCode, std::move(message));
}

std::unique_ptr<TFindResult> TFindMerger::Finish() {
    auto result = std::make_unique<TFindResult>();
    result->Truncated = Truncated_.MaxValue();
    result->TotalCount = TotalCount_.MaxValue();
    result->Strings = std::move(Strings_);

    result->Metrics.reserve(Min<size_t>(Limit_, Metrics_.size()));
    for (auto& [key, type]: Metrics_) {
        if (result->Metrics.size() >= Limit_) {
            break;
        }

        result->Metrics.emplace_back();
        auto& metric = result->Metrics.back();

        metric.Name = key.Name;
        metric.Labels = key.Labels;
        metric.Type = type;
    }

    result->Errors = std::move(Errors_);

    return result;
}

} // namespace NSolomon::NDataProxy
