#include "requester.h"
#include "merger.h"

#include <solomon/services/dataproxy/lib/datasource/sts/base/requester.h>

#include <solomon/libs/cpp/labels/known_keys.h>

using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

using yandex::monitoring::memstore::FindRequest;
using yandex::monitoring::memstore::FindResponse;

using TFindReq = TMemStoreShardEvents::TFindReq;
using TFindResp = TMemStoreShardEvents::TFindResp;

class TMemStoreFindRequester: public TMemStoreRequester<TMemStoreFindRequester, TFindReq, TFindResp> {
    using TBase = TMemStoreRequester<TMemStoreFindRequester, TFindReq, TFindResp>;
public:
    TMemStoreFindRequester(
            TRequesterContextPtr ctx,
            TFindQuery query,
            IResultHandlerPtr<TFindResult> handler,
            TSpanId traceCtx)
        : TBase{std::move(ctx), TShardSelector::FromSelectors(query.Project, query.Selectors), query.Deadline, std::move(traceCtx)}
        , Project_{query.Project}
        , Merger_{query.Project, static_cast<size_t>(query.Limit)}
        , Request_{ToMemStoreRequest(std::move(query))}
        , Handler_{std::move(handler)}
    {
    }

    void OnResponse(TClusterId clusterId, const TShardSubKey& shardKey, const FindResponse& resp) {
        Merger_.AddResponse(clusterId, shardKey, resp);
    }

    void OnError(TClusterId clusterId, TShardId shardId, grpc::StatusCode statusCode, TString message) {
        Merger_.AddError(clusterId, shardId, statusCode, std::move(message));
    }

    void OnFinish(EDataSourceStatus status, TString&& message) {
        if (status == EDataSourceStatus::OK) {
            Handler_->OnSuccess(Merger_.Finish());
        } else {
            // TODO: merge errors
            Handler_->OnError(std::move(Project_), status, std::move(message));
        }
    }

    const FindRequest& RequestTemplate() const {
        return Request_;
    }

private:
    TString Project_;
    TFindMerger Merger_;
    const FindRequest Request_;
    IResultHandlerPtr<TFindResult> Handler_;
};

} // namespace

FindRequest ToMemStoreRequest(TFindQuery query) {
    // clear PCS from selectors, because MemStore handle only per shard requests
    query.Selectors.Remove(NLabels::LABEL_PROJECT);
    query.Selectors.Remove(NLabels::LABEL_CLUSTER);
    query.Selectors.Remove(NLabels::LABEL_SERVICE);

    // prepare request template for all shards (field `num_id` intentionally  not filled)
    FindRequest req;
    req.set_selectors(ToString(query.Selectors));
    req.set_from_millis(query.Time.From.MilliSeconds());
    req.set_to_millis(query.Time.To.MilliSeconds());
    req.set_limit(query.Limit);
    return req;
}

std::unique_ptr<NActors::IActor> MemStoreFindRequester(
        TRequesterContextPtr ctx,
        TFindQuery query,
        IResultHandlerPtr<TFindResult> handler,
        TSpanId traceCtx)
{
    return std::make_unique<TMemStoreFindRequester>(std::move(ctx), std::move(query), std::move(handler), std::move(traceCtx));
}

} // namespace NSolomon::NDataProxy
