#include <solomon/services/dataproxy/lib/datasource/sts/find/merger.h>
#include <solomon/services/memstore/api/memstore_service.pb.h>

#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/libs/cpp/string_pool/string_pool.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace yandex::monitoring::memstore;

using NMonitoring::EMetricType;

const TShardSubKey ShardKey{"prod", "fetcher"};

constexpr TClusterId SasR0{EDc::Sas, EReplica::R0};
constexpr TClusterId SasR1{EDc::Sas, EReplica::R1};

FindResponse CreateResponse(const std::vector<std::pair<TLabels<TString>, EMetricType>>& metrics) {
    FindResponse resp;
    NStringPool::TStringPoolBuilder strings;

    for (const auto& [labels, type]: metrics) {
        auto* m = resp.add_metrics();
        m->set_type(ToProto(type));

        for (const auto& label: labels) {
            m->add_labels_idx(strings.Put(label.Key));
            m->add_labels_idx(strings.Put(label.Value));
        }
    }

    auto pool = strings.Build(yandex::solomon::common::StringPool_Compression_LZ4);
    pool.Swap(resp.mutable_string_pool());
    return resp;
}

// TODO: move to a better place
TLabels<TStringBuf> Resolve(const NStringPool::TStringPool& strings, const TLabels<ui32>& labels) {
    TLabels<TStringBuf> resolved;
    resolved.reserve(labels.size());

    for (const auto& l: labels) {
        resolved.push_back({strings[l.Key], strings[l.Value]});
    }
    return resolved;
}

// TODO: move to a better place
TLabels<TStringBuf> Labels(const std::vector<const char*>& strs) {
    Y_ENSURE(strs.size() % 2 == 0);

    TLabels<TStringBuf> labels;
    labels.reserve(strs.size() / 2);

    for (size_t i = 0; i < strs.size(); ) {
        auto* key = strs[i++];
        auto* value = strs[i++];
        labels.emplace_back(key, value);
    }
    return labels;
}

TEST(TFindMergerTest, EmptyResponse) {
    TFindMerger merger{"solomon", 10};
    merger.AddResponse(SasR0, ShardKey, FindResponse{});

    auto result = merger.Finish();
    ASSERT_TRUE(result);
    EXPECT_EQ(result->Strings.Size(), 0u);
    EXPECT_EQ(result->Metrics.size(), 0u);
    EXPECT_EQ(result->TotalCount, 0u);
    EXPECT_TRUE(!result->Truncated);
}

TEST(TFindMergerTest, SingleResponse) {
    auto resp = CreateResponse({
        {{{"sensor", "cpu_usage"}, {"host", "fetcher-sas-00"}}, EMetricType::GAUGE},
        {{{"sensor", "mem_usage"}, {"host", "fetcher-sas-00"}}, EMetricType::RATE},
    });
    resp.set_total_count(100);
    resp.set_truncated(false);

    TFindMerger merger{"solomon", 10};
    merger.AddResponse(SasR0, ShardKey, resp);

    auto result = merger.Finish();
    ASSERT_TRUE(result);
    EXPECT_EQ(result->Strings.Size(), 11u);
    EXPECT_EQ(result->Metrics.size(), 2u);
    EXPECT_EQ(result->TotalCount, 100u);
    EXPECT_FALSE(result->Truncated);

    auto resultStrings = result->Strings.Build();

    for (const auto& metric: result->Metrics) {
        if (metric.Type == EMetricType::GAUGE) {
            // memstore does not support names right now
            EXPECT_EQ(resultStrings[metric.Name], "");
            EXPECT_EQ(
                    Resolve(resultStrings, metric.Labels),
                    Labels({
                        "project", "solomon",
                        "cluster", "prod",
                        "service", "fetcher",
                        "sensor", "cpu_usage",
                        "host", "fetcher-sas-00",
                    }));
            // memstore does not have stockpile ids
            EXPECT_EQ(metric.StockpileIds[EReplica::R0], TStockpileId{});
            EXPECT_EQ(metric.StockpileIds[EReplica::R1], TStockpileId{});
        } else if (metric.Type == EMetricType::RATE) {
            // memstore does not support names right now
            EXPECT_EQ(resultStrings[metric.Name], "");
            EXPECT_EQ(
                    Resolve(resultStrings, metric.Labels),
                    Labels({
                        "project", "solomon",
                        "cluster", "prod",
                        "service", "fetcher",
                        "sensor", "mem_usage",
                        "host", "fetcher-sas-00",
                    }));
            // memstore does not have stockpile ids
            EXPECT_EQ(metric.StockpileIds[EReplica::R0], TStockpileId{});
            EXPECT_EQ(metric.StockpileIds[EReplica::R1], TStockpileId{});
        } else {
            FAIL() << "unexpected metric type";
        }
    }
}

TEST(TFindMergerTest, MultipleResponses_NoIntersecions) {
    auto resp1 = CreateResponse({
        {{{"sensor", "cpu_usage"}, {"host", "fetcher-sas-00"}}, EMetricType::GAUGE},
    });
    resp1.set_total_count(77);
    resp1.set_truncated(false);

    auto resp2 = CreateResponse({
        {{{"sensor", "mem_usage"}, {"host", "fetcher-sas-01"}}, EMetricType::RATE},
    });
    resp2.set_total_count(100);
    resp2.set_truncated(true);

    TFindMerger merger{"solomon", 10};
    merger.AddResponse(SasR0, ShardKey, resp1);
    merger.AddResponse(SasR1, ShardKey, resp2);

    auto result = merger.Finish();
    ASSERT_TRUE(result);
    EXPECT_EQ(result->Strings.Size(), 12u);
    EXPECT_EQ(result->Metrics.size(), 2u);
    EXPECT_EQ(result->TotalCount, 100u);
    EXPECT_TRUE(result->Truncated);

    auto resultStrings = result->Strings.Build();

    for (const auto& metric: result->Metrics) {
        if (metric.Type == EMetricType::GAUGE) {
            // memstore does not support names right now
            EXPECT_EQ(resultStrings[metric.Name], "");
            EXPECT_EQ(
                    Resolve(resultStrings, metric.Labels),
                    Labels({
                        "project", "solomon",
                        "cluster", "prod",
                        "service", "fetcher",
                        "sensor", "cpu_usage",
                        "host", "fetcher-sas-00",
                    }));
            // memstore does not have stockpile ids
            EXPECT_EQ(metric.StockpileIds[EReplica::R0], TStockpileId{});
            EXPECT_EQ(metric.StockpileIds[EReplica::R1], TStockpileId{});
        } else if (metric.Type == EMetricType::RATE) {
            // memstore does not support names right now
            EXPECT_EQ(resultStrings[metric.Name], "");
            EXPECT_EQ(
                    Resolve(resultStrings, metric.Labels),
                    Labels({
                        "project", "solomon",
                        "cluster", "prod",
                        "service", "fetcher",
                        "sensor", "mem_usage",
                        "host", "fetcher-sas-01",
                    }));
            // memstore does not have stockpile ids
            EXPECT_EQ(metric.StockpileIds[EReplica::R0], TStockpileId{});
            EXPECT_EQ(metric.StockpileIds[EReplica::R1], TStockpileId{});
        } else {
            FAIL() << "unexpected metric type";
        }
    }
}

TEST(TFindMergerTest, MultipleResponses_WithIntersecions) {
    auto resp1 = CreateResponse({
        {{{"sensor", "cpu_usage"}, {"host", "fetcher-sas-00"}}, EMetricType::GAUGE},
        {{{"sensor", "mem_usage"}, {"host", "fetcher-sas-01"}}, EMetricType::RATE},
    });
    resp1.set_total_count(77);
    resp1.set_truncated(false);

    auto resp2 = CreateResponse({
        {{{"sensor", "cpu_usage"}, {"host", "fetcher-sas-00"}}, EMetricType::GAUGE},
        {{{"sensor", "mem_usage"}, {"host", "fetcher-sas-01"}}, EMetricType::RATE},
    });
    resp2.set_total_count(100);
    resp2.set_truncated(true);

    TFindMerger merger{"solomon", 10};
    merger.AddResponse(SasR0, ShardKey, resp1);
    merger.AddResponse(SasR1, ShardKey, resp2);

    auto result = merger.Finish();
    ASSERT_TRUE(result);
    EXPECT_EQ(result->Strings.Size(), 12u);
    EXPECT_EQ(result->Metrics.size(), 2u);
    EXPECT_EQ(result->TotalCount, 100u);
    EXPECT_TRUE(result->Truncated);

    auto resultStrings = result->Strings.Build();

    for (const auto& metric: result->Metrics) {
        if (metric.Type == EMetricType::GAUGE) {
            // memstore does not support names right now
            EXPECT_EQ(resultStrings[metric.Name], "");
            EXPECT_EQ(
                    Resolve(resultStrings, metric.Labels),
                    Labels({
                        "project", "solomon",
                        "cluster", "prod",
                        "service", "fetcher",
                        "sensor", "cpu_usage",
                        "host", "fetcher-sas-00",
                    }));
            // memstore does not have stockpile ids
            EXPECT_EQ(metric.StockpileIds[EReplica::R0], TStockpileId{});
            EXPECT_EQ(metric.StockpileIds[EReplica::R1], TStockpileId{});
        } else if (metric.Type == EMetricType::RATE) {
            // memstore does not support names right now
            EXPECT_EQ(resultStrings[metric.Name], "");
            EXPECT_EQ(
                    Resolve(resultStrings, metric.Labels),
                    Labels({
                        "project", "solomon",
                        "cluster", "prod",
                        "service", "fetcher",
                        "sensor", "mem_usage",
                        "host", "fetcher-sas-01",
                    }));
            // memstore does not have stockpile ids
            EXPECT_EQ(metric.StockpileIds[EReplica::R0], TStockpileId{});
            EXPECT_EQ(metric.StockpileIds[EReplica::R1], TStockpileId{});
        } else {
            FAIL() << "unexpected metric type";
        }
    }
}

TEST(TFindMergerTest, MultipleResponses_Limit) {
    auto resp1 = CreateResponse({
        {{{"sensor", "cpu_usage"}, {"host", "fetcher-sas-00"}}, EMetricType::GAUGE},
        {{{"sensor", "mem_usage"}, {"host", "fetcher-sas-00"}}, EMetricType::RATE},
        {{{"sensor", "cpu_usage"}, {"host", "fetcher-sas-01"}}, EMetricType::GAUGE},
        {{{"sensor", "mem_usage"}, {"host", "fetcher-sas-01"}}, EMetricType::RATE},
    });
    resp1.set_total_count(77);
    resp1.set_truncated(false);

    auto resp2 = CreateResponse({
        {{{"sensor", "cpu_usage"}, {"host", "fetcher-sas-02"}}, EMetricType::GAUGE},
        {{{"sensor", "mem_usage"}, {"host", "fetcher-sas-02"}}, EMetricType::RATE},
        {{{"sensor", "cpu_usage"}, {"host", "fetcher-sas-03"}}, EMetricType::GAUGE},
        {{{"sensor", "mem_usage"}, {"host", "fetcher-sas-03"}}, EMetricType::RATE},
    });
    resp2.set_total_count(100);
    resp2.set_truncated(true);

    TFindMerger merger{"solomon", 2};
    merger.AddResponse(SasR0, ShardKey, resp1);
    merger.AddResponse(SasR1, ShardKey, resp2);

    auto result = merger.Finish();
    ASSERT_TRUE(result);
    EXPECT_EQ(result->Strings.Size(), 11u);
    EXPECT_EQ(result->Metrics.size(), 2u);
    EXPECT_EQ(result->TotalCount, 100u);
    EXPECT_TRUE(result->Truncated);

    auto resultStrings = result->Strings.Build();

    for (const auto& metric: result->Metrics) {
        if (metric.Type == EMetricType::GAUGE) {
            // memstore does not support names right now
            EXPECT_EQ(resultStrings[metric.Name], "");
            EXPECT_EQ(
                    Resolve(resultStrings, metric.Labels),
                    Labels({
                        "project", "solomon",
                        "cluster", "prod",
                        "service", "fetcher",
                        "sensor", "cpu_usage",
                        "host", "fetcher-sas-00",
                    }));
            // memstore does not have stockpile ids
            EXPECT_EQ(metric.StockpileIds[EReplica::R0], TStockpileId{});
            EXPECT_EQ(metric.StockpileIds[EReplica::R1], TStockpileId{});
        } else if (metric.Type == EMetricType::RATE) {
            // memstore does not support names right now
            EXPECT_EQ(resultStrings[metric.Name], "");
            EXPECT_EQ(
                    Resolve(resultStrings, metric.Labels),
                    Labels({
                        "project", "solomon",
                        "cluster", "prod",
                        "service", "fetcher",
                        "sensor", "mem_usage",
                        "host", "fetcher-sas-00",
                    }));
            // memstore does not have stockpile ids
            EXPECT_EQ(metric.StockpileIds[EReplica::R0], TStockpileId{});
            EXPECT_EQ(metric.StockpileIds[EReplica::R1], TStockpileId{});
        } else {
            FAIL() << "unexpected metric type";
        }
    }
}

TEST(TFindMergerTest, SumFromSameReplicas) {
    auto resp = CreateResponse({
        {{{"sensor", "cpu_usage"}, {"host", "fetcher-sas-00"}}, EMetricType::GAUGE},
    });
    resp.set_total_count(10);

    TFindMerger merger{"solomon", 100};
    merger.AddResponse(TClusterId{EDc::Sas, EReplica::R0}, ShardKey, resp);
    merger.AddResponse(TClusterId{EDc::Vla, EReplica::R0}, ShardKey, resp);
    merger.AddResponse(TClusterId{EDc::Man, EReplica::R0}, ShardKey, resp);

    auto result = merger.Finish();
    ASSERT_TRUE(result);
    EXPECT_EQ(result->TotalCount, 30u);
}

TEST(TFindMergerTest, MaxFromDifferentReplica) {
    auto resp = CreateResponse({
        {{{"sensor", "cpu_usage"}, {"host", "fetcher-sas-00"}}, EMetricType::GAUGE},
    });
    resp.set_total_count(10);

    TFindMerger merger{"solomon", 100};
    merger.AddResponse(TClusterId{EDc::Sas, EReplica::R0}, ShardKey, resp);
    merger.AddResponse(TClusterId{EDc::Vla, EReplica::R0}, ShardKey, resp);
    merger.AddResponse(TClusterId{EDc::Man, EReplica::R0}, ShardKey, resp);

    merger.AddResponse(TClusterId{EDc::Sas, EReplica::R1}, ShardKey, resp);
    merger.AddResponse(TClusterId{EDc::Vla, EReplica::R1}, ShardKey, resp);
    merger.AddResponse(TClusterId{EDc::Man, EReplica::R1}, ShardKey, resp);
    merger.AddResponse(TClusterId{EDc::Myt, EReplica::R1}, ShardKey, resp);

    auto result = merger.Finish();
    ASSERT_TRUE(result);
    EXPECT_EQ(result->TotalCount, 40u);
}
