#include "requester.h"
#include "merger.h"

#include <solomon/services/dataproxy/lib/datasource/sts/base/requester.h>

#include <solomon/libs/cpp/labels/known_keys.h>

using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

using TLabelKeysReq = TMemStoreShardEvents::TLabelKeysReq;
using TLabelKeysResp = TMemStoreShardEvents::TLabelKeysResp;

using yandex::monitoring::memstore::LabelKeysRequest;
using yandex::monitoring::memstore::LabelKeysResponse;

class TMemStoreLabelKeysRequester: public TMemStoreRequester<TMemStoreLabelKeysRequester, TLabelKeysReq, TLabelKeysResp> {
    using TBase = TMemStoreRequester<TMemStoreLabelKeysRequester, TLabelKeysReq, TLabelKeysResp>;
public:
    TMemStoreLabelKeysRequester(
            TRequesterContextPtr ctx,
            TLabelKeysQuery query,
            IResultHandlerPtr<TLabelKeysResult> handler,
            TSpanId span)
        : TBase{std::move(ctx), TShardSelector::FromSelectors(query.Project, query.Selectors), query.Deadline, std::move(span)}
        , Project_{query.Project}
        , Merger_{query.Selectors, query.Project}
        , Request_{ToMemStoreRequest(std::move(query))}
        , Handler_{std::move(handler)}
    {
    }

    void OnResponse(TClusterId, const TShardSubKey&, const LabelKeysResponse& resp) {
        Merger_.AddResponse(resp);
    }

    void OnError(TClusterId clusterId, TShardId shardId, grpc::StatusCode statusCode, TString message) {
        Merger_.AddError(clusterId, shardId, statusCode, std::move(message));
    }

    void OnFinish(EDataSourceStatus status, TString&& message) {
        if (status == EDataSourceStatus::OK) {
            Handler_->OnSuccess(Merger_.Finish());
        } else {
            // TODO: merge errors
            Handler_->OnError(std::move(Project_), status, std::move(message));
        }
    }

    const LabelKeysRequest& RequestTemplate() const {
        return Request_;
    }

private:
    TString Project_;
    TLabelKeysMerger Merger_;
    const LabelKeysRequest Request_;
    IResultHandlerPtr<TLabelKeysResult> Handler_;
};

} // namespace

LabelKeysRequest ToMemStoreRequest(TLabelKeysQuery query) {
    // clear PCS from selectors, because MemStore handle only per shard requests
    query.Selectors.Remove(NLabels::LABEL_PROJECT);
    query.Selectors.Remove(NLabels::LABEL_CLUSTER);
    query.Selectors.Remove(NLabels::LABEL_SERVICE);

    // prepare request template for all shards (field `num_id` intentionally  not filled)
    LabelKeysRequest req;
    req.set_selectors(ToString(query.Selectors));
    req.set_from_millis(query.Time.From.MilliSeconds());
    req.set_to_millis(query.Time.To.MilliSeconds());
    return req;
}

std::unique_ptr<NActors::IActor> MemStoreLabelKeysRequester(
        TRequesterContextPtr ctx,
        TLabelKeysQuery query,
        IResultHandlerPtr<TLabelKeysResult> handler,
        TSpanId traceCtx)
{
    return std::make_unique<TMemStoreLabelKeysRequester>(std::move(ctx), std::move(query), std::move(handler), std::move(traceCtx));
}

} // namespace NSolomon::NDataProxy
