#include <solomon/services/dataproxy/lib/datasource/sts/label_keys/merger.h>
#include <solomon/services/memstore/api/memstore_service.pb.h>

#include <solomon/libs/cpp/yasm/constants/labels.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace yandex::monitoring::memstore;

LabelKeysResponse CreateResponse(const std::vector<TString>& keys) {
    LabelKeysResponse resp;
    NStringPool::TStringPoolBuilder strings;

    for (const auto& key: keys) {
        resp.add_keys_idx(strings.Put(key));
    }

    auto pool = strings.Build(yandex::solomon::common::StringPool_Compression_LZ4);
    pool.Swap(resp.mutable_string_pool());
    return resp;
}

// TODO: move to a better place
std::vector<TStringBuf> Resolve(const NStringPool::TStringPool& strings, const std::vector<ui32>& ids) {
    std::vector<TStringBuf> result;
    result.reserve(ids.size());

    for (ui32 id: ids) {
        result.push_back(strings[id]);
    }
    return result;
}

TEST(TLabelKeysMergerTest, EmptyResponse) {
    TLabelKeysMerger merger{TSelectors{}, "solomon"};
    merger.AddResponse(LabelKeysResponse{});

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 2u);
    EXPECT_EQ(result->Keys.size(), 2u);

    auto resultStrings = result->Strings.Build();

    EXPECT_THAT(
            Resolve(resultStrings, result->Keys),
            testing::ElementsAreArray({"cluster", "service"}));
}

TEST(TLabelKeysMergerTest, NoClusterService) {
    auto resp = CreateResponse({"zero", "one", "two", "three"});

    TLabelKeysMerger merger{ParseSelectors("{some = metric, cluster=foo, service=bar}"), "solomon"};
    merger.AddResponse(resp);

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 4u);
    EXPECT_EQ(result->Keys.size(), 4u);

    auto resultStrings = result->Strings.Build();

    EXPECT_THAT(
            Resolve(resultStrings, result->Keys),
            testing::ElementsAreArray({"zero", "one", "two", "three"}));
}

TEST(TLabelKeysMergerTest, MultipleResponses_NoIntersecions) {
    auto resp1 = CreateResponse({"zero", "one", "two"});
    auto resp2 = CreateResponse({"three", "four", "five"});

    TLabelKeysMerger merger{TSelectors{}, "solomon"};
    merger.AddResponse(resp1);
    merger.AddResponse(resp2);

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 8u);
    EXPECT_EQ(result->Keys.size(), 8u);

    auto resultStrings = result->Strings.Build();

    EXPECT_THAT(
            Resolve(resultStrings, result->Keys),
            testing::ElementsAreArray({"zero", "one", "two", "three", "four", "five", "cluster", "service"}));
}

TEST(TLabelKeysMergerTest, MultipleResponses_WithIntersecions) {
    auto resp1 = CreateResponse({"zero", "one", "two"});
    auto resp2 = CreateResponse({"one", "two", "three"});

    TLabelKeysMerger merger{TSelectors{}, "solomon"};
    merger.AddResponse(resp1);
    merger.AddResponse(resp2);

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 6u);
    EXPECT_EQ(result->Keys.size(), 6u);

    auto resultStrings = result->Strings.Build();

    EXPECT_THAT(
            Resolve(resultStrings, result->Keys),
            testing::ElementsAreArray({"zero", "one", "two", "three", "cluster", "service"}));
}

// for viewing YASM data SOLOMON-7804
TEST(TLabelKeysMergerTest, ResponseWithSelf) {
    auto resp = CreateResponse({NYasm::AGGREGATED_MARKER, NYasm::AGGREGATED_MARKER, "two", "three"});

    TLabelKeysMerger merger{TSelectors{}, "solomon"};
    merger.AddResponse(resp);

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 4u);
    EXPECT_EQ(result->Keys.size(), 4u);

    auto resultStrings = result->Strings.Build();

    EXPECT_THAT(
            Resolve(resultStrings, result->Keys),
            testing::ElementsAreArray({"two", "three", "cluster", "service"}));
}
