#include "merger.h"

#include <solomon/services/dataproxy/lib/datasource/yasm/labels.h>
#include <solomon/services/memstore/api/memstore_service.pb.h>

#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/libs/cpp/yasm/constants/labels.h>

namespace NSolomon::NDataProxy {

using yandex::monitoring::memstore::LabelValuesResponse;

void TLabelValuesMerger::AddResponse(TClusterId clusterId, const LabelValuesResponse& resp) {
    NStringPool::TStringPool respPool{resp.string_pool()};

    if (Labels_.empty()) {
        Labels_.reserve(resp.labels_size());
    }

    for (const auto& labelProto: resp.labels()) {
        ui32 keyId = Strings_.Put(respPool[labelProto.key_idx()]);

        auto& labelState = Labels_[keyId];
        labelState.MetricCount[clusterId.Replica()] += labelProto.total_count();
        labelState.Truncated[clusterId.Replica()] |= labelProto.truncated();

        if (labelState.Values.empty()) {
            labelState.Values.reserve(labelProto.values_idx_size());
        }

        for (const auto& valueIdx: labelProto.values_idx()) {
            if (labelState.Values.size() >= Limit_) {
                break;
            }
            if (respPool[valueIdx] != NYasm::AGGREGATED_MARKER) { // for viewing YASM data SOLOMON-7804
                labelState.Values.insert(Strings_.Put(respPool[valueIdx]));
            }
        }
    }

    MetricCount_[clusterId.Replica()] += resp.metric_count();
}

void TLabelValuesMerger::AddError(TClusterId clusterId, TShardId shardId, grpc::StatusCode statusCode, TString message) {
    Y_UNUSED(clusterId, shardId, statusCode, message);
    // TODO: aggregate per shard errors
    Errors_.emplace_back(statusCode, std::move(message));
}

std::unique_ptr<TLabelValuesResult> TLabelValuesMerger::Finish() {
    auto result = std::make_unique<TLabelValuesResult>();

    result->Project = std::move(Project_);
    result->Strings = std::move(Strings_);

    ui32 group;
    ui32 host;
    ui32 cluster;
    ui32 service;

    if (RemoveNonYasmLabels_) {
        group = result->Strings.Put(NLabels::LABEL_GROUP);
        host = result->Strings.Put(NLabels::LABEL_HOST);
        cluster = result->Strings.Put(NLabels::LABEL_CLUSTER);
        service = result->Strings.Put(NLabels::LABEL_SERVICE);
    }

    result->Labels.reserve(Labels_.size());
    for (auto& [keyId, state]: Labels_) {
        if (RemoveNonYasmLabels_ && (keyId == group || keyId == host || keyId == cluster || keyId == service)) {
            continue;
        }
        result->Labels.emplace_back();
        auto& label = result->Labels.back();

        label.Values = TVector<ui32>{std::begin(state.Values), std::end(state.Values)};
        label.Key = keyId;
        label.MetricCount = state.MetricCount.MaxValue();
        label.Truncated = state.Truncated.MaxValue();
    }

    result->Errors = std::move(Errors_);
    result->MetricCount = MetricCount_.MaxValue();
    return result;
}

} // namespace NSolomon::NDataProxy
