#include "requester.h"
#include "merger.h"

#include <solomon/services/dataproxy/lib/datasource/sts/base/requester.h>

#include <solomon/libs/cpp/labels/known_keys.h>

using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

using TLabelValuesReq = TMemStoreShardEvents::TLabelValuesReq;
using TLabelValuesResp = TMemStoreShardEvents::TLabelValuesResp;

using yandex::monitoring::memstore::LabelValuesRequest;
using yandex::monitoring::memstore::LabelValuesResponse;

class TMemStoreLabelValuesRequester: public TMemStoreRequester<TMemStoreLabelValuesRequester, TLabelValuesReq, TLabelValuesResp> {
    using TBase = TMemStoreRequester<TMemStoreLabelValuesRequester, TLabelValuesReq, TLabelValuesResp>;
public:
    TMemStoreLabelValuesRequester(
            TRequesterContextPtr ctx,
            TLabelValuesQuery query,
            IResultHandlerPtr<TLabelValuesResult> handler,
            TSpanId traceCtx)
        : TBase{std::move(ctx), TShardSelector::FromSelectors(query.Project, query.Selectors), query.Deadline, std::move(traceCtx)}
        , Project_{query.Project}
        , Merger_{static_cast<size_t>(query.Limit), query.Project, NYasm::IsYasmProject(query.Project) && query.Keys.empty()}
        , Request_{ToMemStoreRequest(std::move(query))}
        , Handler_{std::move(handler)}
    {
    }

    void OnResponse(TClusterId clusterId, const TShardSubKey&, const LabelValuesResponse& resp) {
        Merger_.AddResponse(clusterId, resp);
    }

    void OnError(TClusterId clusterId, TShardId shardId, grpc::StatusCode statusCode, TString message) {
        Merger_.AddError(clusterId, shardId, statusCode, std::move(message));
    }

    void OnFinish(EDataSourceStatus status, TString&& message) {
        if (status == EDataSourceStatus::OK) {
            Handler_->OnSuccess(Merger_.Finish());
        } else {
            // TODO: merge errors
            Handler_->OnError(std::move(Project_), status, std::move(message));
        }
    }

    const LabelValuesRequest& RequestTemplate() const {
        return Request_;
    }

private:
    TString Project_;
    TLabelValuesMerger Merger_;
    const LabelValuesRequest Request_;
    IResultHandlerPtr<TLabelValuesResult> Handler_;
};

} // namespace

LabelValuesRequest ToMemStoreRequest(TLabelValuesQuery query) {
    // clear PCS from selectors, because MemStore handle only per shard requests
    query.Selectors.Remove(NLabels::LABEL_PROJECT);
    query.Selectors.Remove(NLabels::LABEL_CLUSTER);
    query.Selectors.Remove(NLabels::LABEL_SERVICE);

    // prepare request template for all shards (field `num_id` intentionally  not filled)
    LabelValuesRequest req;
    req.set_selectors(ToString(query.Selectors));
    req.set_text_filter(std::move(query.TextFilter));

    auto* keys = req.mutable_keys();
    keys->Reserve(static_cast<int>(query.Keys.size()));
    for (auto& key: query.Keys) {
        keys->Add(std::move(key));
    }

    req.set_from_millis(query.Time.From.MilliSeconds());
    req.set_to_millis(query.Time.To.MilliSeconds());
    req.set_limit(query.Limit);
    return req;
}

std::unique_ptr<NActors::IActor> MemStoreLabelValuesRequester(
        TRequesterContextPtr ctx,
        TLabelValuesQuery query,
        IResultHandlerPtr<TLabelValuesResult> handler,
        TSpanId traceCtx)
{
    return std::make_unique<TMemStoreLabelValuesRequester>(std::move(ctx), std::move(query), std::move(handler), std::move(traceCtx));
}

} // namespace NSolomon::NDataProxy
