#include <solomon/services/dataproxy/lib/datasource/sts/label_values/merger.h>
#include <solomon/services/memstore/api/memstore_service.pb.h>

#include <solomon/libs/cpp/yasm/constants/labels.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace yandex::monitoring::memstore;

constexpr TClusterId SasR0{EDc::Sas, EReplica::R0};
constexpr TClusterId SasR1{EDc::Sas, EReplica::R1};

struct TLabelValues {
    TString Key;
    std::vector<TString> Values;
    ui32 TotalCount{0};
    bool Truncated{false};
};

LabelValuesResponse CreateResponse(const std::vector<TLabelValues>& labelValuesVec, ui32 metricCount) {
    NStringPool::TStringPoolBuilder strings;
    LabelValuesResponse resp;
    resp.set_metric_count(metricCount);

    for (const auto& lv: labelValuesVec) {
        auto* labelValues = resp.add_labels();
        labelValues->set_key_idx(strings.Put(lv.Key));
        labelValues->set_truncated(lv.Truncated);
        labelValues->set_total_count(lv.TotalCount);

        for (const auto& val: lv.Values) {
            labelValues->add_values_idx(strings.Put(val));
        }
    }

    auto pool = strings.Build(yandex::solomon::common::StringPool_Compression_LZ4);
    pool.Swap(resp.mutable_string_pool());
    return resp;
}

// TODO: move to a better place
std::vector<TStringBuf> Resolve(const NStringPool::TStringPool& strings, const std::vector<ui32>& ids) {
    std::vector<TStringBuf> result;
    result.reserve(ids.size());

    for (ui32 id: ids) {
        result.push_back(strings[id]);
    }
    return result;
}

TEST(TLabelValuesMergerTest, EmptyResponse) {
    TLabelValuesMerger merger{10, "solomon", false};
    merger.AddResponse(SasR0, LabelValuesResponse{});

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 0u);
    EXPECT_TRUE(result->Labels.empty());
    EXPECT_EQ(result->MetricCount, 0u);
}

TEST(TLabelValuesMergerTest, SingleResponse) {
    auto resp = CreateResponse({
        {"en", {"zero", "one", "two"}, 5, true},
        {"de", {"nul", "ein", "zwei", "drei", "vier"}, 10, false},
    }, 10);

    TLabelValuesMerger merger{10, "solomon", false};
    merger.AddResponse(SasR0, resp);

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 10u);
    EXPECT_EQ(result->Labels.size(), 2u);
    EXPECT_EQ(result->MetricCount, 10u);

    auto resultStrings = result->Strings.Build();

    for (auto& l: result->Labels) {
        if (resultStrings[l.Key] == "en") {
            EXPECT_THAT(
                    Resolve(resultStrings, l.Values),
                    testing::UnorderedElementsAreArray({"zero", "one", "two"}));
            EXPECT_EQ(5u, l.MetricCount);
            EXPECT_TRUE(l.Truncated);
        } else if (resultStrings[l.Key] == "de") {
            EXPECT_THAT(
                    Resolve(resultStrings, l.Values),
                    testing::UnorderedElementsAreArray({"nul", "ein", "zwei", "drei", "vier"}));
            EXPECT_EQ(10u, l.MetricCount);
            EXPECT_FALSE(l.Truncated);
        } else {
            FAIL() << "unexpected label key";
        }
    }
}

TEST(TLabelValuesMergerTest, SingleResponses_Limit) {
    auto resp = CreateResponse({
        {"en", {"zero", "one", "two", "three", "four" }, 5, false},
        {"de", {"nul", "ein", "zwei", "drei", "vier"}, 5, false},
    }, 10);

    TLabelValuesMerger merger{3, "solomon", false};
    merger.AddResponse(SasR0, resp);

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 8u);
    EXPECT_EQ(result->Labels.size(), 2u);
    EXPECT_EQ(result->MetricCount, 10u);

    auto resultStrings = result->Strings.Build();

    for (auto& l: result->Labels) {
        if (resultStrings[l.Key] == "en") {
            EXPECT_THAT(
                    Resolve(resultStrings, l.Values),
                    testing::UnorderedElementsAreArray({"zero", "one", "two"}));
            EXPECT_EQ(5u, l.MetricCount);
            EXPECT_FALSE(l.Truncated);
        } else if (resultStrings[l.Key] == "de") {
            EXPECT_THAT(
                    Resolve(resultStrings, l.Values),
                    testing::UnorderedElementsAreArray({"nul", "ein", "zwei"}));
            EXPECT_EQ(5u, l.MetricCount);
            EXPECT_FALSE(l.Truncated);
        } else {
            FAIL() << "unexpected label key";
        }
    }
}

TEST(TLabelValuesMergerTest, MultipleResponses_NoIntersecions) {
    auto resp1 = CreateResponse({
        {"en", {"zero", "one", "two"}, 5, true},
        {"de", {"nul", "ein", "zwei", "drei", "vier"}, 10, false},
    }, 10);

    auto resp2 = CreateResponse({
        {"fr", {"zéro", "une", "deux"}, 3, true},
        {"es", {"cero", "uno", "dos", "tres"}, 4, false},
    }, 7);

    TLabelValuesMerger merger{10, "solomon", false};
    merger.AddResponse(SasR0, resp1);
    merger.AddResponse(SasR1, resp2);

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 19u);
    EXPECT_EQ(result->Labels.size(), 4u);
    EXPECT_EQ(result->MetricCount, 10u);

    auto resultStrings = result->Strings.Build();

    for (auto& l: result->Labels) {
        if (resultStrings[l.Key] == "en") {
            EXPECT_THAT(
                    Resolve(resultStrings, l.Values),
                    testing::UnorderedElementsAreArray({"zero", "one", "two"}));
            EXPECT_EQ(l.MetricCount, 5u);
            EXPECT_TRUE(l.Truncated);
        } else if (resultStrings[l.Key] == "de") {
            EXPECT_THAT(
                    Resolve(resultStrings, l.Values),
                    testing::UnorderedElementsAreArray({"nul", "ein", "zwei", "drei", "vier"}));
            EXPECT_EQ(l.MetricCount, 10u);
            EXPECT_FALSE(l.Truncated);
        } else if (resultStrings[l.Key] == "fr") {
            EXPECT_THAT(
                    Resolve(resultStrings, l.Values),
                    testing::UnorderedElementsAreArray({"zéro", "une", "deux"}));
            EXPECT_EQ(l.MetricCount, 3u);
            EXPECT_TRUE(l.Truncated);
        } else if (resultStrings[l.Key] == "es") {
            EXPECT_THAT(
                    Resolve(resultStrings, l.Values),
                    testing::UnorderedElementsAreArray({"cero", "uno", "dos", "tres"}));
            EXPECT_EQ(l.MetricCount, 4u);
            EXPECT_FALSE(l.Truncated);
        } else {
            FAIL() << "unexpected label key";
        }
    }
}

TEST(TLabelValuesMergerTest, MultipleResponses_WithIntersecions) {
    auto resp1 = CreateResponse({
        {"en", {"zero", "two"}, 2, false},
        {"de", {"ein", "drei"}, 3, false},
    }, 4);

    auto resp2 = CreateResponse({
        {"en", {"zero", "one", "two"}, 5, true},
        {"de", {"nul", "ein", "zwei"}, 10, false},
    }, 10);

    TLabelValuesMerger merger{10, "solomon", false};
    merger.AddResponse(SasR0, resp1);
    merger.AddResponse(SasR1, resp2);

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 9u);
    EXPECT_EQ(result->Labels.size(), 2u);
    EXPECT_EQ(result->MetricCount, 10u);

    auto resultStrings = result->Strings.Build();

    for (auto& l: result->Labels) {
        if (resultStrings[l.Key] == "en") {
            EXPECT_THAT(
                    Resolve(resultStrings, l.Values),
                    testing::UnorderedElementsAreArray({"zero", "one", "two"}));
            EXPECT_EQ(5u, l.MetricCount);
            EXPECT_TRUE(l.Truncated);
        } else if (resultStrings[l.Key] == "de") {
            EXPECT_THAT(
                    Resolve(resultStrings, l.Values),
                    testing::UnorderedElementsAreArray({"nul", "ein", "zwei", "drei"}));
            EXPECT_EQ(10u, l.MetricCount);
            EXPECT_FALSE(l.Truncated);
        } else {
            FAIL() << "unexpected label key";
        }
    }
}

TEST(TLabelValuesMergerTest, SumFromSameReplicas) {
    auto resp = CreateResponse({
        {"key1", {}, 5, false},
        {"key2", {}, 7, true},
    }, 11);

    TLabelValuesMerger merger{10, "solomon", false};
    merger.AddResponse(TClusterId{EDc::Sas, EReplica::R0}, resp);
    merger.AddResponse(TClusterId{EDc::Vla, EReplica::R0}, resp);
    merger.AddResponse(TClusterId{EDc::Man, EReplica::R0}, resp);

    auto result = merger.Finish();
    ASSERT_TRUE(result);
    EXPECT_EQ(result->MetricCount, 33u);

    auto resultStrings = result->Strings.Build();

    for (const auto& l: result->Labels) {
        if (resultStrings[l.Key] == "key1") {
            EXPECT_EQ(l.MetricCount, 15u);
            EXPECT_FALSE(l.Truncated);
        } else if (resultStrings[l.Key] == "key2") {
            EXPECT_EQ(l.MetricCount, 21u);
            EXPECT_TRUE(l.Truncated);
        } else {
            FAIL() << "unexpected label key";
        }
    }
}

TEST(TLabelValuesMergerTest, MaxFromDifferentReplica) {
    auto resp = CreateResponse({
        {"key1", {}, 5, false},
        {"key2", {}, 7, true},
    }, 11);

    TLabelValuesMerger merger{10, "solomon", false};
    merger.AddResponse(TClusterId{EDc::Sas, EReplica::R0}, resp);
    merger.AddResponse(TClusterId{EDc::Vla, EReplica::R0}, resp);
    merger.AddResponse(TClusterId{EDc::Man, EReplica::R0}, resp);

    merger.AddResponse(TClusterId{EDc::Sas, EReplica::R1}, resp);
    merger.AddResponse(TClusterId{EDc::Vla, EReplica::R1}, resp);
    merger.AddResponse(TClusterId{EDc::Man, EReplica::R1}, resp);
    merger.AddResponse(TClusterId{EDc::Myt, EReplica::R1}, resp);

    auto result = merger.Finish();
    ASSERT_TRUE(result);
    EXPECT_EQ(result->MetricCount, 44u);

    auto resultStrings = result->Strings.Build();

    for (const auto& l: result->Labels) {
        if (resultStrings[l.Key] == "key1") {
            EXPECT_EQ(l.MetricCount, 20u);
            EXPECT_FALSE(l.Truncated);
        } else if (resultStrings[l.Key] == "key2") {
            EXPECT_EQ(l.MetricCount, 28u);
            EXPECT_TRUE(l.Truncated);
        } else {
            FAIL() << "unexpected label key";
        }
    }
}

// for viewing YASM data SOLOMON-7804
TEST(TLabelValuesMergerTest, ResponseWithSelf) {
    auto resp = CreateResponse({
        {"en", {NYasm::AGGREGATED_MARKER, NYasm::AGGREGATED_MARKER, "two"}},
    }, 10);

    TLabelValuesMerger merger{10, "solomon", false};
    merger.AddResponse(SasR0, resp);

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Labels.size(), 1u);
    EXPECT_EQ(result->Strings.Size(), 2u);

    auto resultStrings = result->Strings.Build();

    auto& l = result->Labels[0];
    EXPECT_THAT(Resolve(resultStrings, l.Values), testing::UnorderedElementsAreArray({"two"}));
}
