#include "merger.h"

#include <solomon/services/dataproxy/lib/timeseries/merger.h>
#include <solomon/services/dataproxy/lib/timeseries/protobuf.h>
#include <solomon/services/memstore/api/memstore_service.pb.h>

#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/yasm/constants/labels.h>

#include <solomon/libs/cpp/proto_convert/metric_type.h>

namespace NSolomon::NDataProxy {

void TReadManyMerger::AddResponse(TClusterId clusterId, const TShardSubKey& shardKey, const yandex::monitoring::memstore::ReadManyResponse& resp) {
    Y_UNUSED(clusterId);

    NStringPool::TStringPool respPool{resp.string_pool()};

    // TODO: make it once
    ui32 projectKey = Strings_.Put(NLabels::LABEL_PROJECT);
    ui32 projectValue = Strings_.Put(Project_);

    ui32 clusterKey = Strings_.Put(NLabels::LABEL_CLUSTER);
    ui32 clusterValue = Strings_.Put(shardKey.Cluster);

    ui32 serviceKey = Strings_.Put(NLabels::LABEL_SERVICE);
    ui32 serviceValue = Strings_.Put(shardKey.Service);

    for (const auto& metric: resp.metrics()) {
        TMetricKey<ui32> metricKey;
        metricKey.Labels.emplace_back(projectKey, projectValue);
        metricKey.Labels.emplace_back(clusterKey, clusterValue);
        metricKey.Labels.emplace_back(serviceKey, serviceValue);

        // TODO: support names
        for (int i = 0, size = metric.labels_idx_size(); i < size; ) {
            // TODO: labels must be sorted by the key string
            auto key = respPool[metric.labels_idx(i++)];
            auto value = respPool[metric.labels_idx(i++)];

            // TODO: remove as soon as metric name is fully supported
            if (key == NYasm::LABEL_SIGNAL) {
                metricKey.Name = Strings_.Put(value);
                continue;
            }

            ui32 keyId = Strings_.Put(key);
            ui32 valueId = Strings_.Put(value);
            metricKey.Labels.emplace_back(keyId, valueId);
        }

        auto it = Metrics_.find(metricKey);
        if (it == Metrics_.end()) {
            if (Metrics_.size() >= Limit_) {
                // do not add more metrics
                continue;
            }
            it = Metrics_.emplace(metricKey, TMetricData{}).first;
        }

        auto& metricData = it->second;
        // TODO: propperly merge metrics in one replica from different DCs
        metricData.Types[clusterId.Replica()] = NSolomon::FromProto(metric.type());
        metricData.TimeSeries[clusterId.Replica()] = FromProto(metric.type(), metric.time_series());
    }
}

void TReadManyMerger::AddError(TClusterId clusterId, TShardId shardId, grpc::StatusCode statusCode, TString message) {
    Y_UNUSED(clusterId, shardId, statusCode, message);
    // TODO: aggregate per shard errors
    Errors_.emplace_back(statusCode, std::move(message));
}

std::unique_ptr<ITimeSeries> TReadManyMerger::Merge(TMetricData& data) {
    auto metricType = NMonitoring::EMetricType::UNKNOWN;
    TMergingIterator<TMaxMerger> it;

    for (EReplica r: KnownReplicas) {
        if (auto& ts = data.TimeSeries[r]) {
            it.AddTimeSeries(*ts);
            if (metricType == NMonitoring::EMetricType::UNKNOWN) {
                metricType = ts->Type();
            }
        }
    }

    if (it.Empty()) {
        return nullptr;
    }

    // TODO: do not recompress single non empty time series
    return std::make_unique<TCompressedTimeSeries>(metricType, it.Columns(), it);
}

std::unique_ptr<TReadManyResult> TReadManyMerger::Finish() {
    auto result = std::make_unique<TReadManyResult>();
    result->Strings = std::move(Strings_);
    result->Metrics.reserve(Metrics_.size());

    for (auto& [key, data]: Metrics_) {
        result->Metrics.emplace_back();
        auto& metric = result->Metrics.back();
        metric.TimeSeries = Merge(data);

        // TODO: support aggregates

        metric.Meta.Type = (metric.TimeSeries != nullptr)
             ? metric.TimeSeries->Type()
             : data.Types.MaxValue(); // TODO: better types merge
        metric.Meta.Name = key.Name;
        metric.Meta.Labels = key.Labels; // TODO: better to move labels
    }

    result->Errors = std::move(Errors_);
    return result;
}

} // namespace NSolomon::NDataProxy
