#include "requester.h"
#include "merger.h"

#include <solomon/services/dataproxy/lib/datasource/sts/base/requester.h>

#include <solomon/libs/cpp/labels/known_keys.h>

#include <util/generic/mem_copy.h>

using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

using TReadManyReq = TMemStoreShardEvents::TReadManyReq;
using TReadManyResp = TMemStoreShardEvents::TReadManyResp;

using yandex::monitoring::memstore::ReadManyRequest;
using yandex::monitoring::memstore::ReadManyResponse;
using yandex::solomon::common::StringPool;

TShardSelector CreateShardSelector(const TReadManyQuery& query) {
    if (query.Lookup) {
        return TShardSelector::FromSelectors(query.Project, query.Lookup->Selectors);
    }

    if (auto& rk = query.ResolvedKeys) {
        return TShardSelector::FromLabels(query.Project, rk->Strings, rk->CommonLabels);
    }

    Y_FAIL("ReadMany query without Lookup and ResolvedKeys");
}

class TMemStoreReadManyRequester: public TMemStoreRequester<TMemStoreReadManyRequester, TReadManyReq, TReadManyResp> {
    using TBase = TMemStoreRequester<TMemStoreReadManyRequester, TReadManyReq, TReadManyResp>;
public:
    TMemStoreReadManyRequester(
            TRequesterContextPtr ctx,
            TReadManyQuery query,
            IResultHandlerPtr<TReadManyResult> handler,
            TSpanId traceCtx)
        : TBase{std::move(ctx), CreateShardSelector(query), query.Deadline, std::move(traceCtx)}
        , Project_{query.Project}
        , Merger_{query.Lookup ? static_cast<size_t>(query.Lookup->Limit) : Max<size_t>(), query.Project}
        , Request_{ToMemStoreRequest(std::move(query))}
        , Handler_{std::move(handler)}
    {
    }

    void OnResponse(TClusterId clusterId, const TShardSubKey& key, const ReadManyResponse& resp) {
        try {
            Merger_.AddResponse(clusterId, key, resp);
        } catch (...) {
            MON_WARN(Sts, CurrentExceptionMessage());
            OnError(clusterId, 0, grpc::StatusCode::INTERNAL, CurrentExceptionMessage());
        }
    }

    void OnError(TClusterId clusterId, TShardId shardId, grpc::StatusCode statusCode, TString message) {
        Merger_.AddError(clusterId, shardId, statusCode, std::move(message));
    }

    void OnFinish(EDataSourceStatus status, TString&& message) {
        if (status == EDataSourceStatus::OK) {
            Handler_->OnSuccess(Merger_.Finish());
        } else {
            // TODO: merge errors
            Handler_->OnError(std::move(Project_), status, std::move(message));
        }
    }

    const ReadManyRequest& RequestTemplate() const {
        return Request_;
    }

private:
    TString Project_;
    TReadManyMerger Merger_;
    const ReadManyRequest Request_;
    IResultHandlerPtr<TReadManyResult> Handler_;
};

void LabelsToProto(const TLabels<ui32>& labels, google::protobuf::RepeatedField<ui32>* field) {
    field->Reserve(static_cast<int>(labels.size()) * 2);
    for (const auto& label: labels) {
        *field->AddAlreadyReserved() = label.Key;
        *field->AddAlreadyReserved() = label.Value;
    }
}

void LabelsWithoutShardToProto(const TLabels<ui32>& labels, const NStringPool::TStringPool& strings, google::protobuf::RepeatedField<ui32>* field) {
    field->Reserve(static_cast<int>(labels.size()) * 2);
    for (const auto& label: labels) {
        TStringBuf labelKeyStr = strings[label.Key];
        if (labelKeyStr == NLabels::LABEL_PROJECT ||
            labelKeyStr == NLabels::LABEL_CLUSTER ||
            labelKeyStr == NLabels::LABEL_SERVICE)
        {
            continue;
        }
        *field->AddAlreadyReserved() = label.Key;
        *field->AddAlreadyReserved() = label.Value;
    }
}

void FillLookup(std::unique_ptr<TReadManyQuery::TLookup> lookup, ReadManyRequest* req) {
    // clear PCS from selectors, because MemStore handle only per shard requests
    lookup->Selectors.Remove(NLabels::LABEL_PROJECT);
    lookup->Selectors.Remove(NLabels::LABEL_CLUSTER);
    lookup->Selectors.Remove(NLabels::LABEL_SERVICE);

    auto* lookupProto = req->mutable_lookup();
    lookupProto->set_selectors(ToString(lookup->Selectors));
    lookupProto->set_limit(lookup->Limit);
}

void FillResolvedKeys(std::unique_ptr<TReadManyQuery::TResolvedKeys> resolvedKeys, ReadManyRequest* req) {
    auto* resolvedKeysProto = req->mutable_resolved_keys();
    resolvedKeys->Strings.ToProto(StringPool::LZ4, resolvedKeysProto->mutable_string_pool());

    if (!resolvedKeys->CommonLabels.empty()) {
        LabelsWithoutShardToProto(resolvedKeys->CommonLabels, resolvedKeys->Strings, resolvedKeysProto->mutable_common_labels_idx());
    }

    auto& metricKeys = resolvedKeys->MetricKeys;
    if (!metricKeys.empty()) {
        auto* metricKeysProto = resolvedKeysProto->mutable_metric_labels();
        metricKeysProto->Reserve(static_cast<int>(metricKeys.size()));

        for (const auto& metricKey: metricKeys) {
            // TODO: support metric name

            auto* labelsProto = metricKeysProto->Add();
            LabelsToProto(metricKey.Labels, labelsProto->mutable_labels_idx());
        }
    }
}

} // namespace

ReadManyRequest ToMemStoreRequest(TReadManyQuery query) {
    // prepare request template for all shards (field `num_id` intentionally  not filled)
    ReadManyRequest req;
    req.set_max_timeseries_format(query.MaxTimeSeriesFormat);
    req.set_from_millis(query.Time.From.MilliSeconds());
    req.set_to_millis(query.Time.To.MilliSeconds());

    if (!query.Operations.empty()) {
        auto* operationsProto = req.mutable_operations();
        operationsProto->Reserve(static_cast<int>(query.Operations.size()));
        for (auto& operation: query.Operations) {
            operationsProto->Add()->Swap(&operation);
        }
    }

    if (query.Lookup) {
        FillLookup(std::move(query.Lookup), &req);
    } else if (query.ResolvedKeys) {
        FillResolvedKeys(std::move(query.ResolvedKeys), &req);
    }

    return req;
}

std::unique_ptr<NActors::IActor> MemStoreReadManyRequester(
        TRequesterContextPtr ctx,
        TReadManyQuery query,
        IResultHandlerPtr<TReadManyResult> handler,
        TSpanId traceCtx)
{
    return std::make_unique<TMemStoreReadManyRequester>(std::move(ctx), std::move(query), std::move(handler), std::move(traceCtx));
}

} // namespace NSolomon::NDataProxy
