#include <solomon/services/dataproxy/lib/datasource/sts/read_many/requester.h>
#include <solomon/services/memstore/api/memstore_service.pb.h>

#include <solomon/libs/cpp/stockpile_codec/format.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <google/protobuf/util/message_differencer.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace yandex::monitoring::memstore;

using google::protobuf::util::MessageDifferencer;

TEST(TReadManyRequesterTest, ToMemStoreRequest_Lookup) {
    auto from = TInstant::ParseIso8601("2021-02-03T04:05:06");
    auto to = TInstant::ParseIso8601("2021-02-03T05:06:07");

    TReadManyQuery query;
    query.Project = "solomon";
    query.Deadline = TDuration::Seconds(30).ToDeadLine();
    query.Time = TTimeRange{.From = from, .To = to};
    query.Producer = yandex::solomon::common::RequestProducer::STAFF;
    query.MaxTimeSeriesFormat = static_cast<ui32>(NStockpile::EFormat::DELETED_SHARDS_39);

    yandex::solomon::math::Operation op1;
    op1.mutable_cast()->set_type(yandex::solomon::model::MetricType::DGAUGE);
    query.Operations.push_back(op1);

    yandex::solomon::math::Operation op2;
    op2.mutable_downsampling()->set_grid_millis(15'000);
    query.Operations.push_back(op2);

    {
        query.Lookup = std::make_unique<TReadManyQuery::TLookup>();
        query.Lookup->Selectors = ParseSelectors("{service=fetcher, signal=cpu_usage, cluster=testing}");
        query.Lookup->Limit = 10;
    }

    auto req = ToMemStoreRequest(std::move(query));
    EXPECT_EQ(req.num_id(), 0u); // num_id must not be filled
    EXPECT_EQ(req.from_millis(), from.MilliSeconds());
    EXPECT_EQ(req.to_millis(), to.MilliSeconds());
    EXPECT_EQ(req.max_timeseries_format(), static_cast<ui32>(NStockpile::EFormat::DELETED_SHARDS_39));

    ASSERT_EQ(req.operations_size(), 2);
    EXPECT_TRUE(MessageDifferencer::Equals(req.operations(0), op1));
    EXPECT_TRUE(MessageDifferencer::Equals(req.operations(1), op2));

    ASSERT_TRUE(req.has_lookup());
    auto& lookup = req.lookup();

    EXPECT_EQ(lookup.selectors(), "{'signal' == 'cpu_usage'}"); // PCS were removed
    EXPECT_EQ(lookup.limit(), 10u);
}

TEST(TReadManyRequesterTest, ToMemStoreRequest_ResolvedKeys) {
    auto from = TInstant::ParseIso8601("2021-02-03T04:05:06");
    auto to = TInstant::ParseIso8601("2021-02-03T05:06:07");

    TReadManyQuery query;
    query.Project = "solomon";
    query.Deadline = TDuration::Seconds(30).ToDeadLine();
    query.Time = TTimeRange{.From = from, .To = to};
    query.Producer = yandex::solomon::common::RequestProducer::STAFF;
    query.MaxTimeSeriesFormat = static_cast<ui32>(NStockpile::EFormat::IDEMPOTENT_WRITE_38);

    yandex::solomon::math::Operation op1;
    op1.mutable_cast()->set_type(yandex::solomon::model::MetricType::DGAUGE);
    query.Operations.push_back(op1);

    yandex::solomon::math::Operation op2;
    op2.mutable_downsampling()->set_grid_millis(15'000);
    query.Operations.push_back(op2);

    {
        query.ResolvedKeys = std::make_unique<TReadManyQuery::TResolvedKeys>();

        NStringPool::TStringPoolBuilder strings;
        query.ResolvedKeys->CommonLabels.emplace_back(strings.Put("cluster"), strings.Put("test"));
        query.ResolvedKeys->CommonLabels.emplace_back(strings.Put("service"), strings.Put("stockpile"));

        query.ResolvedKeys->MetricKeys.push_back(TMetricKey<ui32>{
            .Name = 0,
            .Labels = {
                {strings.Put("signal"), strings.Put("cpu_usage")},
                {strings.Put("host"), strings.Put("stockpile-sas-00")}
            }
        });
        query.ResolvedKeys->MetricKeys.push_back(TMetricKey<ui32>{
            .Name = 0,
            .Labels = {
                {strings.Put("signal"), strings.Put("mem_usage")},
                {strings.Put("host"), strings.Put("stockpile-sas-01")}
            }
        });

        query.ResolvedKeys->Strings = strings.Build();
    }

    auto req = ToMemStoreRequest(std::move(query));
    EXPECT_EQ(req.num_id(), 0u); // num_id must not be filled
    EXPECT_EQ(req.from_millis(), from.MilliSeconds());
    EXPECT_EQ(req.to_millis(), to.MilliSeconds());
    EXPECT_EQ(req.max_timeseries_format(), static_cast<ui32>(NStockpile::EFormat::IDEMPOTENT_WRITE_38));

    ASSERT_EQ(req.operations_size(), 2);
    EXPECT_TRUE(MessageDifferencer::Equals(req.operations(0), op1));
    EXPECT_TRUE(MessageDifferencer::Equals(req.operations(1), op2));

    ASSERT_TRUE(req.has_resolved_keys());
    auto& resolvedKeys = req.resolved_keys();

    ASSERT_TRUE(resolvedKeys.has_string_pool());
    auto strings = NStringPool::TStringPool{resolvedKeys.string_pool()};
    EXPECT_EQ(strings.Size(), 10u);

    ASSERT_EQ(resolvedKeys.common_labels_idx_size(), 0); // PCS were removed

    ASSERT_EQ(resolvedKeys.metric_labels_size(), 2);
    {
        auto& m = resolvedKeys.metric_labels(0);
        ASSERT_EQ(m.labels_idx_size(), 4);
        EXPECT_EQ(strings[m.labels_idx(0)], "signal");
        EXPECT_EQ(strings[m.labels_idx(1)], "cpu_usage");
        EXPECT_EQ(strings[m.labels_idx(2)], "host");
        EXPECT_EQ(strings[m.labels_idx(3)], "stockpile-sas-00");
    }
    {
        auto& m = resolvedKeys.metric_labels(1);
        ASSERT_EQ(m.labels_idx_size(), 4);
        EXPECT_EQ(strings[m.labels_idx(0)], "signal");
        EXPECT_EQ(strings[m.labels_idx(1)], "mem_usage");
        EXPECT_EQ(strings[m.labels_idx(2)], "host");
        EXPECT_EQ(strings[m.labels_idx(3)], "stockpile-sas-01");
    }
}
