#include "sts_page.h"

#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/actors/poison/poisoner.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

namespace NSolomon::NDataProxy {
namespace {

class TClusterSelfMonPage: public NActors::TActor<TClusterSelfMonPage> {
public:
    explicit TClusterSelfMonPage(NActors::TActorId cluster) noexcept
        : NActors::TActor<TClusterSelfMonPage>{&TThis::StateFunc}
        , Cluster_{cluster}
    {
    }

    STFUNC(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            HFunc(NSelfMon::TEvPageDataReq, OnRequest)
            hFunc(NActors::TEvents::TEvPoison, OnPoison)
        }
    }

    void OnRequest(NSelfMon::TEvPageDataReq::TPtr ev, const NActors::TActorContext& ctx) {
        ctx.Send(ev->Forward(Cluster_));
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

public:
    NActors::TActorId Cluster_;
};

class TStsSelfMonPage: public NActors::TActorBootstrapped<TStsSelfMonPage> {
public:
    explicit TStsSelfMonPage(std::map<TClusterId, NActors::TActorId> clusters) noexcept
        : Clusters_{std::move(clusters)}
    {
    }

    void Bootstrap() {
        Become(&TThis::StateFunc);

        for (auto& [clusterId, clusterBackend]: Clusters_) {
            TString clusterIdStr = ToString(clusterId);

            auto memStorePage = NSelfMon::RegisterPage(
                    *NActors::TActorContext::ActorSystem(),
                    "/sts/memstore/" + clusterIdStr,
                    "MemStore " + clusterIdStr,
                    std::make_unique<TClusterSelfMonPage>(clusterBackend),
                    true);

            SubPages_.insert(memStorePage);
        }
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSelfMon::TEvPageDataReq, OnRequest)
            hFunc(NActors::TEvents::TEvPoison, OnPoison)
        }
    }

private:
    void OnRequest(NSelfMon::TEvPageDataReq::TPtr ev) {
        using namespace yandex::monitoring::selfmon;

        Page page;
        auto* component = page.mutable_component();
        auto* list = component->mutable_list();
        auto* links = list->mutable_reference();

        for (auto& [clusterId, clusterBackend]: Clusters_) {
            TString clusterIdStr = ToString(clusterId);

            auto* memstoreRef = links->add_values();
            memstoreRef->set_page("/sts/memstore/" + clusterIdStr);
            memstoreRef->set_title("MemStore " + clusterIdStr);
        }

        Send(ev->Sender, new NSelfMon::TEvPageDataResp{std::move(page)});
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr ev) {
        PoisonAll(ev, SubPages_);
        PassAway();
    }

private:
    std::map<TClusterId, NActors::TActorId> Clusters_;
    std::set<NActors::TActorId> SubPages_;
};

} // namespace

std::unique_ptr<NActors::IActor> StsSelfMonPage(std::map<TClusterId, NActors::TActorId> clusters) {
    return std::make_unique<TStsSelfMonPage>(std::move(clusters));
}

} // namespace NSolomon::NDataProxy
