#include "sts.h"

#include <solomon/services/dataproxy/config/datasource_config.pb.h>
#include <solomon/services/dataproxy/lib/cluster_id/cluster_id.h>
#include <solomon/services/dataproxy/lib/cluster_map/cluster_map.h>
#include <solomon/services/dataproxy/lib/datasource/datasource.h>
#include <solomon/services/dataproxy/lib/datasource/sts/find/requester.h>
#include <solomon/services/dataproxy/lib/datasource/sts/label_keys/requester.h>
#include <solomon/services/dataproxy/lib/datasource/sts/label_values/requester.h>
#include <solomon/services/dataproxy/lib/datasource/sts/read_many/requester.h>
#include <solomon/services/dataproxy/lib/datasource/sts/selfmon/sts_page.h>
#include <solomon/services/dataproxy/lib/datasource/sts/unique_labels/requester.h>
#include <solomon/services/dataproxy/lib/memstore/cluster.h>

#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>
#include <solomon/libs/cpp/clients/memstore/rpc.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

using namespace NActors;
using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

class TShortTermSource: public IDataSource {
public:
    TShortTermSource(TActorRuntime& runtime, std::map<TClusterId, TActorId> clusters)
        : Runtime_{runtime}
    {
        RequesterCtx_ = MakeIntrusive<TRequesterContext>();
        RequesterCtx_->Clusters = std::move(clusters);
    }

    bool CanHandle(const TQuery& /*query*/) const override {
//        if (query.Time.To < (TInstant::Now() - STS_STORAGE_PERIOD)) {
//            // time range outside of STS storage
//            return false;
//        }
        // TODO: return false for TResolveOneQuery and TResolveManyQuery
        return true;
    }

    void Find(TFindQuery query, IResultHandlerPtr<TFindResult> handler, TSpanId traceCtx) const override {
        // TODO: define executor pool
        Runtime_.Register(MemStoreFindRequester(RequesterCtx_, std::move(query), std::move(handler), std::move(traceCtx)));
    }

    void ResolveOne(TResolveOneQuery query, IResultHandlerPtr<TResolveOneResult> handler, TSpanId traceCtx) const override {
        TRACING_SPAN_END(traceCtx);
        // TODO: better error status
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "ResolveOne is not supported by STS datasource");
    }

    void ResolveMany(TResolveManyQuery query, IResultHandlerPtr<TResolveManyResult> handler, TSpanId traceCtx) const override {
        TRACING_SPAN_END(traceCtx);
        // TODO: better error status
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "ResolveOne is not supported by STS datasource");
    }

    void MetricNames(TMetricNamesQuery query, IResultHandlerPtr<TMetricNamesResult> handler, TSpanId traceCtx) const override {
        TRACING_SPAN_END(traceCtx);
        // TODO: better error status
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "MetricNames is not yet implemented in STS datasource");
    }

    void LabelKeys(TLabelKeysQuery query, IResultHandlerPtr<TLabelKeysResult> handler, TSpanId traceCtx) const override {
        Runtime_.Register(MemStoreLabelKeysRequester(RequesterCtx_, std::move(query), std::move(handler), std::move(traceCtx)));
    }

    void LabelValues(TLabelValuesQuery query, IResultHandlerPtr<TLabelValuesResult> handler, TSpanId traceCtx) const override {
        Runtime_.Register(MemStoreLabelValuesRequester(RequesterCtx_, std::move(query), std::move(handler), std::move(traceCtx)));
    }

    void UniqueLabels(TUniqueLabelsQuery query, IResultHandlerPtr<TUniqueLabelsResult> handler, TSpanId traceCtx) const override {
        Runtime_.Register(MemStoreUniqueLabelsRequester(RequesterCtx_, std::move(query), std::move(handler), std::move(traceCtx)));
    }

    void ReadOne(TReadOneQuery query, IResultHandlerPtr<TReadOneResult> handler, TSpanId) const override {
        // TODO: better error status
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "ReadOne is not supported by STS datasource");
    }

    void ReadMany(TReadManyQuery query, IResultHandlerPtr<TReadManyResult> handler, TSpanId traceCtx) const override {
        Runtime_.Register(MemStoreReadManyRequester(RequesterCtx_, std::move(query), std::move(handler), std::move(traceCtx)));
    }

    void WaitUntilInitialized() const override {
        // TODO: wait until RequesterCtx_->Clusters is initialized
    }

private:
    TActorRuntime& Runtime_;
    TIntrusivePtr<TRequesterContext> RequesterCtx_;
};

} // namespace

TIntrusivePtr<IDataSource> ShortTermSource(
        TActorRuntime& runtime,
        const TShortTermSourceConfig& config,
        const TClusterMap& clusterMap,
        NMonitoring::TMetricRegistry& registry,
        TString clientId)
{
    auto memStoreRpc = NMemStore::CreateClusterGrpc(config.GetMemStoreClient(), registry, std::move(clientId));

    std::map<TClusterId, std::vector<TString>> nodesPerCluster;
    for (const TString& address: memStoreRpc->Addresses()) {
        auto clusterId = clusterMap[address];
        Y_ENSURE(clusterId.has_value(), "cannot determine cluster id for address \'" << address << '\'');
        nodesPerCluster[*clusterId].push_back(address);
    }
    Y_ENSURE(nodesPerCluster.size() <= (std::size(KnownDcs) * std::size(KnownReplicas)),
             "too many STS clusters: " << nodesPerCluster.size());

    std::map<TClusterId, TActorId> clusters;
    for (auto& [clusterId, addresses]: nodesPerCluster) {
        Y_ENSURE(!addresses.empty(), "empty MemStore addresses list in cluster " << clusterId);

        clusters[clusterId] = runtime.Register(
                MemStoreCluster({
                    .ClusterId = clusterId,
                    .Addresses = std::move(addresses),
                    .Rpc = memStoreRpc,
                }).release(),
                TMailboxType::HTSwap,
                runtime.FindExecutorByName(config.GetMemStoreClient().GetThreadPoolName()));
    }

    NSelfMon::RegisterPage(runtime.ActorSystem(), "/sts", "STS Clusters", StsSelfMonPage(clusters));
    return MakeIntrusive<TShortTermSource>(runtime, std::move(clusters));
}

} // namespace NSolomon::NDataProxy
