#include "requester.h"
#include "merger.h"

#include <solomon/services/dataproxy/lib/datasource/sts/base/requester.h>

#include <solomon/libs/cpp/labels/known_keys.h>

using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

using TUniqueLabelsReq = TMemStoreShardEvents::TUniqueLabelsReq;
using TUniqueLabelsResp = TMemStoreShardEvents::TUniqueLabelsResp;

using yandex::monitoring::memstore::UniqueLabelsRequest;
using yandex::monitoring::memstore::UniqueLabelsResponse;

class TMemStoreUniqueLabelsRequester: public TMemStoreRequester<TMemStoreUniqueLabelsRequester, TUniqueLabelsReq, TUniqueLabelsResp> {
    using TBase = TMemStoreRequester<TMemStoreUniqueLabelsRequester, TUniqueLabelsReq, TUniqueLabelsResp>;
public:
    TMemStoreUniqueLabelsRequester(
            TRequesterContextPtr ctx,
            TUniqueLabelsQuery query,
            IResultHandlerPtr<TUniqueLabelsResult> handler,
            TSpanId traceCtx)
        : TBase{std::move(ctx), TShardSelector::FromSelectors(query.Project, query.Selectors), query.Deadline, std::move(traceCtx)}
        , Project_{query.Project}
        , Request_{ToMemStoreRequest(std::move(query))}
        , Handler_{std::move(handler)}
    {
    }

    void OnResponse(TClusterId, const TShardSubKey&, const UniqueLabelsResponse& resp) {
        Merger_.AddResponse(resp);
    }

    void OnError(TClusterId clusterId, TShardId shardId, grpc::StatusCode statusCode, TString message) {
        Merger_.AddError(clusterId, shardId, statusCode, std::move(message));
    }

    void OnFinish(EDataSourceStatus status, TString&& message) {
        if (status == EDataSourceStatus::OK) {
            Handler_->OnSuccess(Merger_.Finish());
        } else {
            // TODO: merge errors
            Handler_->OnError(std::move(Project_), status, std::move(message));
        }
    }

    const UniqueLabelsRequest& RequestTemplate() const {
        return Request_;
    }

private:
    TString Project_;
    TUniqueLabelsMerger Merger_;
    const UniqueLabelsRequest Request_;
    IResultHandlerPtr<TUniqueLabelsResult> Handler_;
};

} // namespace

UniqueLabelsRequest ToMemStoreRequest(TUniqueLabelsQuery query) {
    // clear PCS from selectors, because MemStore handle only per shard requests
    query.Selectors.Remove(NLabels::LABEL_PROJECT);
    query.Selectors.Remove(NLabels::LABEL_CLUSTER);
    query.Selectors.Remove(NLabels::LABEL_SERVICE);

    // prepare request template for all shards (field `num_id` intentionally  not filled)
    UniqueLabelsRequest req;
    req.set_selectors(ToString(query.Selectors));

    auto* keys = req.mutable_keys();
    keys->Reserve(static_cast<int>(query.Keys.size()));
    for (auto& key: query.Keys) {
        keys->Add(std::move(key));
    }

    req.set_from_millis(query.Time.From.MilliSeconds());
    req.set_to_millis(query.Time.To.MilliSeconds());
    return req;
}

std::unique_ptr<NActors::IActor> MemStoreUniqueLabelsRequester(
        TRequesterContextPtr ctx,
        TUniqueLabelsQuery query,
        IResultHandlerPtr<TUniqueLabelsResult> handler,
        TSpanId traceCtx)
{
    return std::make_unique<TMemStoreUniqueLabelsRequester>(std::move(ctx), std::move(query), std::move(handler), std::move(traceCtx));
}

} // namespace NSolomon::NDataProxy
