#include <solomon/services/dataproxy/lib/datasource/sts/unique_labels/merger.h>
#include <solomon/services/memstore/api/memstore_service.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace yandex::monitoring::memstore;

UniqueLabelsResponse CreateResponse(const std::vector<TLabels<TString>>& labelsVec) {
    NStringPool::TStringPoolBuilder strings;

    UniqueLabelsResponse resp;
    for (const auto& labels: labelsVec) {
        auto* labelsProto = resp.add_labels();
        for (const auto& label: labels) {
            labelsProto->add_labels_idx(strings.Put(label.Key));
            labelsProto->add_labels_idx(strings.Put(label.Value));
        }
    }

    auto pool = strings.Build(yandex::solomon::common::StringPool_Compression_LZ4);
    pool.Swap(resp.mutable_string_pool());
    return resp;
}

// TODO: move to a better place
TLabels<TStringBuf> Resolve(const NStringPool::TStringPool& strings, const TLabels<ui32>& labels) {
    TLabels<TStringBuf> resolved;
    resolved.reserve(labels.size());

    for (const auto& l: labels) {
        resolved.push_back({strings[l.Key], strings[l.Value]});
    }
    return resolved;
}

// TODO: move to a better place
TLabel<TStringBuf> Label(TStringBuf key, TStringBuf value) {
    return {key, value};
}

TEST(TLabelsKeysMergerTest, EmptyResponse) {
    TUniqueLabelsMerger merger;
    merger.AddResponse(UniqueLabelsResponse{});

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 0u);
    EXPECT_TRUE(result->Labels.empty());
}

TEST(TLabelsKeysMergerTest, SingleResponse) {
    auto resp = CreateResponse({
        {{"key1", "value1"}, {"key2", "value2"}},
        {{"key3", "value3"}, {"key4", "value4"}, {"key5", "value5"}},
    });

    TUniqueLabelsMerger merger;
    merger.AddResponse(resp);

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 10u);
    EXPECT_EQ(result->Labels.size(), 2u);

    auto resultStrings = result->Strings.Build();

    for (const auto& labels: result->Labels) {
        if (labels.size() == 2) {
            EXPECT_THAT(Resolve(resultStrings, labels), testing::ElementsAreArray({
                Label("key1", "value1"),
                Label("key2", "value2"),
            }));
        } else if (labels.size() == 3) {
            EXPECT_THAT(Resolve(resultStrings, labels), testing::ElementsAreArray({
                Label("key3", "value3"),
                Label("key4", "value4"),
                Label("key5", "value5"),
            }));
        } else {
            FAIL() << "unexpected labels size";
        }
    }
}

TEST(TLabelsKeysMergerTest, MultipleResponses_NoIntersecions) {
    auto resp1 = CreateResponse({
        {{"key1", "value1"}, {"key2", "value2"}},
    });

    auto resp2 = CreateResponse({
        {{"key3", "value3"}, {"key4", "value4"}, {"key5", "value5"}},
    });

    TUniqueLabelsMerger merger;
    merger.AddResponse(resp1);
    merger.AddResponse(resp2);

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 10u);
    EXPECT_EQ(result->Labels.size(), 2u);

    auto resultStrings = result->Strings.Build();

    for (const auto& labels: result->Labels) {
        if (labels.size() == 2) {
            EXPECT_THAT(Resolve(resultStrings, labels), testing::ElementsAreArray({
                Label("key1", "value1"),
                Label("key2", "value2"),
            }));
        } else if (labels.size() == 3) {
            EXPECT_THAT(Resolve(resultStrings, labels), testing::ElementsAreArray({
                Label("key3", "value3"),
                Label("key4", "value4"),
                Label("key5", "value5"),
            }));
        } else {
            FAIL() << "unexpected labels size";
        }
    }
}

TEST(TLabelsKeysMergerTest, MultipleResponses_WithIntersecions) {
    auto resp1 = CreateResponse({
        {{"key1", "value1"}, {"key2", "value2"}},
    });

    auto resp2 = CreateResponse({
        {{"key1", "value1"}, {"key2", "value2"}},
        {{"key3", "value3"}, {"key4", "value4"}, {"key5", "value5"}},
    });

    TUniqueLabelsMerger merger;
    merger.AddResponse(resp1);
    merger.AddResponse(resp2);

    auto result = merger.Finish();
    ASSERT_TRUE(result);

    EXPECT_EQ(result->Strings.Size(), 10u);
    EXPECT_EQ(result->Labels.size(), 2u);

    auto resultStrings = result->Strings.Build();

    for (const auto& labels: result->Labels) {
        if (labels.size() == 2) {
            EXPECT_THAT(Resolve(resultStrings, labels), testing::ElementsAreArray({
                Label("key1", "value1"),
                Label("key2", "value2"),
            }));
        } else if (labels.size() == 3) {
            EXPECT_THAT(Resolve(resultStrings, labels), testing::ElementsAreArray({
                Label("key3", "value3"),
                Label("key4", "value4"),
                Label("key5", "value5"),
            }));
        } else {
            FAIL() << "unexpected labels size";
        }
    }
}
