#include "top.h"

#include <solomon/libs/cpp/ts_math/aggregation.h>
#include <solomon/libs/cpp/ts_model/aggregator.h>
#include <solomon/libs/cpp/ts_model/aggregator_avg.h>
#include <solomon/libs/cpp/ts_model/aggregator_count.h>
#include <solomon/libs/cpp/ts_model/aggregator_last.h>
#include <solomon/libs/cpp/ts_model/aggregator_min.h>
#include <solomon/libs/cpp/ts_model/aggregator_max.h>
#include <solomon/libs/cpp/ts_model/aggregator_sum.h>
#include <solomon/libs/cpp/ts_model/visit.h>
#include <solomon/libs/cpp/ts_math/error.h>

#include <solomon/protos/math/operation.pb.h>

namespace NSolomon::NDataProxy {

namespace {

NTsModel::EPointType FromMetricType(NMonitoring::EMetricType metricType) {
    switch (metricType) {
        case NMonitoring::EMetricType::UNKNOWN:
            return NTsModel::EPointType::Unknown;
        case NMonitoring::EMetricType::GAUGE:
            return NTsModel::EPointType::DGauge;
        case NMonitoring::EMetricType::COUNTER:
            return NTsModel::EPointType::Counter;
        case NMonitoring::EMetricType::RATE:
            return NTsModel::EPointType::Rate;
        case NMonitoring::EMetricType::IGAUGE:
            return NTsModel::EPointType::IGauge;
        case NMonitoring::EMetricType::HIST:
            return NTsModel::EPointType::Hist;
        case NMonitoring::EMetricType::HIST_RATE:
            return NTsModel::EPointType::HistRate;
        case NMonitoring::EMetricType::DSUMMARY:
            return NTsModel::EPointType::DSummary;
        case NMonitoring::EMetricType::LOGHIST:
            return NTsModel::EPointType::LogHist;
    }
}

template <typename TAggregator>
double Aggregate(const std::unique_ptr<ITimeSeriesIter>& iterator) {
    TAggregator aggregator;
    NTs::TVariantPoint point;
    while (iterator->Next(&point)) {
        aggregator.Add(point);
    }

    if constexpr (std::is_same_v<typename TAggregator::TOutput, NTsModel::TGaugePoint>) {
        return aggregator.Finish().ValueDivided();
    } else {
        return aggregator.Finish().Value;
    }
}

class TTopHandler: public IResultHandler<TReadManyResult> {
public:
    TTopHandler(IResultHandlerPtr<TReadManyResult> originalHandler, yandex::solomon::math::OperationTop operationTop)
        : OriginalHandler_(std::move(originalHandler))
        , OperationTop_(std::move(operationTop))
    {
    }

private:
    void OnSuccess(std::unique_ptr<TReadManyResult> result) override {
        TVector<std::pair<double, TReadManyResult::TMetricData*>> top;
        top.reserve(result->Metrics.size());
        for (auto& metric: result->Metrics) {
            auto pointType = FromMetricType(metric.Meta.Type);
            auto aggregation = NTsMath::AggregationFunction(OperationTop_.time_aggregation(), pointType);

            try {
                auto summary = NTsModel::Visit(
                        pointType,
                        [aggregation, iterator = metric.TimeSeries->Iterator()](auto traits) -> double {
                            using TPoint = typename decltype(traits)::TPoint;

                            if (aggregation == NTsModel::EAggregationFunction::Count) {
                                return Aggregate<NTsModel::TCountAggregator<TPoint>>(std::move(iterator));
                            }

                            if constexpr (traits.IsScalar) {
                                switch (aggregation) {
                                    case NTsModel::EAggregationFunction::Min:
                                        return Aggregate<NTsModel::TMinAggregator<TPoint>>(iterator);
                                    case NTsModel::EAggregationFunction::Max:
                                        return Aggregate<NTsModel::TMaxAggregator<TPoint>>(iterator);
                                    case NTsModel::EAggregationFunction::Sum:
                                        return Aggregate<NTsModel::TSumAggregator<TPoint>>(iterator);
                                    case NTsModel::EAggregationFunction::Last:
                                        return Aggregate<NTsModel::TLastAggregator<TPoint>>(iterator);
                                    case NTsModel::EAggregationFunction::Avg:
                                        return Aggregate<NTsModel::TAvgAggregator<TPoint>>(iterator);
                                    case NTsModel::EAggregationFunction::Count:
                                        return Aggregate<NTsModel::TCountAggregator<TPoint>>(iterator);
                                }
                            } else {
                                ythrow NTsMath::TTypeError{} << "unable to build top for time series of type "
                                                             << NTsModel::TPointTraits<TPoint>{}.Type();
                            }
                        });

                top.emplace_back(summary, &metric);
            } catch (const NTsMath::TTypeError& error) {
                OriginalHandler_->OnError("unknown_project", EDataSourceStatus::BAD_REQUEST, error.what());
                return;
            }
        }

        if (OperationTop_.asc()) {
            std::sort(top.begin(), top.end());
        } else {
            std::sort(top.begin(), top.end(), std::greater<>());
        }

        auto topResult = std::make_unique<TReadManyResult>();
        topResult->Strings = std::move(result->Strings);
        topResult->Errors = std::move(result->Errors);

        size_t metricCount = std::min<size_t>(top.size(), OperationTop_.limit());
        topResult->Metrics.reserve(metricCount);
        for (size_t i = 0; i < metricCount; ++i) {
            topResult->Metrics.push_back(std::move(*top[i].second));
        }
        OriginalHandler_->OnSuccess(std::move(topResult));
    }

    void OnError(TString&& project, EDataSourceStatus status, TString&& message) override {
        OriginalHandler_->OnError(std::move(project), status, std::move(message));
    }

private:
    IResultHandlerPtr<TReadManyResult> OriginalHandler_;
    yandex::solomon::math::OperationTop OperationTop_;
};

} // namespace

IResultHandlerPtr<TReadManyResult> MakeTopHandler(IResultHandlerPtr<TReadManyResult> originalHandler, yandex::solomon::math::OperationTop operationTop) {
    return MakeIntrusive<TTopHandler>(std::move(originalHandler), std::move(operationTop));
}

} // namespace NSolomon::NDataProxy
