#include "placement.h"

#include <solomon/services/dataproxy/lib/initialization/enable_initialization.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/http/client/curl/client.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/string_map/string_map.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/resource/resource.h>

#include <util/stream/file.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace NActors;
using namespace NSolomon::NTsdb;

constexpr TDuration REQUEST_INTERVAL = TDuration::Minutes(5);

class TPlacementActor: public NActors::TActorBootstrapped<TPlacementActor>, TPrivateEvents, public TEnableInitialization<TPlacementActor> {
    enum {
        RequestTsdbNodes = SpaceBegin,
        Error,
        Response,
        End,
    };
    static_assert(End < SpaceEnd, "too many events");

    struct TRequestTsdbNodes: NActors::TEventLocal<TRequestTsdbNodes, RequestTsdbNodes> {
    };

    struct TError: NActors::TEventLocal<TError, Error> {
        TString Group;
        TString Host;
        TString Message;

        TError(TString group, TString host, TString message)
            : Group{std::move(group)}
            , Host{std::move(host)}
            , Message{std::move(message)}
        {
        }
    };

    struct TResponse: NActors::TEventLocal<TResponse, Response> {
        TString Group;
        TString Host;
        TVector<TString> UserHosts;

        TResponse(TString group, TString host, TVector<TString> hosts)
            : Group{std::move(group)}
            , Host{std::move(host)}
            , UserHosts{std::move(hosts)}
        {
        }
    };

public:
    TPlacementActor(
            TDuration interval,
            std::shared_ptr<const TYasmGroupToTsdbHosts> groupToHosts,
            std::shared_ptr<ITsdbClusterRpc> tsdbCluster,
            TString fsCachePath,
            TDuration fsCacheLifetime)
        : RequestInterval_{interval}
        , GroupToHosts_{std::move(groupToHosts)}
        , TsdbCluster_{std::move(tsdbCluster)}
        , FsCachePath_{std::move(fsCachePath)}
        , FsCacheLifetime_{fsCacheLifetime}
    {
        Y_VERIFY(
                interval >= TDuration::Minutes(5),
                "interval should be >= 5min. Otherwise, wait for inflight requests correctly");
    }

    void Bootstrap() {
        Become(&TThis::Main);

        if (FsCachePath_) {
            try {
                if (auto userHostToGroups = ParseUserHostsPlacementFile(FsCachePath_, FsCacheLifetime_)) {
                    for (const auto& [userHost, groups]: *userHostToGroups) {
                        for (const auto& group: groups) {
                            UserHostToGroups_[userHost][group] = TInstant::Now();
                        }
                    }
                } else {
                    MON_INFO(TsdbPlacement, "fs cache is stale");
                }
            } catch (...) {
                MON_INFO(TsdbPlacement, "failed to parse a fs cache: " << CurrentExceptionMessage());
            }
        }

        OnRequestTsdbNodes(true);
    }

    STATEFN(Main) {
        switch (ev->GetTypeRewrite()) {
            sFunc(TRequestTsdbNodes, OnRequestTsdbNodes);
            hFunc(TError, OnError);
            hFunc(TResponse, OnResponse);
            hFunc(TPlacementActorEvents::TResolveHosts, OnResolveHosts);
            hFunc(TInitializationEvents::TSubscribe, OnInitializationSubscribe);
        }
    }

    static constexpr char ActorName[] = "Tsdb/Placement";

private:
    // FIXME(ivanzhukov): retries
    void OnError(const TError::TPtr& evPtr) {
        const auto& ev = *evPtr->Get();

        if (--InitialInflight_ == 0) {
            FinishInitialization();
        }

        MON_INFO(
                TsdbPlacement,
                "failed to get fetch_hosts response from " << ev.Host << " for group " << ev.Group << ": " << ev.Message);
    }

    void OnResponse(const TResponse::TPtr& evPtr) {
        const auto& ev = *evPtr->Get();

        if (--InitialInflight_ == 0) {
            FinishInitialization();
        }

        MON_DEBUG(TsdbPlacement, "got a response from " << ev.Host << " for group " << ev.Group);

        for (const auto& host : ev.UserHosts) {
            auto& groups = UserHostToGroups_[host];
            groups[ev.Group] = TInstant::Now();
        }
    }

    void FetchHosts(const TString& group, const TString& host) {
        auto* client = TsdbCluster_->Get(host);
        auto* as = NActors::TActorContext::ActorSystem();

        client->FetchHosts(group).Subscribe([=, self{SelfId()}](TFetchHostsResponseAsyncResponse f) {
            try {
                auto res = f.ExtractValueSync();

                if (res.Success()) {
                    as->Send(self, new TResponse{group, host, res.Value().Hosts});
                    return;
                }

                if (res.Error().Type() == TRequestError::EType::RequestInitializationFailed) {
                    return; // request has been cancelled. Actor system may be already dead SOLOMON-8372
                }

                as->Send(self, new TError{group, host, res.Error().Message()});
            } catch (...) {
                as->Send(self, new TError{group, host, CurrentExceptionMessage()});
            }
        });
    }

    void DeleteOldRecords() {
        auto now = TActivationContext::Now();

        for (auto hostIt = UserHostToGroups_.begin(); hostIt != UserHostToGroups_.end();) {
            auto& groups = hostIt->second;

            for (auto it = groups.begin(); it != groups.end();) {
                if (now - it->second > TDuration::Minutes(30)) {
                    groups.erase(it++);
                } else {
                    ++it;
                }
            }

            if (groups.empty()) {
                UserHostToGroups_.erase(hostIt++);
            } else {
                ++hostIt;
            }
        }

    }

    void SaveCacheToFs() {
        if (UserHostToGroups_.empty()) {
            return;
        }

        NJsonWriter::TBuf json;

        json.BeginObject();
        json.WriteKey("savedAtSeconds").WriteString(ToString(TActivationContext::Now().Seconds()));
        json.WriteKey("mapping");
        json.BeginObject();

        for (const auto& [userHost, groups]: UserHostToGroups_) {
            json.WriteKey(userHost);
            json.BeginList();

            for (const auto& [group, _]: groups) {
                json.WriteString(group);
            }

            json.EndList();
        }

        json.EndObject();
        json.EndObject();

        try {
            TUnbufferedFileOutput file{FsCachePath_};
            json.FlushTo(&file);
        } catch (...) {
            MON_ERROR(TsdbPlacement, "failed to write a fs cache: " << CurrentExceptionMessage());
        }
    }

    void OnRequestTsdbNodes(bool initial = false) {
        if (initial && !UserHostToGroups_.empty() && FsCacheLifetime_) {
            MON_INFO(TsdbPlacement, "starting from cache. Will request TSDB nodes after " << FsCacheLifetime_);
            FinishInitialization();
            Schedule(FsCacheLifetime_, new TRequestTsdbNodes{});
            return;
        } else {
            Schedule(RequestInterval_, new TRequestTsdbNodes{});
        }

        DeleteOldRecords();
        if (FsCachePath_) {
            SaveCacheToFs();
        }

        // TODO(ivanzhukov): scatter requests
        for (const auto& [group, hosts] : *GroupToHosts_) {
            for (const auto& host : hosts) {
                if (initial) {
                    ++InitialInflight_;
                }

                FetchHosts(group, host);
            }
        }
    }

    void OnResolveHosts(const TPlacementActorEvents::TResolveHosts::TPtr& evPtr) {
        auto& ev = *evPtr->Get();
        TStringMap<absl::flat_hash_set<TString>> userHostToGroups;

        for (const auto& host : ev.UserHosts) {
            if (auto it = UserHostToGroups_.find(host); it != UserHostToGroups_.end()) {
                for (const auto& groupIt : it->second) {
                    userHostToGroups[host].emplace(groupIt.first);
                }
            }
        }

        Send(evPtr->Sender, new TPlacementActorEvents::TResolveHostsResponse{std::move(userHostToGroups)});
    }

private:
    size_t InitialInflight_ = 0;
    TDuration RequestInterval_;
    std::shared_ptr<const TYasmGroupToTsdbHosts> GroupToHosts_;
    TStringMap<TStringMap<TInstant>> UserHostToGroups_;
    std::shared_ptr<ITsdbClusterRpc> TsdbCluster_;
    TString FsCachePath_;
    TDuration FsCacheLifetime_;
};

} // namespace

std::unique_ptr<NActors::IActor> PlacementActor(
        std::shared_ptr<const TYasmGroupToTsdbHosts> groupToHosts,
        std::shared_ptr<ITsdbClusterRpc> tsdbCluster,
        TString fsCachePath,
        TDuration fsCacheLifetime)
{
    return std::make_unique<TPlacementActor>(
            REQUEST_INTERVAL,
            std::move(groupToHosts),
            std::move(tsdbCluster),
            std::move(fsCachePath),
            fsCacheLifetime);
}

} // namespace NSolomon::NDataProxy
