#include "placement_config.h"

#include <solomon/libs/cpp/string_map/string_map.h>

#include <library/cpp/containers/absl_flat_hash/flat_hash_set.h>
#include <library/cpp/json/json_reader.h>

#include <util/generic/string.h>
#include <util/stream/file.h>
#include <util/string/cast.h>

namespace NSolomon::NDataProxy {
namespace {

NJson::TJsonValue ReadJsonFile(const TString& filePath) {
    NJson::TJsonValue jsonValue;

    TFileInput in{filePath};
    NJson::TJsonReaderConfig readerConfig;
    readerConfig.DontValidateUtf8 = true;

    NJson::ReadJsonTree(&in, &readerConfig, &jsonValue, true);
    return jsonValue;
}

} // namespace

std::shared_ptr<TYasmGroupToTsdbHosts> ParseGroupPlacementFile(const TString& filePath) {
    auto res = std::make_shared<TYasmGroupToTsdbHosts>();
    NJson::TJsonValue jsonValue;

    try {
        jsonValue = ReadJsonFile(filePath);
    } catch (...) {
        ythrow yexception() << "failed to parse a group placement in TSDB: " << CurrentExceptionMessage() << Endl;
    }

    for (const auto& [group, hosts]: jsonValue.GetMap()) {
        auto& resHosts = (*res)[group];
        for (auto hostIt = hosts.GetArray().cbegin(); hostIt != hosts.GetArray().cend(); ++hostIt) {
            resHosts.emplace_back(hostIt->GetString());
        }
        std::sort(resHosts.begin(), resHosts.end());
    }

    return res;
}

std::shared_ptr<TUserHostToGroups> ParseUserHostsPlacementFile(const TString& filePath, TDuration cacheLifetime) {
    auto res = std::make_shared<TUserHostToGroups>();
    NJson::TJsonValue jsonValue = ReadJsonFile(filePath);

    auto savedAt = TInstant::Seconds(FromString<ui64>(jsonValue["savedAtSeconds"].GetString()));
    if (TInstant::Now() - savedAt > cacheLifetime) {
        return {};
    }

    for (const auto& [userHost, groups]: jsonValue["mapping"].GetMap()) {
        auto& resUserHosts = (*res)[userHost];
        for (auto groupIt = groups.GetArray().cbegin(); groupIt != groups.GetArray().cend(); ++groupIt) {
            resUserHosts.emplace_back(groupIt->GetString());
        }
    }

    return res;
}

} // namespace NSolomon::NDataProxy
