#include "read_many_actor.h"
#include "placement.h"
#include "tsdb_timeseries.h"

#include <solomon/services/dataproxy/lib/datasource/request_manager.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/serializer.h>
#include <solomon/services/dataproxy/lib/timeseries/empty.h>
#include <solomon/services/dataproxy/lib/timeseries/vector.h>

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/libs/cpp/string_map/string_map.h>
#include <solomon/libs/cpp/ts_model/point_type.h>
#include <solomon/libs/cpp/ts_model/visit.h>
#include <solomon/libs/cpp/yasm/constants/labels.h>
#include <solomon/libs/cpp/yasm/constants/project.h>
#include <solomon/libs/cpp/yasm/convertion/convertion.h>

#include <solomon/protos/model/metric.pb.h>

#include <infra/yasm/interfaces/internal/history_api.pb.h>
#include <infra/yasm/stockpile_client/points.h>
#include <infra/yasm/stockpile_client/record_serialize_state.h>
#include <infra/yasm/zoom/components/serialization/history/history.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

#include <type_traits>

using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

using namespace ::NYasm::NInterfaces::NInternal;
using namespace NActors;
using namespace NHistDb::NStockpile;
using namespace NMonitoring;
using namespace NSolomon::NYasm;
using namespace NZoom::NAccumulators;
using namespace NZoom::NProtobuf;
using namespace yandex::solomon::model;

EDataSourceStatus RequestErrorToDataSourceStatus(TRequestError::EType reqError) {
    switch (reqError) {
        case TRequestError::EType::Unknown:
            return EDataSourceStatus::UNKNOWN;
        case TRequestError::EType::ConnectFailed:
            return EDataSourceStatus::BACKEND_UNAVAILABLE;
        case TRequestError::EType::ReadTimeout:
            return EDataSourceStatus::BACKEND_TIMEOUT;
        case TRequestError::EType::ResponseTooLarge:
            return EDataSourceStatus::BAD_REQUEST;
        case TRequestError::EType::DnsFailure:
            return EDataSourceStatus::BACKEND_UNAVAILABLE;
        case TRequestError::EType::UnsupportedProtocol:
            return EDataSourceStatus::BAD_REQUEST;
        case TRequestError::EType::UrlMalformat:
            return EDataSourceStatus::BAD_REQUEST;
        case TRequestError::EType::SocketError:
            return EDataSourceStatus::BACKEND_UNAVAILABLE;
        case TRequestError::EType::RequestInitializationFailed:
            return EDataSourceStatus::UNKNOWN;
        case TRequestError::EType::RequestQueueOverflow:
            return EDataSourceStatus::UNKNOWN;
    };
}

EDataSourceStatus TsdbErrorToDataSourceStatus(THistoryAggregatedSeries::EStatusCode statusCode) {
    switch (statusCode) {
        case THistoryAggregatedSeries::UNKNOWN:
            return EDataSourceStatus::UNKNOWN;
        case THistoryAggregatedSeries::OK:
            return EDataSourceStatus::OK;
        case THistoryAggregatedSeries::LIMIT_EXCEEDED:
            return EDataSourceStatus::BAD_REQUEST;
        case THistoryAggregatedSeries::TIME_EXCEEDED:
            return EDataSourceStatus::BACKEND_TIMEOUT;
        case THistoryAggregatedSeries::INTERNAL_ERROR:
            return EDataSourceStatus::BACKEND_ERROR;
        case THistoryAggregatedSeries::NOT_FOUND:
            return EDataSourceStatus::NOT_FOUND;
        case THistoryAggregatedSeries::PARTIAL:
            return EDataSourceStatus::BACKEND_ERROR;
        case THistoryAggregatedSeries::EStatusCode::
                THistoryAggregatedSeries_EStatusCode_THistoryAggregatedSeries_EStatusCode_INT_MAX_SENTINEL_DO_NOT_USE_:
        case THistoryAggregatedSeries::EStatusCode::
                THistoryAggregatedSeries_EStatusCode_THistoryAggregatedSeries_EStatusCode_INT_MIN_SENTINEL_DO_NOT_USE_:
            Y_FAIL("Incorrect enum value");
    }
}

struct TRequestWithUserHost {
    const TMetricKey<ui32>* MetricKey;
    TStringBuf UserHost;

    TRequestWithUserHost(const TMetricKey<ui32>* metricKey, TStringBuf userHost)
        : MetricKey{std::move(metricKey)}
        , UserHost{std::move(userHost)}
    {
    }
};

struct TPrivateEvents: private NSolomon::TPrivateEvents {
    enum {
        Response = SpaceBegin,
        Error,
        Timeout,
        End,
    };
    static_assert(End < SpaceEnd, "too many events");

    struct TResponse: NActors::TEventLocal<TResponse, Response> {
        TString Host;
        TMetricKeys MetricKeys;
        THistoryReadAggregatedResponse Response;

        TResponse(TString host, TMetricKeys metricKeys, THistoryReadAggregatedResponse resp)
            : Host{std::move(host)}
            , MetricKeys{std::move(metricKeys)}
            , Response{std::move(resp)}
        {
        }
    };

    struct TError: NActors::TEventLocal<TError, Error> {
        TString TsdbHost;
        TRequestError Error;

        TError(TString tsdbHost, TRequestError reqError)
            : TsdbHost{std::move(tsdbHost)}
            , Error{std::move(reqError)}
        {
        }
    };

    struct TTimeout: public TEventLocal<TTimeout, Timeout> {
        const TMetricKey<ui32>* MetricKey;

        explicit TTimeout(const TMetricKey<ui32>* metricKey)
            : MetricKey{metricKey}
        {
        }
    };
};

class TReadManyActor: public TActorBootstrapped<TReadManyActor> {
    using THosts = TVector<TString>;
    using TRequests = std::pair<TMetricKeys, THistoryReadAggregatedRequest>;

public:
    TReadManyActor(
            std::shared_ptr<const TYasmGroupToTsdbHosts> groupToHosts,
            NActors::TActorId placementActor,
            std::shared_ptr<NSolomon::NTsdb::ITsdbClusterRpc> tsdbCluster,
            TReadManyQuery query,
            IResultHandlerPtr<TReadManyResult> handler,
            NMonitoring::TMetricRegistry& registry,
            TSpanId traceCtx)
        : GroupToHosts_{std::move(groupToHosts)}
        , PlacementActor_{placementActor}
        , TsdbCluster_{std::move(tsdbCluster)}
        , Query_{std::move(query)}
        , Handler_{std::move(handler)}
        , Period_{TSDB_GRID}
        , Registry_{registry}
        , TraceCtx_{std::move(traceCtx)}
    {
        if (!Query_.Operations.empty()) {
            for (const auto& query: Query_.Operations) {
                if (!query.has_downsampling()) {
                    continue;
                }

                const auto& downsampling = query.downsampling();

                if (downsampling.grid_millis()) {
                    Period_ = TDuration::MilliSeconds(downsampling.grid_millis());
                    break;
                }
            }
        }
    }

    void Bootstrap() {
        Become(&TThis::Main);
        RequestManager_ = std::make_unique<TRequestsManager>(TsdbReadMany, SelfId());
        SendRequests();
    }

    STATEFN(Main) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TPrivateEvents::TResponse, OnResponse);
            hFunc(TPlacementActorEvents::TResolveHostsResponse, OnResolvedHosts);
            hFunc(TPrivateEvents::TError, OnError);
            hFunc(TPrivateEvents::TTimeout, OnTimeout);
        }
    }

    static constexpr char ActorName[] = "Tsdb/ReadMany";

private:
    NMonitoring::TLabels LabelsFromMetricKey(const TMetricKey<ui32>& metricKey) {
        NMonitoring::TLabels labels;
        const auto& pool = Query_.ResolvedKeys->Strings;
        for (const auto& l: metricKey.Labels) {
            labels.Add(pool[l.Key], pool[l.Value]);
        }
        for (const auto& cl: Query_.ResolvedKeys->CommonLabels) {
            labels.Add(pool[cl.Key], pool[cl.Value]);
        }
        return labels;
    }

    std::pair<TInstant, TInstant> ComputeTimeRange() {
        auto now = TActivationContext::Now();
        auto tsdbRangeRight = TInstant::Seconds(now.Seconds() - now.Seconds() % Period_.Seconds());
        auto tsdbRangeLeft = tsdbRangeRight - TDuration::Minutes(25);
        auto start = Max(tsdbRangeLeft, Query_.Time.From);
        start = Min(start, tsdbRangeRight);
        auto end = Min(tsdbRangeRight, Query_.Time.To);
        end = Max(end, tsdbRangeLeft);

        return {start, end};
    }

    THistoryRequest ConstructProtoRequest(const NMonitoring::TLabels& labels, TInstant start, TInstant end) {
        TStringBuf itype;
        if (!TStringBuf{Query_.Project}.AfterPrefix(YASM_PROJECT_PREFIX_STS1, itype) &&
            !TStringBuf{Query_.Project}.AfterPrefix(YASM_PROJECT_PREFIX_STS2, itype) &&
            !TStringBuf{Query_.Project}.AfterPrefix(STOCKPILE_YASM_PROJECTS_PREFIX, itype))
        {
            ythrow yexception() << "yasm project without a prefix: " << Query_.Project;
        }

        // FIXME(ivanzhukov): construct right from resolved keys (i.e. a string pool), without creating a new labels obj
        auto yasmKey = TYasmKey::FromLabels(itype, labels);

        NZoom::NProtobuf::THistoryRequest request{
                .Start = start,
                .End = end,
                .Period = Period_,
                .HostName = std::move(yasmKey.HostName),
                .RequestKey = std::move(yasmKey.RequestKey),
                .SignalName = std::move(yasmKey.SignalName),
                .ReplicaIndex = {},
                .DontAggregate = true,
        };
        return request;
    }

    TString DebugOutputForRequests(const absl::flat_hash_map<THosts, TRequests>& requests) {
        TStringBuilder sb;
        size_t hostsIdx = 0;

        for (const auto& [hosts, request]: requests) {
            if (hostsIdx++) { sb << ", "; }

            sb << '[';
            size_t hostIdx = 0;

            for (const auto& host: hosts) {
                if (hostIdx++) { sb << ", "; }
                sb << host;
            }

            sb << "]: [";

            size_t metricIdx = 0;
            for (const auto& metricKey: request.first) {
                if (metricIdx++) { sb << ", "; }

                sb << Query_.ResolvedKeys->Strings[metricKey->Name];
                sb << '{';

                size_t labelIdx = 0;
                for (const auto& l: metricKey->Labels) {
                    if (labelIdx++) { sb << ", "; }
                    sb << Query_.ResolvedKeys->Strings[l.Key] << '=' << Query_.ResolvedKeys->Strings[l.Value];
                }

                sb << '}';
            }

            sb << ']';
        }

        return sb;
    }

    void SendRequestsToHosts(const absl::flat_hash_map<THosts, TRequests>& requests) {
        MON_TRACE(TsdbReadMany, "gathering metrics from hosts: " << DebugOutputForRequests(requests));

        auto* as = TActivationContext::ActorSystem();
        auto selfId = SelfId();

        for (const auto& [hosts, request]: requests) {
            auto& metricKeys = request.first;
            auto& protoRequest = request.second;

            for (const auto& host: hosts) {
                auto* tsdbRpc = TsdbCluster_->Get(host);
                ++MaxInflight_;

                auto reqId = RequestManager_->NewRequest(EReplica::Unknown, metricKeys);

                auto span = TRACING_NEW_SPAN_START(TraceCtx_, "ReadAggregated on " << host);
                auto spanHolder = std::make_shared<TSpanId>(std::move(span));
                tsdbRpc->ReadAggregated(protoRequest).Subscribe([=](auto f) {
                    try {
                        auto v = f.ExtractValueSync();

                        if (v.Success()) {
                            as->Send(new IEventHandle{
                                    selfId, {}, new TPrivateEvents::TResponse{host, metricKeys, v.Extract()},
                                    0, reqId, nullptr, TSpanId(*spanHolder)
                            });
                            return;
                        }

                        as->Send(new IEventHandle{
                                selfId, {}, new TPrivateEvents::TError{host, v.ExtractError()},
                                0, reqId, nullptr, TSpanId(*spanHolder)
                        });
                    } catch (...) {
                        as->Send(new IEventHandle{
                                selfId, {},
                                new TPrivateEvents::TError{host, TRequestError{TRequestError::EType::Unknown,
                                        CurrentExceptionMessage()}},
                                0, reqId, nullptr, TSpanId(*spanHolder)
                        });
                    }
                });
            }
        }
        // TODO(ivanzhukov): timeout
        // TODO(ivanzhukov): retries
    }

    void AddRequestToHosts(
            const TMetricKey<ui32>* metricKey,
            const THistoryRequest& request,
            const TVector<TString>& hosts,
            absl::flat_hash_map<THosts, TRequests>& hostsToRequests)
    {
        auto& requests = hostsToRequests[hosts];
        requests.first.emplace_back(std::move(metricKey));
        NZoom::NProtobuf::THistoryRequestWriter reqWriter{requests.second};
        reqWriter.Add(request);
    }

    void SendRequests() {
        MON_TRACE(TsdbReadMany, "sending requests to TSDB to get " << Query_.ResolvedKeys->MetricKeys.size() << " metrics");
        QueryTimeRange_ = ComputeTimeRange();
        auto [start, end] = QueryTimeRange_;

        std::optional<TVector<TString>> userHostsToResolve;
        absl::flat_hash_map<THosts, TRequests> requests;
        for (const auto& metricKey : Query_.ResolvedKeys->MetricKeys) {
            TStringBuf group;
            TStringBuf host;

            for (auto l: metricKey.Labels) {
                const auto& key = Query_.ResolvedKeys->Strings[l.Key];
                const auto& value = Query_.ResolvedKeys->Strings[l.Value];

                if (key == NLabels::LABEL_GROUP) {
                    group = value;
                    break;
                } else if (key == NLabels::LABEL_HOST && value != NSolomon::NYasm::AGGREGATED_MARKER) {
                    host = value;
                    break;
                }
            }

            if (!group && !host) {
                TStringBuilder message;
                message << "query without the host or grop labels";
                MetricKeyToErrors_[&metricKey].emplace_back(EDataSourceStatus::BAD_REQUEST, std::move(message));
                continue;
            }

            if (group) {
                auto it = GroupToHosts_->find(group);
                if (it == GroupToHosts_->end()) {
                    MON_DEBUG(TsdbReadMany, "didn't find group " << group);
                    continue;
                }

                auto protoRequest = ConstructProtoRequest(LabelsFromMetricKey(metricKey), start, end);
                AddRequestToHosts(&metricKey, protoRequest, it->second, requests);
            } else {
                if (!userHostsToResolve) {
                    userHostsToResolve = TVector<TString>{};
                }

                userHostsToResolve->emplace_back(host);
                RequestsWithUserHost_.emplace_back(&metricKey, std::move(host));
            }
        }

        if (requests.empty() && !userHostsToResolve) {
            OnTsdbDone();
            return;
        }

        SendRequestsToHosts(requests);

        if (userHostsToResolve) {
            auto span = TRACING_NEW_SPAN_START(TraceCtx_, "resolve hosts");
            Send(PlacementActor_, new TPlacementActorEvents::TResolveHosts{std::move(*userHostsToResolve)}, 0, 0, std::move(span));
        }
    }

    void OnResolvedHosts(const TPlacementActorEvents::TResolveHostsResponse::TPtr& evPtr) {
        TRACING_SPAN_END_EV(evPtr);
        const auto& userHostToGroups = evPtr->Get()->UserHostToGroups;
        bool didSendAny = false;
        auto [start, end] = QueryTimeRange_;

        absl::flat_hash_map<THosts, TRequests> requests;
        for (const auto& request: RequestsWithUserHost_) {
            auto it = userHostToGroups.find(request.UserHost);
            if (it == userHostToGroups.end()) {
                TStringBuilder message;
                message << "didn't find any groups for user host " << request.UserHost;
                MetricKeyToErrors_[request.MetricKey].emplace_back(EDataSourceStatus::BAD_REQUEST, std::move(message));
                continue;
            }

            for (const auto& group: it->second) {
                auto groupIt = GroupToHosts_->find(group);
                if (groupIt == GroupToHosts_->end()) {
                    MON_DEBUG(TsdbReadMany, "didn't find group " << group << " for user host " << request.UserHost);
                    continue;
                }

                const auto& hosts = groupIt->second;
                didSendAny = true;

                auto labels = LabelsFromMetricKey(*request.MetricKey);
                labels.Add(NLabels::LABEL_GROUP, group);

                auto protoRequest = ConstructProtoRequest(labels, start, end);
                AddRequestToHosts(request.MetricKey, protoRequest, hosts, requests);
                break;
            }
        }
        SendRequestsToHosts(requests);
        RequestsWithUserHost_.clear();
        RequestsWithUserHost_.shrink_to_fit();

        if (!didSendAny) {
            OnTsdbDone();
            return;
        }
    }

    void OnError(const TPrivateEvents::TError::TPtr& evPtr) {
        TRACING_SPAN_END_EV(evPtr);
        auto& ev = *evPtr->Get();

        ui64 reqId = evPtr->Cookie;
        RequestManager_->ReportError(reqId);

        MON_TRACE(TsdbReadMany, "got an error from " << ev.TsdbHost << ": " << ev.Error.Message());
        // TODO(ivanzhukov): don't propagate internal errors that an end user shouldn't see
        TsdbHostToError_.emplace_back(std::move(ev.TsdbHost), std::move(ev.Error));

        if (RequestManager_->IsCompleted()) {
            OnTsdbDone();
        }
    }

    void OnResponse(const TPrivateEvents::TResponse::TPtr& evPtr) {
        TRACING_SPAN_END_EV(evPtr);
        auto& ev = *evPtr->Get();
        MON_TRACE(TsdbReadMany, "got a response from " << ev.Host << ": " << ev.Response);

        ui64 reqId = evPtr->Cookie;

        auto parsed = NZoom::NProtobuf::THistoryResponse::FromProto(ev.Response);
        Y_VERIFY(parsed.size() == ev.MetricKeys.size(), "must be one response for each metric keys");
        for (size_t index = 0; index < parsed.size(); ++index) {
            auto& response = parsed[index];
            const auto* metricKey = ev.MetricKeys[index];

            Registry_.Rate(
                    {{"sensor", "tsdb.response.status"},
                     {"status", THistoryAggregatedSeries::EStatusCode_Name(response.StatusCode)}})->Inc();

            if (response.StatusCode == THistoryAggregatedSeries::NOT_FOUND) {
                RequestManager_->ReportError(reqId, metricKey);
                MON_TRACE(TsdbReadMany, "metric not found " << LabelsFromMetricKey(*metricKey));
                continue;
            }

            if (response.StatusCode == THistoryAggregatedSeries::OK) {
                RequestManager_->ReportResponse(reqId, metricKey, Query_.SoftDeadline, [&](TInstant deadline, const TMetricKey<ui32>* metricKey) {
                    this->Schedule(deadline, new TPrivateEvents::TTimeout{metricKey});
                });
                MetricKeyToResponses_[metricKey].emplace_back(std::move(response));
                MON_TRACE(TsdbReadMany, "got metric " << LabelsFromMetricKey(*ev.MetricKeys[index]));
            } else {
                RequestManager_->ReportError(reqId, metricKey);
                TStringBuilder message;
                message << LabelsFromMetricKey(*metricKey) << ": "
                        << THistoryAggregatedSeries::EStatusCode_Name(response.StatusCode);

                MON_WARN(TsdbReadMany, "got an error from TSDB " << message);
                MetricKeyToErrors_[metricKey].emplace_back(
                        TsdbErrorToDataSourceStatus(response.StatusCode),
                        std::move(message));
            }
        }

        if (RequestManager_->IsCompleted()) {
            OnTsdbDone();
        }
    }

    void OnTimeout(TPrivateEvents::TTimeout::TPtr& ev) {
        auto metricKey = ev->Get()->MetricKey;
        RequestManager_->ReportTimeout(metricKey);

        if (RequestManager_->IsCompleted()) {
            OnTsdbDone();
        }
    }

    TDataSourceError MergeErrors() {
        EDataSourceStatus mergedStatus = EDataSourceStatus::UNKNOWN;
        TStringBuilder reply;

        reply << "got error from all TSDB nodes: ";

        for (const auto& [tsdbHost, error]: TsdbHostToError_) {
            auto status = RequestErrorToDataSourceStatus(error.Type());
            if (status != EDataSourceStatus::NOT_FOUND) {
                mergedStatus = status;
            }

            reply << '[' << tsdbHost << "] " << error.Type() << ": " << error.Message();
        }

        return TDataSourceError{mergedStatus, std::move(reply)};
    }

    void OnTsdbDone() {
        if (MetricKeyToResponses_.empty() && MaxInflight_ && TsdbHostToError_.size() == MaxInflight_) {
            auto mergedError = MergeErrors();
            ReportErrorAndPassAway(mergedError.Status, std::move(mergedError.Message));
            return;
        }

        auto readManyResult = std::make_unique<TReadManyResult>();

        for (auto& [metricKey, responses]: MetricKeyToResponses_) {
            auto& metric = readManyResult->Metrics.emplace_back();
            for (const auto& cl: Query_.ResolvedKeys->CommonLabels) {
                ui32 keyIdx = readManyResult->Strings.Put(Query_.ResolvedKeys->Strings[cl.Key]);
                ui32 valueIdx = readManyResult->Strings.Put(Query_.ResolvedKeys->Strings[cl.Value]);
                metric.Meta.Labels.emplace_back(keyIdx, valueIdx);
            }

            metric.Meta.Name = readManyResult->Strings.Put(Query_.ResolvedKeys->Strings[metricKey->Name]);
            for (const auto& l: metricKey->Labels) {
                // a hack to unify labels representation with one returned from the ResolveMany method in LTS,
                // so that metrics from STS and LTS are merged correctly
                if (Query_.ResolvedKeys->Strings[l.Key] == LABEL_SIGNAL) {
                    metric.Meta.Name = readManyResult->Strings.Put(Query_.ResolvedKeys->Strings[l.Value]);
                    continue;
                }

                ui32 keyIdx = readManyResult->Strings.Put(Query_.ResolvedKeys->Strings[l.Key]);
                ui32 valueIdx = readManyResult->Strings.Put(Query_.ResolvedKeys->Strings[l.Value]);
                metric.Meta.Labels.emplace_back(keyIdx, valueIdx);
            }

            // a hack to unify labels representation with one returned from the ResolveMany method in LTS,
            // so that metrics from STS and LTS are merged correctly
            metric.Meta.Labels.emplace_back(
                    readManyResult->Strings.Put(NLabels::LABEL_PROJECT),
                    readManyResult->Strings.Put(Query_.Project));

            auto merged = THistoryResponse::MergeMultiple(responses);

            if (merged.size() == 0) {
                ReportErrorAndPassAway(EDataSourceStatus::BACKEND_ERROR, "merged result was empty for " + ToString(LabelsFromMetricKey(*metricKey)));
                return;
            } 

            if (merged.size() > 1) {
                TStringBuilder message;
                message << "more than one response after merging by a single metric: size=" << merged.size() << " " << LabelsFromMetricKey(*metricKey);
                ReportErrorAndPassAway(EDataSourceStatus::BACKEND_ERROR, std::move(message));
                return;
            }

            auto& response = merged[0];
            if (response.Series.GetValues().empty()) {
                metric.Meta.Type = EMetricType::UNKNOWN;
                metric.TimeSeries = std::make_unique<TEmptyTimeSeries>(metric.Meta.Type);
            } else {
                auto aggrType = NHistDb::NStockpile::GetAggregationType(response.Series.GetName());
                NHistDb::NStockpile::TValueTypeDetector typeDetector{aggrType};
                for (auto&& point: response.Series.GetValues()) {
                    typeDetector.OnValue(point.GetValue());
                }
                auto protoType = typeDetector.GetType();
                auto pointTypeOrErr = NTsModel::FromProto(protoType);
                if (pointTypeOrErr.Fail()) {
                    TStringBuilder message;
                    message << "failed to convert a proto type " << protoType << " to EPointType " << pointTypeOrErr.Error();
                    MetricKeyToErrors_[metricKey].emplace_back(EDataSourceStatus::BACKEND_ERROR, std::move(message));
                    continue;
                }

                metric.TimeSeries = std::make_unique<TTsdbTimeSeries>(std::move(response), protoType, aggrType, Period_);
                metric.Meta.Type = metric.TimeSeries->Type();
            }
        }

        for (const auto& [tsdbHost, error]: TsdbHostToError_) {
            Registry_.Rate({{"sensor", "tsdb.host.error"}, {"type", ToString(error.Type())}})->Inc();

            TStringBuilder msg;
            msg << '[' << tsdbHost << "] " << error.Type() << ": " << error.Message();
            MON_WARN(TsdbReadMany, msg);

            readManyResult->Errors.emplace_back(RequestErrorToDataSourceStatus(error.Type()), std::move(msg));
        }

        for (const auto& [metricKey, errors]: MetricKeyToErrors_) {
            for (auto& error: errors) {
                Registry_.Rate({{"sensor", "tsdb.metric.error"}, {"type", ToString(error.Status)}})->Inc();

                TStringBuilder msg;
                msg << LabelsFromMetricKey(*metricKey) << ": " << error.Message;
                MON_WARN(TsdbReadMany, msg);

                readManyResult->Errors.emplace_back(error.Status, std::move(msg));
            }
        }

        TRACING_SPAN_END(TraceCtx_);
        Handler_->OnSuccess(std::move(readManyResult));
        PassAway();
    }

    void ReportErrorAndPassAway(EDataSourceStatus status, TString error) {
        MON_ERROR(TsdbReadMany, error);
        TRACING_SPAN_END(TraceCtx_);
        Handler_->OnError(std::move(Query_.Project), status, std::move(error));
        PassAway();
    }

private:
    std::shared_ptr<const TYasmGroupToTsdbHosts> GroupToHosts_;
    NActors::TActorId PlacementActor_;
    std::shared_ptr<NSolomon::NTsdb::ITsdbClusterRpc> TsdbCluster_;
    TReadManyQuery Query_;
    std::pair<TInstant, TInstant> QueryTimeRange_;
    IResultHandlerPtr<TReadManyResult> Handler_;
    TDuration Period_;
    size_t MaxInflight_{0};
    absl::flat_hash_map<const TMetricKey<ui32>*, TVector<THistoryResponse>> MetricKeyToResponses_;
    absl::flat_hash_map<const TMetricKey<ui32>*, TVector<TDataSourceError>> MetricKeyToErrors_;
    TVector<TRequestWithUserHost> RequestsWithUserHost_;
    TVector<std::pair<TString, TRequestError>> TsdbHostToError_;
    NMonitoring::TMetricRegistry& Registry_;
    TSpanId TraceCtx_;
    std::unique_ptr<TRequestsManager> RequestManager_;
};

} // namespace

std::unique_ptr<NActors::IActor> ReadManyActor(
        std::shared_ptr<const TYasmGroupToTsdbHosts> groupToHosts,
        NActors::TActorId placementActor,
        std::shared_ptr<NSolomon::NTsdb::ITsdbClusterRpc> tsdbCluster,
        TReadManyQuery query,
        IResultHandlerPtr<TReadManyResult> handler,
        NMonitoring::TMetricRegistry& registry,
        TSpanId traceCtx)
{
    return std::make_unique<TReadManyActor>(
            std::move(groupToHosts),
            placementActor,
            std::move(tsdbCluster),
            std::move(query),
            std::move(handler),
            registry,
            std::move(traceCtx));
}

} // namespace NSolomon::NDataProxy
