#include "read_many_actor.h"
#include "tsdb.h"

#include <solomon/services/dataproxy/config/datasource_config.pb.h>
#include <solomon/services/dataproxy/lib/datasource/datasource.h>
#include <solomon/services/dataproxy/lib/initialization/events.h>
#include <solomon/libs/cpp/trace/trace.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/string/cast.h>

using namespace NActors;
using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

using namespace NSolomon::NTsdb;

class TTsdbSource: public IDataSource {
public:
    TTsdbSource(
            TActorRuntime& runtime,
            TActorId placement,
            std::shared_ptr<const TYasmGroupToTsdbHosts> groupToHosts,
            std::shared_ptr<ITsdbClusterRpc> tsdbCluster,
            NMonitoring::TMetricRegistry& registry)
        : Runtime_{runtime}
        , GroupToHosts_{std::move(groupToHosts)}
        , Placement_{placement}
        , TsdbCluster_{std::move(tsdbCluster)}
        , Registry_{registry}
    {
    }

    bool CanHandle(const TQuery&) const override {
        return false; // because it's only used from the yasm datasource directly
    }

    void Find(TFindQuery query, IResultHandlerPtr<TFindResult> handler, TSpanId traceCtx) const override {
        TRACING_SPAN_END(traceCtx);
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "Find is not supported by TSDB datasource");
    }

    void ResolveOne(TResolveOneQuery query, IResultHandlerPtr<TResolveOneResult> handler, TSpanId traceCtx) const override {
        TRACING_SPAN_END(traceCtx);
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "ResolveOne is not supported by TSDB datasource");
    }

    void ResolveMany(TResolveManyQuery query, IResultHandlerPtr<TResolveManyResult> handler, TSpanId traceCtx) const override {
        TRACING_SPAN_END(traceCtx);
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "ResolveOne is not supported by TSDB datasource");
    }

    void MetricNames(TMetricNamesQuery query, IResultHandlerPtr<TMetricNamesResult> handler, TSpanId traceCtx) const override {
        TRACING_SPAN_END(traceCtx);
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "MetricNames is not supported in TSDB datasource");
    }

    void LabelKeys(TLabelKeysQuery query, IResultHandlerPtr<TLabelKeysResult> handler, TSpanId traceCtx) const override {
        TRACING_SPAN_END(traceCtx);
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "LabelKeys is not supported in TSDB datasource");
    }

    void LabelValues(TLabelValuesQuery query, IResultHandlerPtr<TLabelValuesResult> handler, TSpanId traceCtx) const override {
        TRACING_SPAN_END(traceCtx);
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "LabelValues is not supported in TSDB datasource");
    }

    void UniqueLabels(TUniqueLabelsQuery query, IResultHandlerPtr<TUniqueLabelsResult> handler, TSpanId traceCtx) const override {
        TRACING_SPAN_END(traceCtx);
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "UniqueLabels is not supported in TSDB datasource");
    }

    void ReadOne(TReadOneQuery query, IResultHandlerPtr<TReadOneResult> handler, TSpanId traceCtx) const override {
        TRACING_SPAN_END(traceCtx);
        handler->OnError(std::move(query.Project), EDataSourceStatus::UNKNOWN, "ReadOne is not supported by TSDB datasource");
    }

    void ReadMany(TReadManyQuery query, IResultHandlerPtr<TReadManyResult> handler, TSpanId traceCtx) const override {
        Runtime_.Register(ReadManyActor(
                GroupToHosts_,
                Placement_,
                TsdbCluster_,
                std::move(query),
                std::move(handler),
                Registry_,
                std::move(traceCtx)
        ).release());
    }

    void WaitUntilInitialized() const override {
        auto success = Runtime_.ActorSystem()
                               .Ask<TInitializationEvents::TInitialized>(
                                       Placement_,
                                       MakeHolder<TInitializationEvents::TSubscribe>())
                               .Wait(TDuration::Minutes(1));

        Y_VERIFY(success);
    }

private:
    TActorRuntime& Runtime_;
    std::shared_ptr<const TYasmGroupToTsdbHosts> GroupToHosts_;
    TActorId Placement_;
    TIntrusivePtr<IDataSource> LongTermSource_;
    std::shared_ptr<ITsdbClusterRpc> TsdbCluster_;
    NMonitoring::TMetricRegistry& Registry_;
};

} // namespace

TIntrusivePtr<IDataSource> TsdbSource(
        TActorRuntime& runtime,
        std::shared_ptr<const TYasmGroupToTsdbHosts> groupToHosts,
        std::shared_ptr<NSolomon::NTsdb::ITsdbClusterRpc> tsdbCluster,
        NActors::TActorId placement,
        NMonitoring::TMetricRegistry& registry)
{
    return MakeIntrusive<TTsdbSource>(runtime, placement, std::move(groupToHosts), std::move(tsdbCluster), registry);
}

} // namespace NSolomon::NDataProxy
