#pragma once

#include <solomon/services/dataproxy/lib/timeseries/timeseries.h>
#include <solomon/services/dataproxy/lib/timeseries/type_to_columns.h>

#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/libs/cpp/ts_model/point_type.h>
#include <solomon/libs/cpp/ts_model/visit.h>

#include <infra/yasm/stockpile_client/points.h>
#include <infra/yasm/zoom/components/serialization/history/history.h>

namespace NSolomon::NDataProxy {

constexpr TDuration TSDB_GRID = TDuration::Seconds(5);

class TTsdbTimeSeries final: public ITimeSeries {
    using TVecIt = TVector<NZoom::NValue::TValue>::const_iterator;

private:
    template <typename TPoint>
    class TIter final: public ITimeSeriesIter {
    public:
        TIter(
                const NZoom::NProtobuf::THistoryResponse& tsdbResponse,
                NHistDb::NStockpile::TRecordSerializeState serializationState,
                TDuration period)
            : SerializationState_{std::move(serializationState)}
            , It_{tsdbResponse.Series.GetValues().begin()}
            , End_{tsdbResponse.Series.GetValues().end()}
            , Timestamp_{tsdbResponse.Series.GetStartTimestamp()}
            , Period_{period}
        {
        }

        bool Next(NTs::TVariantPoint* point) override {
            while (true) {
                if (It_ == End_) {
                    return false;
                }
                ProtoPoint_.Clear();
                NHistDb::NStockpile::TValueSerializer serializer(ProtoPoint_, Timestamp_, SerializationState_);

                It_->Update(serializer);
                ++It_;
                Timestamp_ += Period_;

                if (!serializer.Empty()) {
                    break;
                }

                if (serializer.Error()) {
                    // TODO(ivanzhukov): propagate this data to metrics
                    SerializationState_.IncErrorPointsCounter();
                }
            }

            point->Time = TInstant::MilliSeconds(ProtoPoint_.timestampsmillis());
            point->Step = TDuration::MilliSeconds(ProtoPoint_.stepmillis());
            point->Count = ProtoPoint_.count();
            point->Merge = ProtoPoint_.merge();

            if constexpr (TPoint::Type == NTsModel::EPointType::DSummary) {
                const auto& summary = ProtoPoint_.summarydouble();
                NTsModel::TDSummaryPoint sourcePoint;

                sourcePoint.CountValue = summary.count();
                sourcePoint.Sum = summary.sum();
                sourcePoint.Min = summary.min();
                sourcePoint.Max = summary.max();
                sourcePoint.Last = summary.last();

                point->Set(std::move(sourcePoint));
            } else if constexpr (TPoint::Type == NTsModel::EPointType::LogHist) {
                auto& logHist = ProtoPoint_.loghistogram();
                NTsModel::TLogHistPoint sourcePoint;

                sourcePoint.Values.reserve(logHist.buckets_size());
                for (const auto& value: logHist.buckets()) {
                    sourcePoint.Values.emplace_back(value);
                }

                sourcePoint.ZeroCount = logHist.zeroes();
                sourcePoint.StartPower = logHist.start_power();
                sourcePoint.MaxBucketCount = logHist.max_buckets_size();
                sourcePoint.Base = logHist.base();

                point->Set(std::move(sourcePoint));
            } else if constexpr (TPoint::Type == NTsModel::EPointType::Hist) {
                auto& hist = ProtoPoint_.histogram();
                NTsModel::THistPoint sourcePoint;

                sourcePoint.Denom = hist.denom();
                sourcePoint.Buckets.reserve(hist.buckets_size());

                for (int i = 0; i != hist.buckets_size(); ++i) {
                    NTs::NValue::THistogram::TBucket bucket;
                    bucket.UpperBound = hist.bounds(i);
                    bucket.Value = hist.buckets(i);

                    sourcePoint.Buckets.emplace_back(std::move(bucket));
                }

                point->Set(std::move(sourcePoint));
            } else {
                // FIXME(ivanzhukov): check at compile-time
                ythrow yexception() << "unsupported point type: " << TPoint::Type;
            }

            return true;
        }

    private:
        NHistDb::NStockpile::TRecordSerializeState SerializationState_;
        TVecIt It_;
        TVecIt End_;
        TInstant Timestamp_;
        TDuration Period_;
        yandex::solomon::stockpile::TPoint ProtoPoint_;
    };

public:
    TTsdbTimeSeries(
            NZoom::NProtobuf::THistoryResponse tsdbResponse,
            yandex::solomon::model::MetricType protoType,
            NZoom::NAccumulators::EAccumulatorType aggrType,
            TDuration period)
        : TsdbResponse_{std::move(tsdbResponse)}
        , ProtoMetricType_{protoType}
        , AggregationType_{aggrType}
        , Period_{period}
    {
        Type_ = NSolomon::FromProto(ProtoMetricType_);
        auto pointType = NTsModel::FromProto(ProtoMetricType_).Extract();

        NTsModel::Visit(pointType, [&](auto&& pointTraits) {
            using TPoint = typename std::decay_t<decltype(pointTraits)>::TPoint;
            Columns_ = TPoint::SimpleColumns; // XXX(ivanzhukov): are SimpleColumns appropriate ones?
        });

        EnsureValidTypeForColumns(Type_, Columns_);
    }

    NMonitoring::EMetricType Type() const override {
        return Type_;
    }

    NTs::TColumnSet Columns() const override {
        return Columns_;
    }

    ui32 PointCount() const override {
        return TsdbResponse_.Series.GetValues().size();
    }

    std::unique_ptr<ITimeSeriesIter> Iterator() const override {
        auto pointTypeOrErr = NTsModel::FromProto(ProtoMetricType_);
        NHistDb::NStockpile::TSensorId sensorId{
            NHistDb::NStockpile::TShardId{},
            NHistDb::NStockpile::TLocalId{},
            ProtoMetricType_,
        };
        NHistDb::NStockpile::TNumId numId{};

        NHistDb::NStockpile::TRecordSerializeState serializationState{sensorId, numId, AggregationType_};
        if (AggregationType_ == NZoom::NAccumulators::EAccumulatorType::Hgram &&
            sensorId.Type == yandex::solomon::model::MetricType::HIST)
        {
            NHistDb::NStockpile::TRecordUgramMergerVisitor visitor;
            for (const auto& value: TsdbResponse_.Series.GetValues()) {
                visitor.OnValue(value.GetValue());
            }

            serializationState.SetUgramFreezer(visitor.GetFreezer());
        }

        return NTsModel::Visit(pointTypeOrErr.Value(), [&](auto&& pointTraits) -> std::unique_ptr<ITimeSeriesIter> {
            using TPoint = typename std::decay_t<decltype(pointTraits)>::TPoint;

            return std::make_unique<TIter<TPoint>>(TsdbResponse_, std::move(serializationState), Period_);
        });
    }

private:
    NZoom::NProtobuf::THistoryResponse TsdbResponse_;
    yandex::solomon::model::MetricType ProtoMetricType_;
    NZoom::NAccumulators::EAccumulatorType AggregationType_;
    TDuration Period_;
    NMonitoring::EMetricType Type_;
    NTs::TColumnSet Columns_;
};

} // namespace NSolomon::NDataProxy
