#pragma once

#include <solomon/libs/cpp/clients/tsdb/stub.h>

#include <infra/yasm/zoom/components/serialization/history/history.h>

#include <library/cpp/grpc/client/grpc_client_low.h>
#include <library/cpp/threading/future/future.h>

#include <util/thread/pool.h>

namespace NSolomon::NDataProxy {

struct TFakeTsdbNode: public NTsdb::TTsdbRpcStub {
    using TResultType = TErrorOr<NYasm::NInterfaces::NInternal::THistoryReadAggregatedResponse, NSolomon::TRequestError>;

public:
    TFakeTsdbNode(TString address, NYasm::NInterfaces::NInternal::THistoryReadAggregatedResponse response, TDuration sleepFor = TDuration::Seconds(0))
        : Address{std::move(address)}
        , Result_{TResultType::FromValue(std::move(response))}
        , SleepFor_{sleepFor}
        , Worker_(new TThreadPool(TThreadPool::TParams().SetBlocking(true).SetCatching(false)))
    {
        Worker_->Start(2);
    }

    TFakeTsdbNode(TString address, NSolomon::TRequestError error, TDuration sleepFor = TDuration::Seconds(0))
        : Address{std::move(address)}
        , Result_{TResultType::FromError(std::move(error))}
        , SleepFor_{sleepFor}
        , Worker_(new TThreadPool(TThreadPool::TParams().SetBlocking(true).SetCatching(false)))
    {
        Worker_->Start(2);
    }

    ~TFakeTsdbNode() override {
        Terminated_.store(true);
        Delete(std::move(Worker_));
        Y_VERIFY(Called, "ReadAggregated has never been called");
    }

    NTsdb::TReadAggregatedAsyncResponse ReadAggregated(
            const NYasm::NInterfaces::NInternal::THistoryReadAggregatedRequest& request) override
    {
        Y_ENSURE(!Called, "Second request for the same node");
        Called = true;
        NThreading::TPromise<TResultType> promise = NThreading::NewPromise<TResultType>();

        if (SleepFor_) {
            Worker_->SafeAddFunc([this, request, promise]() mutable {
                TDuration current = TDuration::MilliSeconds(0), step = TDuration::MilliSeconds(10);
                while (current < SleepFor_) {
                    if (Terminated_.load()) {
                        break;
                    }
                    current += step;
                    Sleep(step);
                }
                promise.SetValue(std::move(Result_));
                Requests = NZoom::NProtobuf::THistoryRequest::FromProto(request);
            });
        } else {
            promise.SetValue(std::move(Result_));
            Requests = NZoom::NProtobuf::THistoryRequest::FromProto(request);
        }

        return promise.GetFuture();
    }

public:
    TString Address;
    TVector<NZoom::NProtobuf::THistoryRequest> Requests;
    bool Called = false;

private:
    TResultType Result_;
    TDuration SleepFor_;
    THolder<IThreadPool> Worker_;
    std::atomic_bool Terminated_{false};
};

inline std::shared_ptr<NTsdb::ITsdbClusterRpc> MakeFakeRpc(const std::vector<std::shared_ptr<TFakeTsdbNode>>& nodes) {
    THashMap<TString, std::shared_ptr<NTsdb::ITsdbRpc>> nodeRpc;
    for (auto node: nodes) {
        nodeRpc.emplace(node->Address, node);
    }
    return std::make_shared<NTsdb::TTsdbClusterRpcStub>(std::move(nodeRpc));
}

} // namespace NSolomon::NDataProxy
