#include "fake_node.h"

#include <solomon/services/dataproxy/lib/datasource/tsdb/placement.h>
#include <solomon/services/dataproxy/lib/datasource/tsdb/read_many_actor.h>
#include <solomon/services/dataproxy/lib/datasource/ut_utils.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>
#include <solomon/libs/cpp/clients/tsdb/stub.h>

#include <infra/yasm/common/points/value/impl.h>
#include <infra/yasm/zoom/components/serialization/history/history.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NActors;
using namespace NSolomon::NDataProxy;
using namespace NSolomon;
using namespace ::NYasm::NInterfaces::NInternal;
using namespace NZoom::NProtobuf;
using namespace NZoom::NValue;
using namespace testing;

class TReadManyActorTest: public Test {
protected:
    void SetUp() override {
        Registry_ = std::make_shared<NMonitoring::TMetricRegistry>();

        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, true);
        ActorRuntime_->WaitForBootstrap();

        ActorRuntime_->SetLogPriority(ELogComponent::TsdbReadMany, NActors::NLog::PRI_TRACE);

        EdgeId_ = ActorRuntime_->AllocateEdgeActor();
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

    /**
     *  Dispatch `count` events other than logger events
     */
    void DispatchEvents(size_t count) {
        NActors::TDispatchOptions options;
        options.FinalEvents.emplace_back(
                [](IEventHandle& ev) {
                    return ev.GetTypeRewrite() != NActors::NLog::TEvLog::EventType;
                },
                count);

        ActorRuntime_->DispatchEvents(options);
    }

    template <typename TEvent>
    void Send(TActorId actorId, THolder<TEvent> event) {
        ActorRuntime_->Send(actorId, EdgeId_, std::move(event));
    }

    template <typename TEvent>
    typename TEvent::TPtr Receive() {
        return ActorRuntime_->GrabEdgeEvent<TEvent>(EdgeId_, TDuration::Seconds(3));
    }

protected:
    THolder<TTestActorRuntime> ActorRuntime_;
    TActorId EdgeId_;
    std::shared_ptr<NMonitoring::TMetricRegistry> Registry_;
};

TReadManyQuery MakeBaseQuery() {
    TReadManyQuery query;

    query.Project = "yasm_project";
    query.Time = {TInstant::Seconds(100), TInstant::Seconds(120)};
    query.Deadline = TInstant::Now() + TDuration::Seconds(15);
    query.ResolvedKeys = std::make_unique<TReadManyQuery::TResolvedKeys>();

    query.MaxTimeSeriesFormat = 39;

    auto downsampling = query.Operations.emplace_back().mutable_downsampling();
    downsampling->set_grid_millis(TDuration::Seconds(5).MilliSeconds());

    return query;
}

TReadManyQuery MakeHostQuery(TStringBuf host) {
    auto query = MakeBaseQuery();

    NStringPool::TStringPoolBuilder builder;
    auto& metric = query.ResolvedKeys->MetricKeys.emplace_back();
    metric.Name = builder.Put("");
    metric.Labels.emplace_back(builder.Put("signal"), builder.Put("signal"));
    metric.Labels.emplace_back(builder.Put("host"), builder.Put(host));

    query.ResolvedKeys->CommonLabels.emplace_back(builder.Put("service"), builder.Put("service"));
    query.ResolvedKeys->CommonLabels.emplace_back(builder.Put("cluster"), builder.Put("cluster"));

    query.ResolvedKeys->Strings = builder.Build();
    return query;
}

TReadManyQuery MakeGroupQuery(TStringBuf group) {
    auto query = MakeBaseQuery();

    NStringPool::TStringPoolBuilder builder;
    auto& metric = query.ResolvedKeys->MetricKeys.emplace_back();
    metric.Name = builder.Put("");
    metric.Labels.emplace_back(builder.Put("signal"), builder.Put("counter-instance_tmmv"));
    metric.Labels.emplace_back(builder.Put("group"), builder.Put(group));

    query.ResolvedKeys->CommonLabels.emplace_back(builder.Put("service"), builder.Put("service"));
    query.ResolvedKeys->CommonLabels.emplace_back(builder.Put("cluster"), builder.Put("cluster"));

    query.ResolvedKeys->Strings = builder.Build();
    return query;
}

TReadManyResult MakeGroupResult(TStringBuf group) {
    TReadManyResult result;

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<double>(
            {{{
                      {"project", "yasm_project"},
                      {"service", "serivce"},
                      {"cluster", "cluster"},
                      {"signal", "counter-instance_tmmv"},
                      {"group", TString{group}},
              },
              {0.0, 1.0, 2.0, 3.0},
              NMonitoring::EMetricType::DSUMMARY}},
            NZoom::NAccumulators::EAccumulatorType::Summ);

    result.Strings = std::move(readManyStrPool);
    result.Metrics = std::move(readManyMetrics);

    return result;
}

TReadManyResult MakeHostResult(TStringBuf host) {
    TReadManyResult result;

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<double>(
            {{{
                      {"project", "yasm_project"},
                      {"service", "serivce"},
                      {"cluster", "cluster"},
                      {"signal", "counter-instance_tmmv"},
                      {"host", TString{host}},
              },
              {0.0, 1.0, 2.0, 3.0},
              NMonitoring::EMetricType::DSUMMARY}},
            NZoom::NAccumulators::EAccumulatorType::Summ);

    result.Strings = std::move(readManyStrPool);
    result.Metrics = std::move(readManyMetrics);

    return result;
}

THistoryReadAggregatedResponse MakeTsdbResponse(TStringBuf hostName, THistoryAggregatedSeries::EStatusCode status) {
    THistoryRequest request{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(120),
            .Period = TDuration::Seconds(5),
            .HostName = hostName,
            .RequestKey = TStringBuf{"itype=project"},
            .SignalName = TStringBuf{"counter-instance_tmmv"}};

    TVector<TValue> values;

    for (int i = 0; i < 4; i++) {
        values.emplace_back(NZoom::NValue::TFloatValue{static_cast<double>(i)});
    }

    THistoryResponse tsdbResponse{request, TInstant::Seconds(100), std::move(values), status};

    THistoryReadAggregatedResponse responseProto;
    THistoryResponseWriter writer{responseProto};
    writer.Add(tsdbResponse);

    return responseProto;
}

std::shared_ptr<const TYasmGroupToTsdbHosts> MakeGroupToHosts() {
    return std::make_shared<const TYasmGroupToTsdbHosts>(
            TYasmGroupToTsdbHosts{{"group_1", {"host_1", "host_2"}}, {"group_2", {"host_3", "host_4"}}});
}

TEST_F(TReadManyActorTest, ReadGroupMetrics) {
    auto nodeOk1= std::make_shared<TFakeTsdbNode>(
            "host_1",
            MakeTsdbResponse("group_1", THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK));
    auto nodeOk2 = std::make_shared<TFakeTsdbNode>(
            "host_2",
            MakeTsdbResponse("group_1", THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK));

    auto rpc = MakeFakeRpc({nodeOk1, nodeOk2});

    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    // TODO: check the content
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(
            MakeGroupToHosts(),
            EdgeId_,
            rpc,
            MakeGroupQuery("group_1"),
            handlerMock,
            *Registry_,
            NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TReadManyActorTest, ReadHostMetrics) {
    auto nodeOk1 = std::make_shared<TFakeTsdbNode>(
            "host_1",
            MakeTsdbResponse("group_1", THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK));
    auto nodeOk2 = std::make_shared<TFakeTsdbNode>(
            "host_2",
            MakeTsdbResponse("group_1", THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK));

    auto rpc = MakeFakeRpc({nodeOk1, nodeOk2});

    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();


    // TODO: check the content
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(
            MakeGroupToHosts(),
            EdgeId_,
            rpc,
            MakeHostQuery("user_host"),
            handlerMock,
            *Registry_,
            NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    auto event = Receive<TPlacementActorEvents::TResolveHosts>();
    ASSERT_EQ(event->Get()->UserHosts.size(), 1ul);
    ASSERT_EQ(event->Get()->UserHosts[0], "user_host");

    Send(actorId,
         MakeHolder<TPlacementActorEvents::TResolveHostsResponse>(
                 TStringMap<absl::flat_hash_set<TString>>{{"user_host", absl::flat_hash_set<TString>{"group_1"}}}));

    // process responses
    DispatchEvents(2);

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TReadManyActorTest, ReadQueriesWithDifferentTimeRange) {
    auto testRange = [&](const TString& testName, TInstant from, TInstant to, TInstant expectedFrom, TInstant expectedTo) {
        SCOPED_TRACE(testName);

        auto nodeOk1 = std::make_shared<TFakeTsdbNode>(
                "host_1",
                MakeTsdbResponse("group_1", THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK));
        auto nodeOk2 = std::make_shared<TFakeTsdbNode>(
                "host_2",
                MakeTsdbResponse("group_1", THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK));
        auto rpc = MakeFakeRpc({nodeOk1, nodeOk2});
        auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

        auto query = MakeGroupQuery("group_1");
        query.Time.From = from;
        query.Time.To = to;

        auto actor = ReadManyActor(
                MakeGroupToHosts(),
                EdgeId_,
                rpc,
                std::move(query),
                handlerMock,
                *Registry_,
                NTracing::TSpanId{});

        ActorRuntime_->Register(actor.release());
        ActorRuntime_->WaitForBootstrap();

        ASSERT_EQ(nodeOk1->Requests.size(), 1ul);
        ASSERT_EQ(nodeOk2->Requests.size(), 1ul);

        const auto& request = nodeOk1->Requests[0];
        ASSERT_EQ(request, nodeOk2->Requests[0]);

        ASSERT_EQ(request.Start, expectedFrom);
        ASSERT_EQ(request.End, expectedTo);
    };

    {
        ActorRuntime_->AdvanceCurrentTime(TDuration::Hours(2));
        auto now = ActorRuntime_->GetCurrentTime();
        testRange("1", now - TDuration::Hours(1), now, now - TDuration::Minutes(25), now);
    }

    {
        auto now = ActorRuntime_->GetCurrentTime();
        testRange("2", now - TDuration::Hours(1), now + TDuration::Hours(1), now - TDuration::Minutes(25), now);
    }

    {
        auto now = ActorRuntime_->GetCurrentTime();
        testRange("3", now - TDuration::Minutes(2), now - TDuration::Minutes(1), now - TDuration::Minutes(2), now - TDuration::Minutes(1));
    }

    {
        ActorRuntime_->AdvanceCurrentTime(TDuration::MilliSeconds(3000)); // not aligned by a grid
        auto now = ActorRuntime_->GetCurrentTime();
        auto from = now - TDuration::Minutes(2);
        auto to = now;
        testRange("4", from, to, from, TInstant::Seconds(to.Seconds() - to.Seconds() % 5));
    }
}

TEST_F(TReadManyActorTest, UnknownGroup) {
    auto rpc = MakeFakeRpc({});

    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    // expect an empty response
    // TODO: check the content
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(
            MakeGroupToHosts(),
            EdgeId_,
            rpc,
            MakeGroupQuery("unknown_group"),
            handlerMock,
            *Registry_,
            NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TReadManyActorTest, UnknownUserHost) {
    auto rpc = MakeFakeRpc({});

    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    // expect an empty response
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(
            MakeGroupToHosts(),
            EdgeId_,
            rpc,
            MakeHostQuery("unknown_user_host"),
            handlerMock,
            *Registry_,
            NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    auto event = Receive<TPlacementActorEvents::TResolveHosts>();
    ASSERT_EQ(event->Get()->UserHosts.size(), 1ul);
    ASSERT_EQ(event->Get()->UserHosts[0], "unknown_user_host");

    Send(actorId, MakeHolder<TPlacementActorEvents::TResolveHostsResponse>(TStringMap<absl::flat_hash_set<TString>>{}));

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TReadManyActorTest, ErrorFromAllNodes) {
    auto nodeErr1 = std::make_shared<TFakeTsdbNode>("host_1", NSolomon::TRequestError{NSolomon::TRequestError::EType::Unknown, "Error"});
    auto nodeErr2 = std::make_shared<TFakeTsdbNode>("host_2", NSolomon::TRequestError{NSolomon::TRequestError::EType::Unknown, "Error"});

    auto rpc = MakeFakeRpc({nodeErr1, nodeErr2});

    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    // expect an error
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(1);
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(0);

    auto actor = ReadManyActor(
            MakeGroupToHosts(),
            EdgeId_,
            rpc,
            MakeGroupQuery("group_1"),
            handlerMock,
            *Registry_,
            NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}


TEST_F(TReadManyActorTest, ErrorFromOneNode) {
    auto nodeErr1 = std::make_shared<TFakeTsdbNode>("host_1", NSolomon::TRequestError{NSolomon::TRequestError::EType::Unknown, "Error"});
    auto nodeOk2 = std::make_shared<TFakeTsdbNode>(
            "host_2",
            MakeTsdbResponse("group_1", THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK));

    auto rpc = MakeFakeRpc({nodeErr1, nodeOk2});

    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    // TODO: check the content
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(
            MakeGroupToHosts(),
            EdgeId_,
            rpc,
            MakeGroupQuery("group_1"),
            handlerMock,
            *Registry_,
            NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TReadManyActorTest, DontWaitForSecondHost) {
    auto nodeOk1 = std::make_shared<TFakeTsdbNode>("host_1",MakeTsdbResponse("group_1", THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK));
    auto nodeOk2 = std::make_shared<TFakeTsdbNode>(
            "host_2",
            MakeTsdbResponse("group_1", THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK),
            TDuration::Seconds(10));

    auto rpc = MakeFakeRpc({nodeOk1, nodeOk2});

    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    // TODO: check the content
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(
            MakeGroupToHosts(),
            EdgeId_,
            rpc,
            MakeGroupQuery("group_1"),
            handlerMock,
            *Registry_,
            NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    ActorRuntime_->AdvanceCurrentTime(TDuration::Seconds(100));
    DispatchEvents(1);

    ASSERT_EQ(nodeOk1->Requests.size(), 1ul);
    ASSERT_EQ(nodeOk2->Requests.size(), 0ul);

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}
