#include <solomon/services/dataproxy/lib/datasource/tsdb/tsdb_timeseries.h>
#include <infra/yasm/common/points/value/impl.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using namespace NZoom::NProtobuf;
using namespace NYasm::NInterfaces::NInternal;
using namespace NZoom::NValue;

TTsdbTimeSeries MakeTimeSeries(THistoryResponse&& response, TDuration period) {
    auto aggrType = NHistDb::NStockpile::GetAggregationType(response.Series.GetName());
    NHistDb::NStockpile::TValueTypeDetector typeDetector{aggrType};
    for (auto&& point: response.Series.GetValues()) {
        typeDetector.OnValue(point.GetValue());
    }
    auto protoType = typeDetector.GetType();
    auto pointTypeOrErr = NTsModel::FromProto(protoType);
    if (pointTypeOrErr.Fail()) {
        throw std::invalid_argument("Empty series");
    }
    return TTsdbTimeSeries{std::move(response), protoType, aggrType, period};
}

TEST(TsdbTimeSeries, DontSkipEmptyPoints) {
    THistoryRequest request{
        .Start = TInstant::Seconds(100),
        .End = TInstant::Seconds(200),
        .Period = TDuration::Seconds(5),
        .HostName = TStringBuf("sas1-1234"),
        .RequestKey = TStringBuf("itype=testing"),
        .SignalName = TStringBuf("disk_usage_summ")
    };

    TVector<TValue> values;

    for (int i = 0; i < 20; i++) {
        if (i % 5 == 4) {
            values.emplace_back(NZoom::NHgram::THgram::Small({42}, 0));
        } else {
            values.emplace_back(NZoom::NHgram::THgram::EmptyUgram());
        }
    }

    THistoryResponse tsdbResponse(
        request, TInstant::Seconds(100), std::move(values), THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
    );

    auto series = MakeTimeSeries(std::move(tsdbResponse), TDuration::Seconds(5));

    auto it = series.Iterator();
    NTs::TVariantPoint point;
    int count = 0;
    while (it->Next(&point)) {
        count++;
    }
    EXPECT_EQ(4, count);
}

TEST(TsdbTimeSeries, OnlyEmptyPoints) {
    THistoryRequest request{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(200),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
    };

    TVector<TValue> values;

    for (int i = 0; i < 20; i++) {
        values.emplace_back(NZoom::NHgram::THgram::EmptyUgram());
    }

    THistoryResponse tsdbResponse(
            request, TInstant::Seconds(100), std::move(values), THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
    );

    auto series = MakeTimeSeries(std::move(tsdbResponse), TDuration::Seconds(5));

    auto it = series.Iterator();
    NTs::TVariantPoint point;
    int count = 0;
    while (it->Next(&point)) {
        count++;
    }
    EXPECT_EQ(0, count);
}

TEST(TsdbTimeSeries, OnlyNotEmptyPoints) {
    THistoryRequest request{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(200),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
    };

    TVector<TValue> values;

    for (int i = 0; i < 20; i++) {
        values.emplace_back(NZoom::NHgram::THgram::Small({42}, 0));
    }

    THistoryResponse tsdbResponse(
            request, TInstant::Seconds(100), std::move(values), THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
    );

    auto series = MakeTimeSeries(std::move(tsdbResponse), TDuration::Seconds(5));

    auto it = series.Iterator();
    NTs::TVariantPoint point;
    int count = 0;
    while (it->Next(&point)) {
        count++;
    }
    EXPECT_EQ(20, count);
}

TEST(TsdbTimeSeries, EmptyLastPoint) {
    THistoryRequest request{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(200),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
    };

    TVector<TValue> values;

    for (int i = 0; i < 19; i++) {
        values.emplace_back(NZoom::NHgram::THgram::Small({42}, 0));
    }
    values.emplace_back(NZoom::NHgram::THgram::EmptyUgram());

    THistoryResponse tsdbResponse(
            request, TInstant::Seconds(100), std::move(values), THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
    );

    auto series = MakeTimeSeries(std::move(tsdbResponse), TDuration::Seconds(5));

    auto it = series.Iterator();
    NTs::TVariantPoint point;
    int count = 0;
    while (it->Next(&point)) {
        count++;
    }
    EXPECT_EQ(19, count);
}

TEST(TsdbTimeSeries, OnlyNonePoints) {
    THistoryRequest request{
            .Start = TInstant::Seconds(100),
            .End = TInstant::Seconds(200),
            .Period = TDuration::Seconds(5),
            .HostName = TStringBuf("sas1-1234"),
            .RequestKey = TStringBuf("itype=testing"),
            .SignalName = TStringBuf("disk_usage_summ")
    };

    TVector<TValue> values;

    for (int i = 0; i < 20; i++) {
        values.emplace_back(NZoom::NValue::TOptionalFloatValue());
    }

    THistoryResponse tsdbResponse(
            request, TInstant::Seconds(100), std::move(values), THistoryAggregatedSeries::EStatusCode::THistoryAggregatedSeries_EStatusCode_OK
    );

    EXPECT_THROW(MakeTimeSeries(std::move(tsdbResponse), TDuration::Seconds(5)), std::invalid_argument);
}
