#pragma once

#include "datasource.h"
#include "result_handler.h"
#include "query.h"

#include <solomon/services/dataproxy/lib/timeseries/compressed.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/serializer.h>

#include <solomon/libs/cpp/yasm/constants/labels.h>
#include <library/cpp/testing/gtest/gtest.h>

namespace NSolomon::NDataProxy {

using TLabelsList = TVector<std::pair<TString, TString>>;
using TMetricsData = TVector<TReadManyResult::TMetricData>;

class TReadManyResultHandlerMock: public IResultHandler<TReadManyResult> {
public:
    MOCK_METHOD(void, OnSuccess, (std::unique_ptr<TReadManyResult>), (override));
    MOCK_METHOD(void, OnError, (TString&&, EDataSourceStatus, TString&&), (override));
};

class TBaseDataSourceMock: public IDataSource {
public:
    MOCK_METHOD(bool, CanHandle, (const TQuery&), (const, override));
    MOCK_METHOD(void, Find, (TFindQuery, IResultHandlerPtr<TFindResult>, NTracing::TSpanId), (const, override));
    MOCK_METHOD(void, ResolveOne, (TResolveOneQuery, IResultHandlerPtr<TResolveOneResult>, NTracing::TSpanId), (const, override));
    MOCK_METHOD(void, ResolveMany, (TResolveManyQuery, IResultHandlerPtr<TResolveManyResult>, NTracing::TSpanId), (const, override));
    MOCK_METHOD(void, MetricNames, (TMetricNamesQuery, IResultHandlerPtr<TMetricNamesResult>, NTracing::TSpanId), (const, override));
    MOCK_METHOD(void, LabelKeys, (TLabelKeysQuery, IResultHandlerPtr<TLabelKeysResult>, NTracing::TSpanId), (const, override));
    MOCK_METHOD(void, LabelValues, (TLabelValuesQuery, IResultHandlerPtr<TLabelValuesResult>, NTracing::TSpanId), (const, override));
    MOCK_METHOD(void, UniqueLabels, (TUniqueLabelsQuery, IResultHandlerPtr<TUniqueLabelsResult>, NTracing::TSpanId), (const, override));
    MOCK_METHOD(void, ReadOne, (TReadOneQuery, IResultHandlerPtr<TReadOneResult>, NTracing::TSpanId), (const, override));
    MOCK_METHOD(void, ReadMany, (TReadManyQuery, IResultHandlerPtr<TReadManyResult>, NTracing::TSpanId), (const, override));
    MOCK_METHOD(void, WaitUntilInitialized, (), (const, override));
};

MATCHER_P(ReadManyResultEq, readManyResult, "") {
    auto resultStrings = readManyResult->Strings.Copy().Build();
    auto argStrings = arg->Strings.Copy().Build();

    if (readManyResult->Metrics.size() != arg->Metrics.size()) {
        *result_listener << "expected metrics size != actual metrics size: " << readManyResult->Metrics.size()
                         << " != " << arg->Metrics.size();
        return false;
    }

    auto printExpectedLabels = [&](const TReadManyResult::TMetricData& metric) {
        auto resultLabels = metric.Meta.Labels;
        TStringBuilder sb;
        sb << "{ ";
        for (auto [key, value]: resultLabels) {
            sb << resultStrings[key] << "=\"" << resultStrings[value] << "\" ";
        }
        sb << "}";
        return sb;
    };

    auto printActualLabels = [&](const TReadManyResult::TMetricData& metric) {
        auto argLabels = metric.Meta.Labels;
        TStringBuilder sb;
        sb << "{ ";
        for (auto [key, value]: argLabels) {
            sb << argStrings[key] << "=\"" << argStrings[value] << "\" ";
        }
        sb << "}";
        return sb;
    };

    std::sort(
            readManyResult->Metrics.begin(),
            readManyResult->Metrics.end(),
            [&](const TReadManyResult::TMetricData& lhs, const TReadManyResult::TMetricData& rhs) {
                return printExpectedLabels(lhs) < printExpectedLabels(rhs);
            });

    std::sort(
            arg->Metrics.begin(),
            arg->Metrics.end(),
            [&](const TReadManyResult::TMetricData& lhs, const TReadManyResult::TMetricData& rhs) {
                return printActualLabels(lhs) < printActualLabels(rhs);
            });

    for (auto i: xrange(readManyResult->Metrics.size())) {
        auto& resultMetric = readManyResult->Metrics[i];
        auto& argMetric = arg->Metrics[i];
        auto& resultMeta = resultMetric.Meta;
        auto& argMeta = argMetric.Meta;
        auto& resultLabels = resultMeta.Labels;
        auto& argLabels = argMeta.Labels;

        if (resultLabels.size() != argLabels.size()) {
            *result_listener << "expected metric[" << i << "].Labels.size() != actual metric[" << i
                            << "].Labels.size(): " << resultLabels.size() << " != " << argLabels.size();

            return false;
        }

        std::sort(argLabels.begin(), argLabels.end(), [&](const auto& lhs, const auto& rhs) {
            return std::pair(argStrings[lhs.Key], argStrings[lhs.Value]) <
                    std::pair(argStrings[rhs.Key], argStrings[rhs.Value]);
        });

        std::sort(resultLabels.begin(), resultLabels.end(), [&](const auto& lhs, const auto& rhs) {
            return std::pair(resultStrings[lhs.Key], resultStrings[lhs.Value]) <
                    std::pair(resultStrings[rhs.Key], resultStrings[rhs.Value]);
        });

        for (auto j: xrange(resultMeta.Labels.size())) {
            auto [resKey, resValue] = resultMeta.Labels[j];
            auto [argKey, argValue] = argMeta.Labels[j];
            if (argStrings[argKey] != resultStrings[resKey] || argStrings[argValue] != resultStrings[resValue]) {
                *result_listener << "expected labels != actual labels: " << printExpectedLabels(resultMetric) << " != " << printActualLabels(argMetric);
                return false;
            }
        }

        if (argStrings[argMeta.Name] != resultStrings[resultMeta.Name]) {
            *result_listener << "expected metric name != actual metric name: \"" << resultStrings[resultMeta.Name]
                             << "\" != \"" << argStrings[argMeta.Name] << '"';
            return false;
        }

        if (argMeta.Type != resultMeta.Type) {
            *result_listener << "expected metric type != actual metric type";
            return false;
        }

        // TODO: replace this wired code with the good one, when time series matchers will be ready

        auto* m1 = dynamic_cast<TCompressedTimeSeries*>(readManyResult->Metrics[i].TimeSeries.get());
        Y_ENSURE(m1);

        auto* m2 = dynamic_cast<TCompressedTimeSeries*>(arg->Metrics[i].TimeSeries.get());
        Y_ENSURE(m2);

        if (m1->PointCount() != m2->PointCount()) {
            *result_listener << "expected point count != actual point count: " << m1->PointCount() << " != " << m2->PointCount();
            return false;
        }

        if (m1->PointCount() == 0) {
            continue;
        }

        if (m1->Columns() != m2->Columns() || m1->Type() != m2->Type() || m1->Buffer() != m2->Buffer()) {
            *result_listener << "expected timeseries != actual timeseries"
                << ": columns: " << (m1->Columns() == m2->Columns() ? "equal" : "not equal")
                << "; type: " << MetricTypeToStr(m1->Type()) << " vs " << MetricTypeToStr(m2->Type())
                << "; buffer: " << m1->Buffer() << " vs " << m2->Buffer()
                ;

            return false;
        }
    }

    return true;
}

inline std::pair<TVector<TMetric<ui32>>, NStringPool::TStringPoolBuilder> MakeMetricsFromLabels(
    const TVector<std::pair<TLabelsList, NMonitoring::EMetricType>>& labels)
{
    std::pair<TVector<TMetric<ui32>>, NStringPool::TStringPoolBuilder> result;
    result.first.reserve(labels.size());
    NStringPool::TStringPoolBuilder resultPoolBuilder;
    for (auto& [metricLabels, metricType]: labels) {
        result.first.emplace_back();
        auto& curMetric = result.first.back();
        curMetric.Name = resultPoolBuilder.Put("");
        for (const auto& [labelKey, labelValue]: metricLabels) {
            curMetric.Labels.emplace_back(resultPoolBuilder.Put(labelKey), resultPoolBuilder.Put(labelValue));
        }
        curMetric.Type = metricType;
    }
    result.second = std::move(resultPoolBuilder);
    return result;
}

template <typename TValue, bool isAggregatedResult = false>
inline std::pair<TMetricsData, NStringPool::TStringPoolBuilder> MakeMetricsWithTimeseries(
        TVector<std::tuple<TLabelsList, TVector<TValue>, NMonitoring::EMetricType>>&& labels,
        NZoom::NAccumulators::EAccumulatorType accumulatorType,
        TInstant start = TInstant::Zero() + NYasm::YASM_INTERVAL,
        TDuration step = NYasm::YASM_INTERVAL)
{
    TMetricsData metricsData;
    NStringPool::TStringPoolBuilder strings;

    for (auto& [metricLabels, metricValues, metricType]: labels) {
        metricsData.emplace_back();

        TMetric<ui32>& curMetric = metricsData.back().Meta;
        curMetric.Type = metricType;

        curMetric.Name = strings.Put("");
        for (const auto& [labelKey, labelValue]: metricLabels) {
            if constexpr (!isAggregatedResult) {
                if (labelKey == NYasm::LABEL_SIGNAL) {
                    curMetric.Name = strings.Put(labelValue);
                    continue;
                }
            }
            curMetric.Labels.emplace_back(strings.Put(labelKey), strings.Put(labelValue));
        }

        ui32 pointCount = metricValues.size();
        NZoom::NAccumulators::TCompactAccumulatorsArray accumulators(
                accumulatorType,
                pointCount,
                NZoom::NAccumulators::ENullPolicy::Nullable);
        for (auto offset: xrange(pointCount)) {
            accumulators.Mul(NZoom::NValue::TValue(std::move(metricValues[offset])), offset);
        }

        TMetricSerializer serializer(curMetric.Type, accumulators, start, step);

        metricsData.back().TimeSeries = std::make_unique<TCompressedTimeSeries>(
                curMetric.Type,
                TypeToAggrColumns(curMetric.Type),
                pointCount,
                serializer.Serialize());
    }
    return {std::move(metricsData), std::move(strings)};
}

} // namespace NSolomon::NDataProxy
