#include "datasource_actor.h"
#include "events.h"
#include "labels.h"
#include "selectors_collapsing_actor.h"
#include "yasm_common.h"
#include "groups_hosts_resolver.h"

#include <solomon/services/dataproxy/lib/datasource/datasource.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/metric_aggregator.h>
#include <solomon/libs/cpp/trace/trace.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/yasm/constants/labels.h>

#include <infra/yasm/common/groups/metagroup_groups.h>

#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>

using namespace NActors;
using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

class TLabelKeysHandler: public IResultHandler<TLabelKeysResult> {
public:
    explicit TLabelKeysHandler(IResultHandlerPtr<TLabelKeysResult> handler)
        : OriginalHandler_{std::move(handler)}
    {
    }

private:
    void OnSuccess(std::unique_ptr<TLabelKeysResult> res) override {
        auto idx = res->Strings.Put(NLabels::LABEL_HOSTS);

        if (res->Keys.empty() || idx > res->Keys.back()) {
            res->Keys.emplace_back(idx);
        }

        OriginalHandler_->OnSuccess(std::move(res));
    }

    void OnError(TString&& project, EDataSourceStatus status, TString&& message) override {
        OriginalHandler_->OnError(std::move(project), status, std::move(message));
    }

private:
    IResultHandlerPtr<TLabelKeysResult> OriginalHandler_;
};

class TLabelValuesHandler: public IResultHandler<TLabelValuesResult> {
public:
    TLabelValuesHandler(
            IResultHandlerPtr<TLabelValuesResult> handler,
            const THashSet<TString>& metagroups,
            const THashSet<TString>& hosts)
        : OriginalHandler_{std::move(handler)}
        , Metagroups_{metagroups}
        , Hosts_{hosts}
    {
    }

    void OnSuccess(std::unique_ptr<TLabelValuesResult> res) override {
        auto& resState = res->Labels.emplace_back();
        resState.MetricCount = 0;
        resState.Truncated = false;
        resState.Key = res->Strings.Put(NLabels::LABEL_HOSTS);
        resState.Values.reserve(Metagroups_.size());

        for (const auto& metagroup: Metagroups_) {
            resState.Values.emplace_back(res->Strings.Put(metagroup));
        }

        for (const auto& host: Hosts_) {
            resState.Values.emplace_back(res->Strings.Put(host));
        }

        OriginalHandler_->OnSuccess(std::move(res));
    }

    void OnError(TString&& project, EDataSourceStatus status, TString&& message) override {
        OriginalHandler_->OnError(std::move(project), status, std::move(message));
    }

private:
    IResultHandlerPtr<TLabelValuesResult> OriginalHandler_;
    const THashSet<TString>& Metagroups_;
    const THashSet<TString>& Hosts_;
};

class TYasmDataSourceActor: public TActor<TYasmDataSourceActor> {
public:
    TYasmDataSourceActor(
            TYasmConfig yasmConfig,
            IDataSourcePtr yasmDataSource,
            NMonitoring::TMetricRegistry& registry)
        : TActor(&TYasmDataSourceActor::Normal)
        , Resolver_(std::move(yasmConfig.MetagroupToGroups), yasmDataSource)
        , SelectorsTransformer_(std::move(yasmConfig.ShardConfig))
        , YasmDataSource_{std::move(yasmDataSource)}
        , Registry_{registry}
    {
    }

    STATEFN(Normal) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TYasmDataSourceEvents::TReadManyRequest, OnRequest<TYasmDataSourceEvents::TReadManyRequest>)
            hFunc(TYasmDataSourceEvents::TFindRequest, OnRequest<TYasmDataSourceEvents::TFindRequest>)
            hFunc(TYasmDataSourceEvents::TMetricNamesRequest, OnRequest<TYasmDataSourceEvents::TMetricNamesRequest>)
            hFunc(TYasmDataSourceEvents::TLabelKeysRequest, OnRequest<TYasmDataSourceEvents::TLabelKeysRequest>)
            hFunc(TYasmDataSourceEvents::TLabelValuesRequest, OnRequest<TYasmDataSourceEvents::TLabelValuesRequest>)
            hFunc(TYasmDataSourceEvents::TUniqueLabelsRequest, OnRequest<TYasmDataSourceEvents::TUniqueLabelsRequest>)

            hFunc(TDataSourceEvents::TSuccess<TLabelValuesResult>, OnProjectGroupsSuccess)
            hFunc(TDataSourceEvents::TError, OnProjectGroupsError)

            hFunc(TEvents::TEvPoison, OnPoison)
        }
    }

    template <typename TRequestType>
    void OnRequest(typename TRequestType::TPtr& ev) {
        auto& query = ev->Get()->Query;
        auto& handler = ev->Get()->Handler;

        THostsLabel hosts{"", THostsLabel::EType::UNKNOWN};
        auto isMetagroup = [this](TStringBuf hosts) {
            return Resolver_.IsMetagroup(hosts);
        };

        if constexpr (std::is_same_v<decltype(query), TReadManyQuery&>) {
            hosts = THostsLabel::FromSelectors(query.Lookup->Selectors, isMetagroup);
        } else {
            hosts = THostsLabel::FromSelectors(query.Selectors, isMetagroup);
        }

        if (hosts.Type == THostsLabel::EType::METAGROUP) {
            ResolveGroups(std::move(hosts.Value), std::move(query), std::move(handler), std::move(ev->TraceId));
        } else {
            ProcessQueryImpl(std::move(hosts), std::move(query), std::move(handler), std::move(ev->TraceId));
        }
    }

    template <typename TQueryType, typename THandlerType>
    void ResolveGroups(TString&& metagroup, TQueryType&& query, THandlerType&& handler, TSpanId&& traceCtx) {
        auto project = query.Project;

        auto groups = Resolver_.TryResolveGroups(metagroup, project);
        if (groups) {
            ProcessQueryImpl(THostsLabel{*groups, THostsLabel::EType::GROUP}, std::move(query), std::move(handler), std::move(traceCtx));
        } else {
            TProcessingQuery pq(std::move(metagroup), std::move(query), std::move(handler), std::move(traceCtx));
            Resolver_.TrySendProcessingQuery(project, std::move(pq), SelfId());
        }
    }

    void OnProjectGroupsError(TDataSourceEvents::TError::TPtr& ev) {
        TRACING_SPAN_END_EV(ev);
        auto& project = ev->Get()->Project;

        while (auto query = Resolver_.GetProcessingQuery(project)) {
            HandleQueryProcessingError(query.value(), *ev->Get());
        }
    }

    void HandleQueryProcessingError(const TProcessingQuery& processingQuery, const TDataSourceEvents::TError& error) {
        TRACING_SPAN_END(processingQuery.Span);
        std::visit([&error, &processingQuery](const auto& query) {
          using T = std::decay_t<decltype(query)>;

          if constexpr (std::is_same_v<T, TReadManyQuery>) {
              auto& handler = std::get<IResultHandlerPtr<TReadManyResult>>(processingQuery.Handler);
              handler->OnError(TString{error.Project}, error.Status, TString{error.Message});
          } else if constexpr (std::is_same_v<T, TFindQuery>) {
              auto& handler = std::get<IResultHandlerPtr<TFindResult>>(processingQuery.Handler);
              handler->OnError(TString{error.Project}, error.Status, TString{error.Message});
          } else if constexpr (std::is_same_v<T, TMetricNamesQuery>) {
              auto& handler = std::get<IResultHandlerPtr<TMetricNamesResult>>(processingQuery.Handler);
              handler->OnError(TString{error.Project}, error.Status, TString{error.Message});
          } else if constexpr (std::is_same_v<T, TLabelKeysQuery>) {
              auto& handler = std::get<IResultHandlerPtr<TLabelKeysResult>>(processingQuery.Handler);
              handler->OnError(TString{error.Project}, error.Status, TString{error.Message});
          } else if constexpr (std::is_same_v<T, TLabelValuesQuery>) {
              auto& handler = std::get<IResultHandlerPtr<TLabelValuesResult>>(processingQuery.Handler);
              handler->OnError(TString{error.Project}, error.Status, TString{error.Message});
          } else if constexpr (std::is_same_v<T, TUniqueLabelsQuery>) {
              auto& handler = std::get<IResultHandlerPtr<TUniqueLabelsResult>>(processingQuery.Handler);
              handler->OnError(TString{error.Project}, error.Status, TString{error.Message});
          }
        }, processingQuery.Query);
    }

    void OnProjectGroupsSuccess(TDataSourceEvents::TSuccess<TLabelValuesResult>::TPtr& ev) {
        TRACING_SPAN_END_EV(ev);
        auto& result = ev->Get()->Result;
        auto& project = result->Project;

        Resolver_.FillGroupsOrHosts(ev->Get()->Result);

        while (auto processingQuery = Resolver_.GetResolvedProcessingQuery(project)) {
            auto& [resolvedGroups, query] = processingQuery.value();
            ProcessQuery(resolvedGroups, std::move(query));
        }

        while (auto resolvedQuery = Resolver_.GetResolvedQuery(project)) {
            auto& [query, metagroups, hosts] = resolvedQuery.value();
            auto h = MakeIntrusive<TLabelValuesHandler>(query.Handler, metagroups, hosts);
            HandleLabelValuesQueryWithMetagroups(std::move(query.Query), std::move(h), TSpanId(ev->TraceId));
        }
    }

    void ProcessQuery(const TString& resolvedGroups, TProcessingQuery&& processingQuery) {
        std::visit([this, &resolvedGroups, &processingQuery](auto& query) {
            using T = std::decay_t<decltype(query)>;
            auto type = THostsLabel::EType::GROUP;
            TSpanId& span = processingQuery.Span;

            if constexpr (std::is_same_v<T, TReadManyQuery>) {
                auto& handler = std::get<IResultHandlerPtr<TReadManyResult>>(processingQuery.Handler);
                ProcessQueryImpl(THostsLabel{resolvedGroups, type}, std::move(query), std::move(handler), std::move(span));
            } else if constexpr (std::is_same_v<T, TFindQuery>) {
                auto& handler = std::get<IResultHandlerPtr<TFindResult>>(processingQuery.Handler);
                ProcessQueryImpl(THostsLabel{resolvedGroups, type}, std::move(query), std::move(handler), std::move(span));
            } else if constexpr (std::is_same_v<T, TMetricNamesQuery>) {
                auto& handler = std::get<IResultHandlerPtr<TMetricNamesResult>>(processingQuery.Handler);
                ProcessQueryImpl(THostsLabel{resolvedGroups, type}, std::move(query), std::move(handler), std::move(span));
            } else if constexpr (std::is_same_v<T, TLabelKeysQuery>) {
                auto& handler = std::get<IResultHandlerPtr<TLabelKeysResult>>(processingQuery.Handler);
                ProcessQueryImpl(THostsLabel{resolvedGroups, type}, std::move(query), std::move(handler), std::move(span));
            } else if constexpr (std::is_same_v<T, TLabelValuesQuery>) {
                auto& handler = std::get<IResultHandlerPtr<TLabelValuesResult>>(processingQuery.Handler);
                ProcessQueryImpl(THostsLabel{resolvedGroups, type}, std::move(query), std::move(handler), std::move(span));
            } else if constexpr (std::is_same_v<T, TUniqueLabelsQuery>) {
                auto& handler = std::get<IResultHandlerPtr<TUniqueLabelsResult>>(processingQuery.Handler);
                ProcessQueryImpl(THostsLabel{resolvedGroups, type}, std::move(query), std::move(handler), std::move(span));
            }
        }, processingQuery.Query);
    }

    template <typename TQueryType, typename THandlerType>
    void ProcessQueryImpl(THostsLabel&& hosts, TQueryType&& query, THandlerType&& handler, TSpanId traceCtx) {
        if (hosts.Type != THostsLabel::EType::UNKNOWN) {
            SelectorsTransformer_.FillHostOrGroup(query.Selectors, hosts);
            SelectorsTransformer_.FillClusterService(query.Selectors, ExtractItype(query.Project), hosts);
        }
        HandleProcessedQuery(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void ProcessQueryImpl(THostsLabel&& hosts, TReadManyQuery&& query, IResultHandlerPtr<TReadManyResult>&& handler, TSpanId span) {
        if (hosts.Type != THostsLabel::EType::UNKNOWN) {
            SelectorsTransformer_.FillHostOrGroup(query.Lookup->Selectors, hosts);
            SelectorsTransformer_.FillClusterService(query.Lookup->Selectors, ExtractItype(query.Project), hosts);
        }

        bool allowSelectorsCollapsing = true;
        for (const auto& key: query.Lookup->GroupBy) {
            if (key != NLabels::LABEL_HOSTS && key != NYasm::LABEL_SIGNAL && key != NLabels::LABEL_HOST && key != NLabels::LABEL_GROUP) {
                allowSelectorsCollapsing = false;
                break;
            }
        }

        if (allowSelectorsCollapsing) {
            Register(SelectorCollapsingActor(std::move(query), handler, YasmDataSource_, Registry_, std::move(span)).release());
        } else {
            Register(YasmMetricsAggregator(std::move(query), handler, YasmDataSource_, Registry_, std::move(span)).release());
        }
    }

    void ProcessQueryImpl(THostsLabel&& hosts, TLabelKeysQuery&& query, IResultHandlerPtr<TLabelKeysResult>&& handler, TSpanId span) {
        if (hosts.Type != THostsLabel::EType::UNKNOWN) {
            SelectorsTransformer_.FillHostOrGroup(query.Selectors, hosts);
            SelectorsTransformer_.FillClusterService(query.Selectors, ExtractItype(query.Project), hosts);
            HandleProcessedQuery(std::move(query), std::move(handler), std::move(span));
            return;
        }

        HandleProcessedQuery(std::move(query), MakeIntrusive<TLabelKeysHandler>(std::move(handler)), std::move(span));
    }

    void HandleProcessedQuery(TFindQuery&& query, IResultHandlerPtr<TFindResult>&& handler, TSpanId traceCtx) {
        YasmDataSource_->Find(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void HandleProcessedQuery(TMetricNamesQuery&& query, IResultHandlerPtr<TMetricNamesResult>&& handler, TSpanId traceCtx) {
        YasmDataSource_->MetricNames(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void HandleProcessedQuery(TLabelKeysQuery&& query, IResultHandlerPtr<TLabelKeysResult>&& handler, TSpanId traceCtx) {
        YasmDataSource_->LabelKeys(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void HandleProcessedQuery(TLabelValuesQuery&& query, IResultHandlerPtr<TLabelValuesResult>&& handler, TSpanId&& traceCtx) {
        auto project = query.Project;
        bool wasHostsPresent = false;

        for (auto& key: query.Keys) {
            if (key == NLabels::LABEL_HOSTS) {
                std::swap(key, query.Keys.back());
                query.Keys.pop_back();

                wasHostsPresent = true;
                break;
            }
        }

        if (!wasHostsPresent && !query.Keys.empty()) {
            YasmDataSource_->LabelValues(std::move(query), std::move(handler), std::move(traceCtx));
            return;
        }

        auto metagroupsAndHostsOpt = Resolver_.TryResolveMetagroupsAndHosts(query.Project);
        if (metagroupsAndHostsOpt.has_value()) {
            auto [metagroups, hosts] = metagroupsAndHostsOpt.value();
            auto h = MakeIntrusive<TLabelValuesHandler>(std::move(handler), *metagroups, *hosts);
            HandleLabelValuesQueryWithMetagroups(std::move(query), std::move(h), std::move(traceCtx));
        } else {
            TQueryRequiringMetagroupsAndHosts qrmh(std::move(query), std::move(handler), std::move(traceCtx));
            Resolver_.TrySendMetagroupsAndHostsQuery(project, std::move(qrmh), SelfId());
        }
    }

    void HandleLabelValuesQueryWithMetagroups(TLabelValuesQuery&& query, IResultHandlerPtr<TLabelValuesResult>&& handler, TSpanId&& traceCtx) {
        YasmDataSource_->LabelValues(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void HandleProcessedQuery(TUniqueLabelsQuery&& query, IResultHandlerPtr<TUniqueLabelsResult>&& handler, TSpanId traceCtx) {
        YasmDataSource_->UniqueLabels(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void OnPoison(TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new TEvents::TEvPoisonTaken{});
        PassAway();
    }

    static constexpr char ActorName[] = "Yasm/Datasource";

private:
    TGroupsHostsResolver Resolver_;
    TYasmSelectorsTransformer SelectorsTransformer_;
    IDataSourcePtr YasmDataSource_;

    NMonitoring::TMetricRegistry& Registry_;
};

} // namespace

std::unique_ptr<IActor> YasmDataSourceActor(
        TYasmConfig yasmConfig,
        IDataSourcePtr baseDataSource,
        NMonitoring::TMetricRegistry& registry)
{
    return std::make_unique<TYasmDataSourceActor>(
            std::move(yasmConfig),
            std::move(baseDataSource),
            registry);
}

} // namespace NSolomon::NDataProxy
