#include "deserializer.h"

#include <infra/yasm/common/groups/metagroup_groups.h>
#include <infra/yasm/common/points/accumulators/accumulators.h>
#include <infra/yasm/common/points/value/types.h>
#include <infra/yasm/common/points/hgram/hgram.h>

#include <util/generic/ymath.h>

namespace NSolomon::NDataProxy {

using NYasm::NGroups::TMetagroupGroupsConfig;
using NZoom::NValue::TValue;
using namespace NZoom::NHgram;
using namespace NZoom::NAccumulators;

namespace {

TValue DoubleToValue(double value) {
    return IsNan(value) ? TValue() : TValue(value);
}

TValue PointToValue(const NTs::NValue::TLogHistogram& value) {
    TVector<double> bucketWeights;
    bucketWeights.reserve(value.Values.size());
    std::copy(value.Values.begin(), value.Values.end(), std::back_inserter(bucketWeights));
    return TValue(THgram::Normal(std::move(bucketWeights), value.ZeroCount, value.StartPower));
}

TValue PointToValue(const NTs::NValue::THistogram& value) {
    TUgramBuckets buckets;
    if (!value.Buckets.empty()) {
        // TODO (alovtsyus): maybe need to apply Denom?
        buckets.reserve(value.Buckets.size());
        double previousUpperBound = value.Buckets.front().UpperBound;
        for (auto& bucket: value.Buckets) {
            if (previousUpperBound != bucket.UpperBound || bucket.Value > 0) {
                buckets.emplace_back(previousUpperBound, bucket.UpperBound, bucket.Value);
            }
            previousUpperBound = bucket.UpperBound;
        }
    }
    return TValue(THgram::Ugram(std::move(buckets)));
}

TValue PointToValue(const NTs::NValue::TSummaryDouble& value, EAccumulatorType targetType) {
    switch (targetType) {
        case EAccumulatorType::Min: {
            return DoubleToValue(value.Min);
        }
        case EAccumulatorType::Summ:
        case EAccumulatorType::SummNone: {
            return DoubleToValue(value.Sum);
        }
        case EAccumulatorType::Max: {
            return DoubleToValue(value.Max);
        }
        case EAccumulatorType::Last: {
            return DoubleToValue(value.Last);
        }
        case EAccumulatorType::Avg:
        case EAccumulatorType::Average: {
            return TValue(value.Sum, value.CountValue);
        }
        default: {
            ythrow yexception() << "unsupported metric aggregation type: " << targetType;
        }
    }
}

} // namespace

NZoom::NValue::TValue PointToValue(
        NMonitoring::EMetricType metricType,
        const NTs::TVariantPoint& point,
        NZoom::NAccumulators::EAccumulatorType targetType)
{
    switch (metricType) {
        case NMonitoring::EMetricType::LOGHIST:
            return PointToValue(point.Get<NTs::NValue::TLogHistogram>());

        case NMonitoring::EMetricType::HIST:
            return PointToValue(point.Get<NTs::NValue::THistogram>());

        case NMonitoring::EMetricType::DSUMMARY:
            return PointToValue(point.Get<NTs::NValue::TSummaryDouble>(), targetType);

        default:
            ythrow yexception() << "Unable to aggregate metrics of type: " << metricType;
    }
}

} // namespace NSolomon::NDataProxy
