#include "cluster_name.h"
#include "deserializer.h"
#include "dynamic_aggregation.h"
#include "labels.h"
#include "serializer.h"
#include "yasm_common.h"

#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/stockpile_codec/metric_archive.h>
#include <solomon/libs/cpp/ts_codec/points.h>
#include <solomon/libs/cpp/yasm/constants/interval.h>
#include <solomon/libs/cpp/yasm/constants/labels.h>
#include <solomon/libs/cpp/yasm/shard_config/shard_config.h>

#include <solomon/services/dataproxy/lib/datasource/datasource.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>
#include <solomon/services/dataproxy/lib/timeseries/compressed.h>

#include <infra/yasm/common/labels/signal/signal_name.h>
#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/yasm/common/labels/tags/dynamic_filter.h>
#include <infra/yasm/common/points/accumulators/accumulators.h>
#include <infra/yasm/common/points/hgram/ugram/compress/compress.h>
#include <infra/yasm/stockpile_client/common/base_types.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/monlib/metrics/histogram_snapshot.h>

using namespace NActors;
using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

using yandex::solomon::model::MetricType;
using ::NYasm::NGroups::TMetagroupGroupsConfig;
using NTags::TInstanceKey;
using NZoom::NSignal::TSignalName;
using NZoom::NValue::TValue;
using namespace NZoom::NHgram;
using namespace NZoom::NAccumulators;

class TMetricAggregationHandler: public IResultHandler<TReadManyResult> {
public:
    TMetricAggregationHandler(
            TReadManyQuery originalQuery,
            IResultHandlerPtr<TReadManyResult> handler,
            TSpanId traceCtx,
            NActors::TActorSystem* as)
        : RequestResolution_(ExtractAndAdjustResolution(originalQuery))
        , Handler_(std::move(handler))
        , TraceCtx_{std::move(traceCtx)}
        , ActorSystem_{as}
    {
        originalQuery.Time = ExtractAndRoundTimeRange(originalQuery, RequestResolution_);
        FillBaseQueryFields(originalQuery, QueryBase_);
        GroupBy_ = std::move(originalQuery.Lookup->GroupBy);
    }

private:
    void OnSuccess(std::unique_ptr<TReadManyResult> resultPtr) override {
        auto& result = *(resultPtr);

        Errors_ = std::move(result.Errors);

        if (result.Metrics.empty()) {
            ReplyWithEmptyResult();
            return;
        }

        absl::flat_hash_map<TLabels<ui32>, TCompactAccumulatorsArray> accumulators;
        absl::flat_hash_map<TLabels<ui32>, TSignalName> groupToSignal;
        absl::flat_hash_map<TLabels<ui32>, TInstant> groupToMinLastTs;

        for (auto& metric: result.Metrics) {
            metric.Meta.Labels.push_back({result.Strings.Put(NYasm::LABEL_SIGNAL), metric.Meta.Name});
        }

        auto groupByKeys = GroupByKeys(result.Strings);
        auto resultStrings = result.Strings.Build();

        TLabels<ui32> groupByValues(GroupBy_.size());
        auto now = ActorSystem_->Timestamp();
        auto alignedNow = TInstant::Seconds(now.Seconds() - now.Seconds() % RequestResolution_.Seconds());

        for (const auto& metric: result.Metrics) {
            if (!metric.TimeSeries) {
                continue;
            }

            if (!GroupBy_.empty()) {
                for (auto& label: metric.Meta.Labels) {
                    if (auto it = groupByKeys.find(label.Key); it != groupByKeys.end()) {
                        groupByValues[it->second] = {
                                Labels_.Put(resultStrings[label.Key]),
                                Labels_.Put(resultStrings[label.Value])};
                    }
                }
            }

            auto& minTs = groupToMinLastTs[groupByValues];
            if (minTs == TInstant::Zero()) {
                minTs = TInstant::Max();
            }
            minTs = Min(minTs, QueryBase_.Time.From + metric.TimeSeries->PointCount() * RequestResolution_);
            minTs = Max(minTs, alignedNow - TDuration::Seconds(20));
        }

        for (const auto& metric: result.Metrics) {
            NMonitoring::EMetricType type = metric.Meta.Type;
            NTs::TVariantPoint point;
            if (!metric.TimeSeries) {
                // XXX: workaround SOLOMON-7512
                MON_ERROR_C(*ActorSystem_, Yasm, "metric with absent TimeSeries");
                continue;
            }
            if (!GroupBy_.empty()) {
                for (auto& label: metric.Meta.Labels) {
                    if (auto it = groupByKeys.find(label.Key); it != groupByKeys.end()) {
                        groupByValues[it->second] = {
                                Labels_.Put(resultStrings[label.Key]),
                                Labels_.Put(resultStrings[label.Value])};
                    }
                }
            }

            auto signalName = TSignalName::TryNew(TString{resultStrings[metric.Meta.Name]});
            groupToSignal.emplace(groupByValues, *signalName);

            auto aggregationType = signalName->GetAggregationType(EAggregationMethod::MetaGroup).GetRef();
            auto storageType = signalName->GetAggregationType(EAggregationMethod::Rollup).GetRef();

            auto minTs = groupToMinLastTs[groupByValues];
            size_t pointCount = (minTs - QueryBase_.Time.From).GetValue() / RequestResolution_.GetValue();

            if (!accumulators.contains(groupByValues)) {
                accumulators.insert({
                    groupByValues,
                    TCompactAccumulatorsArray(aggregationType, pointCount, ENullPolicy::Nullable)});
            }

            auto& accumulator = accumulators.at(groupByValues);

            for (auto it = metric.TimeSeries->Iterator(); it->Next(&point); ) {
                size_t offset = (point.Time - QueryBase_.Time.From).GetValue() / RequestResolution_.GetValue();
                if (offset >= pointCount) {
                    MON_WARN_C(
                            *ActorSystem_,
                            Yasm,
                            "skip point: ts out of range: project="
                                    << QueryBase_.Project << ", too many points in timeseries"
                                    << ", period=" << QueryBase_.Time.From << " -- " << QueryBase_.Time.To
                                    << ", expected point count=" << pointCount
                                    << ", actual point count=" << metric.TimeSeries->PointCount());
                    break;
                }

                if (point.Time > minTs) {
                    MON_DEBUG_C(
                            *ActorSystem_,
                            Yasm,
                            "skip points: ts (" << point.Time << ") out of range: project=" << QueryBase_.Project
                                                << ", not all responses have points after " << minTs
                                                << ", alignedNow=" << alignedNow);
                    break;
                }

                // TODO: it is better to switch on type only once, instead of on every point
                auto value = PointToValue(type, point, storageType);
                accumulator.Mul(value, offset);
            }
        }

        BuildResultAndReply(accumulators, groupToSignal);
    }

    void OnError(TString&&, EDataSourceStatus status, TString&& message) override {
        ReportErrorAndPassAway(status, std::move(message));
    }

    void ReportErrorAndPassAway(EDataSourceStatus status, TString error) {
        TRACING_SPAN_END(TraceCtx_);
        Handler_->OnError(std::move(QueryBase_.Project), status, std::move(error));
    }

    void ReplyWithEmptyResult() {
        TRACING_SPAN_END(TraceCtx_);
        auto res = std::make_unique<TReadManyResult>();
        res->Errors = std::move(Errors_);
        Handler_->OnSuccess(std::move(res));
    }

    TMetric<ui32> MakeAggregatedMetric(const TLabels<ui32>& groupByLabels,
           const TCompactAccumulatorsArray& accumulators,
           const TLabel<ui32>& labelSignal)
    {
        TMetric<ui32> resultMetric;
        TValueTypeDetector detector;
        for (auto offset: xrange(accumulators.Len())) {
            NZoom::NValue::TValueRef value = accumulators.GetValue(offset);
            detector.OnValue(value);
        }

        resultMetric.Type = detector.GetType();
        resultMetric.Name = Labels_.Put("");
        resultMetric.Labels.emplace_back(labelSignal);

        for (auto& label: groupByLabels) {
            if (label.Key != labelSignal.Key) {
                resultMetric.Labels.emplace_back(label.Key, label.Value);
            }
        }
        // TODO (alovtsyus):
        // Decide whether we need more labels in result. We can not put selectors here, and we accept only single
        // signals. So maybe a single signal label is enough.
        return std::move(resultMetric);
    }

    void BuildResultAndReply(
            absl::flat_hash_map<TLabels<ui32>, TCompactAccumulatorsArray>& accumulators,
            const absl::flat_hash_map<TLabels<ui32>, TSignalName>& groupToSignal)
    {
        auto result = std::make_unique<TReadManyResult>();

        for (auto& [groupByLabels, accumulator]: accumulators) {
            const auto& signalName = groupToSignal.at(groupByLabels);

            auto aggregationType = signalName.GetAggregationType(EAggregationMethod::MetaGroup).GetRef();

            if (aggregationType == EAccumulatorType::Hgram) {
                CompressHistPoints(accumulator);
            }

            TLabel<ui32> labelSignal(Labels_.Put(NYasm::LABEL_SIGNAL), Labels_.Put(signalName.GetName()));
            auto resultMetric = MakeAggregatedMetric(groupByLabels, accumulator, labelSignal);

            TMetricSerializer serializer(resultMetric.Type, accumulator, QueryBase_.Time.From, RequestResolution_);
            NTs::TBitBuffer buffer = serializer.Serialize();
            ui32 pointCount = static_cast<ui32>(accumulator.Len());

            auto timeSeries = std::make_unique<TCompressedTimeSeries>(
                    resultMetric.Type,
                    TypeToAggrColumns(resultMetric.Type),
                    pointCount,
                    std::move(buffer));

            result->Metrics.emplace_back();
            auto& metric = result->Metrics.back();

            metric.Meta = std::move(resultMetric);
            metric.TimeSeries = std::move(timeSeries);
        }

        result->Strings = std::move(Labels_);
        result->Errors = std::move(Errors_);
        TRACING_SPAN_END(TraceCtx_);
        Handler_->OnSuccess(std::move(result));
    }

    THashMap<ui32, ui32> GroupByKeys(NStringPool::TStringPoolBuilder& stringPoolBuilder) {
        THashMap<ui32, ui32> groupByKeys;
        for (ui32 index = 0; index < GroupBy_.size(); ++index) {
            ui32 labelKey = stringPoolBuilder.Put(GroupBy_[index]);
            groupByKeys[labelKey] = index;
        }
        return std::move(groupByKeys);
    }

private:
    std::optional<TValue> TryCompressPoint(const NTs::THistogramPoint& value) {
        TUgramBuckets buckets;
        if (!value.Buckets.empty()) {
            buckets.reserve(value.Buckets.size());
            double previousUpperBound = value.Buckets.front().UpperBound;
            for (auto& bucket: value.Buckets) {
                if (previousUpperBound != bucket.UpperBound || bucket.Value > 0) {
                    buckets.emplace_back(previousUpperBound, bucket.UpperBound, bucket.Value);
                }
                previousUpperBound = bucket.UpperBound;
            }
        }
        size_t limit = NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT;
        size_t finalSize = BucketsSizeAfterSerializing(buckets);
        if (finalSize > limit) {
            size_t newSize = buckets.size() - (finalSize - limit);
            buckets = std::move(NZoom::NHgram::TUgramCompressor::GetInstance().Compress(buckets, newSize));
            return TValue(THgram::Ugram(std::move(buckets)));
        }
        return std::nullopt;
    }

    void CompressHistPoints(TCompactAccumulatorsArray& accumulator) {
        TInstant timestamp = QueryBase_.Time.From;
        for (auto offset: xrange(accumulator.Len())) {
            NZoom::NValue::TValueRef value = accumulator.GetValue(offset);
            TValueToHistPoint valueToPoint(timestamp, RequestResolution_);
            value.Update(valueToPoint);
            auto point = valueToPoint.GetPoint();
            auto newValueOpt = TryCompressPoint(point);
            if (newValueOpt.has_value()) {
                accumulator.Set(newValueOpt.value(), offset);
            }
            timestamp += RequestResolution_;
        }
    }

    TDuration RequestResolution_;
    TQuery QueryBase_;
    IResultHandlerPtr<TReadManyResult> Handler_;
    TSpanId TraceCtx_;
    NActors::TActorSystem* ActorSystem_;
    TVector<TString> GroupBy_;
    NStringPool::TStringPoolBuilder Labels_;
    TDataSourceErrors Errors_;
};

class TYasmDynAggrDataSource: public IDataSource {
public:
    TYasmDynAggrDataSource(TIntrusivePtr<IDataSource> baseDataSource, NActors::TActorSystem* as)
        : BaseDataSource_(std::move(baseDataSource))
        , ActorSystem_{as}
    {
    }

private:
    bool CanHandle(const TQuery& query) const override {
        return query.Project.StartsWith(NHistDb::NStockpile::STOCKPILE_YASM_PROJECTS_PREFIX) &&
               BaseDataSource_->CanHandle(query);
    }

    void ReadMany(TReadManyQuery query, IResultHandlerPtr<TReadManyResult> handler, TSpanId traceCtx) const override {
        Y_ENSURE(query.ResolvedKeys, "keys should already be resolved");

        TDuration period = TDuration::Seconds(5);
        if (!query.Operations.empty()) {
            for (const auto& query: query.Operations) {
                if (!query.has_downsampling()) {
                    continue;
                }

                const auto& downsampling = query.downsampling();

                if (downsampling.grid_millis()) {
                    period = TDuration::MilliSeconds(downsampling.grid_millis());
                    break;
                }
            }
        }

        if (query.Time.From == query.Time.To) {
            handler->OnSuccess(std::make_unique<TReadManyResult>());
            return;
        }
        auto wrapperHandler =
                MakeIntrusive<TMetricAggregationHandler>(query.Clone(), std::move(handler), TSpanId(traceCtx), ActorSystem_);
        BaseDataSource_->ReadMany(std::move(query), std::move(wrapperHandler), std::move(traceCtx));
    }

    void Find(TFindQuery query, IResultHandlerPtr<TFindResult> handler, TSpanId traceCtx) const override {
        BaseDataSource_->Find(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void ResolveOne(TResolveOneQuery query, IResultHandlerPtr<TResolveOneResult> handler, TSpanId traceCtx) const override {
        BaseDataSource_->ResolveOne(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void ResolveMany(TResolveManyQuery query, IResultHandlerPtr<TResolveManyResult> handler, TSpanId traceCtx) const override {
        BaseDataSource_->ResolveMany(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void MetricNames(TMetricNamesQuery query, IResultHandlerPtr<TMetricNamesResult> handler, TSpanId traceCtx) const override {
        BaseDataSource_->MetricNames(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void LabelKeys(TLabelKeysQuery query, IResultHandlerPtr<TLabelKeysResult> handler, TSpanId traceCtx) const override {
        BaseDataSource_->LabelKeys(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void LabelValues(TLabelValuesQuery query, IResultHandlerPtr<TLabelValuesResult> handler, TSpanId traceCtx) const override {
        BaseDataSource_->LabelValues(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void UniqueLabels(TUniqueLabelsQuery query, IResultHandlerPtr<TUniqueLabelsResult> handler, TSpanId traceCtx) const override {
        BaseDataSource_->UniqueLabels(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void ReadOne(TReadOneQuery query, IResultHandlerPtr<TReadOneResult> handler, TSpanId traceCtx) const override {
        BaseDataSource_->ReadOne(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void WaitUntilInitialized() const override {
        BaseDataSource_->WaitUntilInitialized();
    }

private:
    IDataSourcePtr BaseDataSource_;
    NActors::TActorSystem* ActorSystem_;
};

} // namespace

IDataSourcePtr YasmDynAggrDataSource(TIntrusivePtr<IDataSource> baseDataSource, NActors::TActorSystem* as) {
    return MakeIntrusive<TYasmDynAggrDataSource>(std::move(baseDataSource), as);
}

} // namespace NSolomon::NDataProxy
