#include "groups_hosts_resolver.h"

#include <solomon/services/dataproxy/lib/datasource/yasm/selectors.h>
#include <solomon/services/dataproxy/lib/datasource/datasource.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>
#include <solomon/libs/cpp/trace/trace.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <infra/yasm/common/groups/metagroup_groups.h>

#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>

#include <util/generic/queue.h>

using namespace NActors;
using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {

THashMap<TString, TString> BuildGroupToMetagroupMap(const THashMap<TString, TVector<TString>>& metagroupToGroups) {
    THashMap<TString, TString> groupToMetagroup;
    for (const auto& [metagroup, groups]: metagroupToGroups) {
        for (const auto& group: groups) {
            groupToMetagroup[group] = metagroup;
        }
    }

    return groupToMetagroup;
}

TGroupsHostsResolver::TGroupsHostsResolver(
        std::shared_ptr<const THashMap<TString, TVector<TString>>> metagroupToGroups,
        const IDataSourcePtr& yasmDataSource
        )
    : MetagroupToGroups_{std::move(metagroupToGroups)}
    , GroupToMetagroup_{BuildGroupToMetagroupMap(*MetagroupToGroups_)}
    , YasmDataSource_{yasmDataSource}
{
}

bool TGroupsHostsResolver::IsMetagroup(TStringBuf hosts) {
    return MetagroupToGroups_->contains(hosts);
}

TLabelValuesQuery TGroupsHostsResolver::MakeProjectGroupsQuery(const TString& project) {
    TLabelValuesQuery projectGroupsQuery;
    projectGroupsQuery.Project = project;
    projectGroupsQuery.Deadline = TDuration::Seconds(10).ToDeadLine();

    projectGroupsQuery.Selectors.Add(NLabels::LABEL_CLUSTER, ALL_GROUPS);
    projectGroupsQuery.Selectors.Add(NLabels::LABEL_SERVICE, SERVICE_VALUE);

    projectGroupsQuery.Keys = {TString{NLabels::LABEL_GROUP}};
    projectGroupsQuery.Limit = 1000;
    auto now = TInstant::Now();
    projectGroupsQuery.Time.From = now - TDuration::Days(1);
    projectGroupsQuery.Time.To = now;

    return projectGroupsQuery;
}

TLabelValuesQuery TGroupsHostsResolver::MakeProjectHostsQuery(const TString& project) {
    TLabelValuesQuery projectGroupsQuery;
    projectGroupsQuery.Project = project;
    projectGroupsQuery.Deadline = TDuration::Seconds(10).ToDeadLine();

    projectGroupsQuery.Selectors.Add(NLabels::LABEL_CLUSTER, ALL_HOSTS);
    projectGroupsQuery.Selectors.Add(NLabels::LABEL_SERVICE, SERVICE_VALUE);

    projectGroupsQuery.Keys = {TString{NLabels::LABEL_HOST}};
    projectGroupsQuery.Limit = 1000;
    auto now = TInstant::Now();
    projectGroupsQuery.Time.From = now - TDuration::Days(1);
    projectGroupsQuery.Time.To = now;

    return projectGroupsQuery;
}

TString TGroupsHostsResolver::BuildGroups(const TString& metagroup, const THashSet<TString>& projectGroups) const {
    TStringBuilder builder;
    bool first = true;
    for (const auto& group: MetagroupToGroups_->at(metagroup)) {
        if (projectGroups.contains(group)) {
            if (!first) {
                builder << '|';
            }
            builder << group;
            first = false;
        }
    }
    return builder;
}

THashSet<TString> TGroupsHostsResolver::BuildMetagroups(const THashSet<TString>& projectGroups) {
    THashSet<TString> result;
    for (const auto& group: projectGroups) {
        auto it = GroupToMetagroup_.find(group);
        if (!it) {
            MON_ERROR(Yasm, "Got an unknown group " << group);
        } else {
            result.insert(it->second);
        }
    }
    return result;
}

TString* TGroupsHostsResolver::TryResolveGroups(const TString& metagroup, const TString& project) {
    if (auto it = ResolvedGroups_.find(std::pair{project, metagroup}); it != ResolvedGroups_.end()) {
        return &it->second;
    }

    if (auto it = ProjectToGroups_.find(project); it != ProjectToGroups_.end()) {
        auto& groups = ResolvedGroups_[std::pair{project, metagroup}];
        groups = BuildGroups(metagroup, it->second);
        return &groups;
    }
    return nullptr;
}

void TGroupsHostsResolver::TrySendProcessingQuery(const TString& project, TProcessingQuery&& pq, const TActorIdentity& replyTo) {
    if (IsProcessingQuery(project)) {
        AddProcessingQuery(project, std::move(pq));
        return;
    }
    TSpanId spanClone{pq.Span};
    AddProcessingQuery(project, std::move(pq));

    SendProjectGroupsQuery(project, std::move(spanClone), replyTo);
}

bool TGroupsHostsResolver::IsProcessingQuery(const TString& project) {
    return ProjectToProcessingQueries_.contains(project);
}

bool TGroupsHostsResolver::IsQueryRequiringMetagroupsAndHosts(const TString& project) {
    return ProjectToQueriesRequiringMetagroupsAndHosts_.contains(project);
}

void TGroupsHostsResolver::AddProcessingQuery(const TString& project, TProcessingQuery&& query) {
    ProjectToProcessingQueries_[project].emplace(std::move(query));
}

void TGroupsHostsResolver::AddQueryRequiringMetagroupsAndHosts(const TString& project, TQueryRequiringMetagroupsAndHosts&& query) {
    ProjectToQueriesRequiringMetagroupsAndHosts_[project].emplace(std::move(query));
}

TMetagroups* TGroupsHostsResolver::TryResolveMetagroup(const TString& project) {
    if (auto it = ProjectToMetagroups_.find(project); it != ProjectToMetagroups_.end()) {
        return &it->second;
    }

    if (auto it = ProjectToGroups_.find(project); it != ProjectToGroups_.end()) {
        auto& metagroup = ProjectToMetagroups_[project];
        metagroup = BuildMetagroups(it->second);
        return &metagroup;
    }
    return nullptr;
}

THosts* TGroupsHostsResolver::TryResolveHosts(const TString& project) {
    if (auto it = ProjectToHosts_.find(project); it != ProjectToHosts_.end()) {
        return &it->second;
    }
    return nullptr;
}

std::optional<std::pair<TMetagroups*, THosts*>> TGroupsHostsResolver::TryResolveMetagroupsAndHosts(const TString& project) {
    auto* metagroups = TryResolveMetagroup(project);
    auto* hosts = TryResolveHosts(project);
    if (metagroups && hosts) {
        return std::make_pair(metagroups, hosts);
    } else {
        return std::nullopt;
    }
}

void TGroupsHostsResolver::TrySendMetagroupsAndHostsQuery(
        const TString& project,
        TQueryRequiringMetagroupsAndHosts&& qrmh,
        const NActors::TActorIdentity& replyTo)
{
    if (IsQueryRequiringMetagroupsAndHosts(project)) {
        AddQueryRequiringMetagroupsAndHosts(project, std::move(qrmh));
        return;
    }
    TSpanId spanClone{qrmh.Span};
    AddQueryRequiringMetagroupsAndHosts(project, std::move(qrmh));

    if (!IsProcessingQuery(project) && !ProjectToMetagroups_.contains(project)) {
        SendProjectGroupsQuery(project, TSpanId(spanClone), replyTo);
    }
    if (!ProjectToHosts_.contains(project)) {
        SendProjectHostsQuery(project, std::move(spanClone), replyTo);
    }
}

void TGroupsHostsResolver::SendProjectGroupsQuery(const TString& project, TSpanId traceCtx, const TActorIdentity& replyTo) {
    auto projectGroupsQuery = MakeProjectGroupsQuery(project);
    auto span = TRACING_NEW_SPAN_START(std::move(traceCtx), "ResolveProjectGroups");
    YasmDataSource_->LabelValues(std::move(projectGroupsQuery), MakeReplyToHandler<TLabelValuesResult>(replyTo), std::move(span));
}

void TGroupsHostsResolver::SendProjectHostsQuery(const TString& project, TSpanId traceCtx, const TActorIdentity& replyTo) {
    auto projectGroupsQuery = MakeProjectHostsQuery(project);
    auto span = TRACING_NEW_SPAN_START(std::move(traceCtx), "ResolveProjectHosts");
    YasmDataSource_->LabelValues(std::move(projectGroupsQuery), MakeReplyToHandler<TLabelValuesResult>(replyTo), std::move(span));
}

void TGroupsHostsResolver::FillGroupsOrHosts(std::unique_ptr<TLabelValuesResult>& result) {
    auto& project = result->Project;
    auto stringPool = result->Strings.Build();

    if (result->Labels.empty()) {
        return;
    }
    auto& label = result->Labels[0];

    TStringBuf labelKey = stringPool[label.Key];

    THashSet<TString>* values;
    if (labelKey == NLabels::LABEL_GROUP) {
        values = &ProjectToGroups_[project];
    } else if (labelKey == NLabels::LABEL_HOST) {
        values = &ProjectToHosts_[project];
    } else {
        // unknown label key
        return;
    }
    for (auto value: label.Values) {
        values->emplace(stringPool[value]);
    }

    if (labelKey == NLabels::LABEL_GROUP) {
        auto& metagroups = ProjectToMetagroups_[project];
        if (!metagroups) {
            metagroups = BuildMetagroups(*values);
        }
    }
}

std::optional<TProcessingQuery> TGroupsHostsResolver::GetProcessingQuery(const TString& project) {
    if (auto it = ProjectToProcessingQueries_.find(project); it != ProjectToProcessingQueries_.end()) {
        if (!it->second.empty()) {
            auto query = std::move(it->second.front());
            it->second.pop();
            return std::move(query);
        }
        ProjectToProcessingQueries_.erase(it);
    }
    return std::nullopt;
}

std::optional<TResolvedProcessingQuery> TGroupsHostsResolver::GetResolvedProcessingQuery(const TString& project) {
    if (auto it = ProjectToProcessingQueries_.find(project); it != ProjectToProcessingQueries_.end()) {
        auto& queries = it->second;
        if (!queries.empty()) {
            auto query = std::move(queries.front());
            auto projectMetagroupPair = std::pair{project, query.Metagroup};
            auto& resolvedGroups = ResolvedGroups_[projectMetagroupPair];
            if (!resolvedGroups) {
                if (!ProjectToGroups_.contains(project)) {
                    return std::nullopt;
                }
                auto& groups = ProjectToGroups_[project];
                resolvedGroups = BuildGroups(query.Metagroup, groups);
            }
            queries.pop();
            return TResolvedProcessingQuery(resolvedGroups, std::move(query));
        }
        ProjectToProcessingQueries_.erase(it);
    }
    return std::nullopt;
}

std::optional<TResolvedQueryRequiringMetagroupsAndHosts> TGroupsHostsResolver::GetResolvedQuery(const TString& project) {
    if (auto it = ProjectToQueriesRequiringMetagroupsAndHosts_.find(project);
            it != ProjectToQueriesRequiringMetagroupsAndHosts_.end()) {
        if (!ProjectToHosts_.contains(project) || !ProjectToMetagroups_.contains(project)) {
            return std::nullopt;
        }
        auto& hosts = ProjectToHosts_[project];
        auto& metagroups = ProjectToMetagroups_[project];
        if (!it->second.empty()) {
            auto query = std::move(it->second.front());
            it->second.pop();
            return TResolvedQueryRequiringMetagroupsAndHosts(std::move(query), metagroups, hosts);
        }
        ProjectToQueriesRequiringMetagroupsAndHosts_.erase(it);
    }
    return std::nullopt;
}

}
