#pragma once

#include <solomon/services/dataproxy/lib/datasource/yasm/selectors.h>
#include <solomon/services/dataproxy/lib/datasource/datasource.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>
#include <solomon/libs/cpp/trace/trace.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <infra/yasm/common/groups/metagroup_groups.h>

#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>

#include <util/generic/queue.h>

namespace NSolomon::NDataProxy {

using TQueryVariant = std::variant<TReadManyQuery,
        TFindQuery,
        TMetricNamesQuery,
        TLabelKeysQuery,
        TLabelValuesQuery,
        TUniqueLabelsQuery>;

using THandlerVariant = std::variant<IResultHandlerPtr<TReadManyResult>,
        IResultHandlerPtr<TFindResult>,
        IResultHandlerPtr<TMetricNamesResult>,
        IResultHandlerPtr<TLabelKeysResult>,
        IResultHandlerPtr<TLabelValuesResult>,
        IResultHandlerPtr<TUniqueLabelsResult>>;

struct TProcessingQuery {
    TProcessingQuery(TString metagroup, TQueryVariant query, THandlerVariant handler, NTracing::TSpanId span)
            : Metagroup{std::move(metagroup)}
            , Query{std::move(query)}
            , Handler{std::move(handler)}
            , Span(std::move(span))
    {
    }

    TString Metagroup;
    TQueryVariant Query;
    THandlerVariant Handler;
    NTracing::TSpanId Span;
};

using TResolvedProcessingQuery = std::pair<TString, TProcessingQuery>;

struct TQueryRequiringMetagroupsAndHosts {
    TQueryRequiringMetagroupsAndHosts(TLabelValuesQuery query, IResultHandlerPtr<TLabelValuesResult> handler, NTracing::TSpanId span)
        : Query{std::move(query)}
        , Handler{std::move(handler)}
        , Span(std::move(span))
    {
    }

    TLabelValuesQuery Query;
    IResultHandlerPtr<TLabelValuesResult> Handler;
    NTracing::TSpanId Span;
};

using TMetagroups = THashSet<TString>;
using THosts = THashSet<TString>;

struct TResolvedQueryRequiringMetagroupsAndHosts {
    TResolvedQueryRequiringMetagroupsAndHosts(
            TQueryRequiringMetagroupsAndHosts query,
            const TMetagroups& metagroups,
            const THosts& hosts)
        : Query(std::move(query))
        , Metagroups(metagroups)
        , Hosts(hosts)
    {
    }

    TQueryRequiringMetagroupsAndHosts Query;
    const TMetagroups& Metagroups;
    const THosts& Hosts;
};

class TGroupsHostsResolver {
public:
    TGroupsHostsResolver(
            std::shared_ptr<const THashMap<TString, TVector<TString>>> metagroupToGroups,
            const IDataSourcePtr& yasmDataSource);

    bool IsMetagroup(TStringBuf hosts);

    TString* TryResolveGroups(const TString& metagroup, const TString& project);
    std::optional<std::pair<TMetagroups*, THosts*>> TryResolveMetagroupsAndHosts(const TString& project);

    void TrySendProcessingQuery(const TString& project, TProcessingQuery&& pq, const NActors::TActorIdentity& replyTo);
    void TrySendMetagroupsAndHostsQuery(const TString& project, TQueryRequiringMetagroupsAndHosts&& qrmh, const NActors::TActorIdentity& replyTo);

    void FillGroupsOrHosts(std::unique_ptr<TLabelValuesResult>& result);

    std::optional<TProcessingQuery> GetProcessingQuery(const TString& project);
    std::optional<TResolvedProcessingQuery> GetResolvedProcessingQuery(const TString& project);
    std::optional<TResolvedQueryRequiringMetagroupsAndHosts> GetResolvedQuery(const TString& project);

private:
    TMetagroups* TryResolveMetagroup(const TString& project);
    THosts* TryResolveHosts(const TString& project);

    static TLabelValuesQuery MakeProjectGroupsQuery(const TString& project);
    static TLabelValuesQuery MakeProjectHostsQuery(const TString& project);

    TString BuildGroups(const TString& metagroup, const THashSet<TString>& projectGroups) const;
    THashSet<TString> BuildMetagroups(const THashSet<TString>& projectGroups);

    bool IsProcessingQuery(const TString& project);
    bool IsQueryRequiringMetagroupsAndHosts(const TString& project);

    void AddProcessingQuery(const TString& project, TProcessingQuery&& query);
    void AddQueryRequiringMetagroupsAndHosts(const TString& project, TQueryRequiringMetagroupsAndHosts&& query);

    void SendProjectGroupsQuery(const TString& project, NTracing::TSpanId traceCtx, const NActors::TActorIdentity& replyTo);
    void SendProjectHostsQuery(const TString& project, NTracing::TSpanId traceCtx, const NActors::TActorIdentity& replyTo);

private:
    std::shared_ptr<const THashMap<TString, TVector<TString>>> MetagroupToGroups_;
    const THashMap<TString, TString> GroupToMetagroup_;

    THashMap<TString, TQueue<TProcessingQuery>> ProjectToProcessingQueries_;
    THashMap<TString, THashSet<TString>> ProjectToGroups_;
    THashMap<std::pair<TString, TString>, TString> ResolvedGroups_;

    THashMap<TString, THashSet<TString>> ProjectToMetagroups_;
    THashMap<TString, THashSet<TString>> ProjectToHosts_;

    THashMap<TString, TQueue<TQueryRequiringMetagroupsAndHosts>> ProjectToQueriesRequiringMetagroupsAndHosts_;

    IDataSourcePtr YasmDataSource_;
};

}
