#include "cluster_name.h"
#include "deserializer.h"
#include "labels.h"
#include "metric_aggregator.h"
#include "serializer.h"
#include "yasm_common.h"

#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/stockpile_codec/metric_archive.h>
#include <solomon/libs/cpp/ts_codec/points.h>
#include <solomon/libs/cpp/yasm/constants/interval.h>
#include <solomon/libs/cpp/yasm/constants/labels.h>
#include <solomon/libs/cpp/yasm/shard_config/shard_config.h>

#include <solomon/services/dataproxy/lib/datasource/datasource.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>
#include <solomon/services/dataproxy/lib/timeseries/compressed.h>

#include <infra/yasm/common/labels/signal/signal_name.h>
#include <infra/yasm/common/labels/tags/dynamic_filter.h>
#include <infra/yasm/common/labels/tags/instance_key.h>
#include <infra/yasm/common/points/accumulators/accumulators.h>
#include <infra/yasm/stockpile_client/common/base_types.h>
#include <infra/yasm/stockpile_client/points.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;
using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

using yandex::solomon::model::MetricType;
using ::NYasm::NGroups::TMetagroupGroupsConfig;
using NTags::TInstanceKey;
using NZoom::NSignal::TSignalName;
using NZoom::NValue::TValue;
using NHistDb::NStockpile::TSeriesKey;
using namespace NZoom::NHgram;
using namespace NZoom::NAccumulators;

constexpr ui32 FIND_METRICS_LIMIT = 90'000;

class TFallbackTypeHandler: public IResultHandler<TReadManyResult> {
public:
    explicit TFallbackTypeHandler(
            IResultHandlerPtr<TReadManyResult> handler,
            absl::flat_hash_map<TString, NMonitoring::EMetricType>&& signalToType)
        : OriginalHandler_{std::move(handler)}
    {
        for (auto&& [signal, type]: signalToType) {
            if (type == NMonitoring::EMetricType::DSUMMARY) {
                switch (NHistDb::NStockpile::GetAggregationType(signal)) {
                    case NZoom::NAccumulators::EAccumulatorType::Average:
                    case NZoom::NAccumulators::EAccumulatorType::Last:
                    case NZoom::NAccumulators::EAccumulatorType::Avg:
                    case NZoom::NAccumulators::EAccumulatorType::Max:
                    case NZoom::NAccumulators::EAccumulatorType::Min:
                    case NZoom::NAccumulators::EAccumulatorType::Summ:
                    case NZoom::NAccumulators::EAccumulatorType::SummNone:
                        type = NMonitoring::EMetricType::GAUGE;
                        break;
                    case NZoom::NAccumulators::EAccumulatorType::Counter:
                        type = NMonitoring::EMetricType::COUNTER;
                        break;
                    case NZoom::NAccumulators::EAccumulatorType::Hgram:
                        Y_FAIL("Got type DSUMMARY and aggregation type Hgram");
                    case NZoom::NAccumulators::EAccumulatorType::List:
                        Y_FAIL("Got type DSUMMARY and aggregation type List");
                }
            }

            SignalToType_[signal] = type;
        }
    }

private:
    void OnSuccess(std::unique_ptr<TReadManyResult> res) override {
        absl::flat_hash_map<ui32, NMonitoring::EMetricType> signalIndToType;
        for (const auto& [signal, type]: SignalToType_) {
            ui32 signalInd = res->Strings.Put(signal);
            signalIndToType[signalInd] = type;
        }

        ui32 signalLabel = res->Strings.Put(NYasm::LABEL_SIGNAL);

        for (auto& metric: res->Metrics) {
            if (metric.Meta.Type == NMonitoring::EMetricType::UNKNOWN) {
                for (const auto& [key, value]: metric.Meta.Labels) {
                    if (key == signalLabel) {
                        metric.Meta.Type = signalIndToType[value];
                        break;
                    }
                }
            }
        }

        OriginalHandler_->OnSuccess(std::move(res));
    }

    void OnError(TString&& project, EDataSourceStatus status, TString&& message) override {
        OriginalHandler_->OnError(std::move(project), status, std::move(message));
    }

private:
    IResultHandlerPtr<TReadManyResult> OriginalHandler_;
    absl::flat_hash_map<TString, NMonitoring::EMetricType> SignalToType_;
};


void ValidateSignalName(const TReadManyQuery& query) {
    DP_ENSURE(query.Lookup, "no selectors specified in query");

    auto signalIt = query.Lookup->Selectors.Find(NYasm::LABEL_SIGNAL);
    DP_ENSURE(signalIt != query.Lookup->Selectors.end(), "no signal specified in selectors");
}

class TMetricAggregatingActor: public TActorBootstrapped<TMetricAggregatingActor> {
public:
    /**
     * Construct an actor, validating the query passed
     * @param originalQuery ReadMany handler with arguments for aggregate calculation
     * @param handler handler that will be called on result and error (only after the construction)
     * @param dataSource data source used to find and read metrics to aggregate
     * @throws TInvalidReadManyQuery if original query is invalid
     */
    TMetricAggregatingActor(
            TReadManyQuery originalQuery,
            IResultHandlerPtr<TReadManyResult> handler,
            IDataSourcePtr dataSource,
            NMonitoring::TMetricRegistry& registry,
            TSpanId traceCtx)
        : RequestResolution_(ExtractAndAdjustResolution(originalQuery))
        , MaxTimeseriesFormat_(originalQuery.MaxTimeSeriesFormat)
        , OriginalQuery_(std::move(originalQuery))
        , FindQuery_()
        , Handler_(std::move(handler))
        , DataSource_(std::move(dataSource))
        , TraceCtx_{std::move(traceCtx)}
    {
        ValidateSignalName(OriginalQuery_);

        FindTotalCount_ = registry.HistogramRate({{"sensor", "yasm.find.totalCount"}}, NMonitoring::ExponentialHistogram(22, 2, 1));
        FindSize_ = registry.HistogramRate({{"sensor", "yasm.find.resultSize"}}, NMonitoring::ExponentialHistogram(22, 2, 1));
        FindTruncated_ = registry.Rate({{"sensor", "yasm.find.truncated"}});

        ParseIncomingQuery();
    }

    void Bootstrap() {
        Become(&TMetricAggregatingActor::ProcessBaseDataSourceResponse);
        auto span = TRACING_NEW_SPAN_START(TraceCtx_, "Find metrics before dynaggr");
        DataSource_->Find(std::move(FindQuery_), MakeReplyToHandler<TFindResult>(SelfId()), std::move(span));
    }

    STATEFN(ProcessBaseDataSourceResponse) {
        try {
            switch (ev->GetTypeRewrite()) {
                hFunc(TDataSourceEvents::TSuccess<TFindResult>, HandleFoundMetrics);
                hFunc(TDataSourceEvents::TError, HandleBaseDataSourceError);
            }
        } catch (...) {
            ReportErrorAndPassAway(EDataSourceStatus::BACKEND_ERROR, CurrentExceptionMessage());
        }
    }

private:
    void ParseIncomingQuery() {
        DP_ENSURE(OriginalQuery_.Lookup, "no selectors specified in query");

        OriginalQuery_.Time = ExtractAndRoundTimeRange(OriginalQuery_, RequestResolution_);
        FillBaseQueryFields(OriginalQuery_, FindQuery_);
        FindQuery_.Selectors = OriginalQuery_.Lookup->Selectors;
        FindQuery_.Limit = FIND_METRICS_LIMIT;
        FindQuery_.FillMetricName = false;
    }

    template <typename TIt, typename TFunc>
    TString DebugOutputForMetrics(
            TIt begin,
            TIt end,
            TFunc getter,
            const NStringPool::TStringPool& metricsStringPool)
    {
        TStringBuilder sb;
        size_t metricIdx = 0;

        for (auto it = begin; it != end; ++it) { 
            const TMetric<ui32>* const metric = getter(it);
            if (metricIdx++) { sb << ", "; }

            sb << metricsStringPool[metric->Name] << '{';
            size_t labelIdx = 0;

            for (const auto& l: metric->Labels) {
                if (labelIdx++) { sb << ", "; }
                sb << metricsStringPool[l.Key] << '=' << metricsStringPool[l.Value];
            }

            sb << "}";
        }

        return sb;
    }

    void HandleFoundMetrics(TDataSourceEvents::TSuccess<TFindResult>::TPtr& ev) {
        auto& result = *(ev->Get()->Result);
        if (result.Metrics.empty()) {
            ReplyWithEmptyResult();
            return;
        }

        FindTotalCount_->Record(result.TotalCount);
        FindSize_->Record(result.Metrics.size());

        if (result.Metrics.size() != result.TotalCount) {
            Y_VERIFY(result.Metrics.size() < result.TotalCount);

            TStringBuilder message;
            message << "find result is truncated: "
                    << "found " << result.TotalCount << ", got " << result.Metrics.size()
                    << " metrics for " << OriginalQuery_.Lookup->Selectors;

            MON_WARN(Yasm, message);
            FindTruncated_->Inc();

            DP_ENSURE(false, message);
        }

        auto ignoreValue = result.Strings.Put(NSolomon::NYasm::AGGREGATED_MARKER);

        THashMap<TSeriesKey, TMetric<ui32>*> keyToMetric;
        absl::flat_hash_map<TVector<ui32>, NTags::TDynamicFilter> dynamicFilters;
        auto groupByKeys = GroupByKeys(result.Strings);
        auto resultStrings = result.Strings.Build();

        MON_TRACE(
                Yasm,
                "found " << result.Metrics.size() << " metrics: "
                         << DebugOutputForMetrics(
                                    result.Metrics.begin(),
                                    result.Metrics.end(),
                                    [](const auto& it) {
                                        return &(*it);
                                    },
                                    resultStrings));

        const auto& groupBy = OriginalQuery_.Lookup->GroupBy;

        i32 signalGroupByInd = -1;
        for (size_t i = 0; i < groupBy.size(); ++i) {
            if (groupBy[i] == NYasm::LABEL_SIGNAL) {
                signalGroupByInd = i;
            }
        }

        TVector<ui32> groupByValues(groupBy.size());
        absl::flat_hash_map<TString, NMonitoring::EMetricType> signalToType;

        for (auto& metric: result.Metrics) {
            TVector<std::pair<TStringBuf, TStringBuf>> labelKeysAndValues;
            labelKeysAndValues.reserve(metric.Labels.size());
            ui32 valueCount = 0;
            for (auto& label: metric.Labels) {
                if (resultStrings[label.Key] == NYasm::LABEL_SIGNAL) {
                    signalToType.emplace(resultStrings[label.Value], metric.Type);
                }

                if (groupByKeys.contains(label.Key) && label.Value != ignoreValue) {
                    groupByValues[groupByKeys[label.Key]] = label.Value;
                    ++valueCount;
                }
                labelKeysAndValues.emplace_back(resultStrings[label.Key], resultStrings[label.Value]);
            }

            auto seriesKey = NHistDb::NStockpile::TSeriesKey::Make(labelKeysAndValues);
            keyToMetric[seriesKey] = &metric;
            if (valueCount == groupBy.size()) {
                dynamicFilters[groupByValues].Feed(seriesKey.InstanceKey);
            }
        }

        auto signal = OriginalQuery_.Lookup->Selectors.Find(NYasm::LABEL_SIGNAL)->Pattern();
        Handler_ = MakeIntrusive<TFallbackTypeHandler>(std::move(Handler_), std::move(signalToType));

        TVector<TSeriesKey> resolvedKeys;
        int i = 0;
        for (auto& [labelValues, filter]: dynamicFilters) {
            ++i;
            auto resolvedInstances = filter.Resolve();
            TVector<TSeriesKey> resolved;
            for (auto&& instance: resolvedInstances) {
                TStringBuf signalName = signalGroupByInd == -1 ? signal : resultStrings[labelValues[signalGroupByInd]];
                resolved.push_back(TSeriesKey::Make(instance, *TSignalName::TryNew(TString{signalName})));
            }
            
            MON_TRACE(
                    Yasm,
                    "resolved " << resolved.size() << " keys from filter " << i << ": "
                                << DebugOutputForMetrics(
                                           resolved.begin(),
                                           resolved.end(),
                                           [&](const auto& it) {
                                               return keyToMetric.at(*it);
                                           },
                                           resultStrings));
            resolvedKeys.insert(resolvedKeys.end(), resolved.begin(), resolved.end());
        }

        if (resolvedKeys.empty()) {
            ReplyWithEmptyResult();
            return;
        }

        if (TActivationContext::Now() < OriginalQuery_.Deadline) {
            auto readMetricsQuery = PrepareReadQuery(resolvedKeys, keyToMetric, resultStrings);
            readMetricsQuery.Lookup = std::make_unique<TReadManyQuery::TLookup>();
            readMetricsQuery.Lookup->GroupBy = std::move(OriginalQuery_.Lookup->GroupBy);
            readMetricsQuery.Lookup->Limit = Max<ui32>();

            auto span = TRACING_NEW_SPAN_START(TraceCtx_, "Read by resolved keys");
            DataSource_->ReadMany(std::move(readMetricsQuery), Handler_, std::move(span));
            PassAway();
        } else {
            ReportErrorAndPassAway(EDataSourceStatus::BACKEND_TIMEOUT, "Deadline exceeded after metric selection step.");
        }
    }

    void HandleBaseDataSourceError(TDataSourceEvents::TError::TPtr& ev) {
        ReportErrorAndPassAway(ev->Get()->Status, std::move(ev->Get()->Message));
    }

    void ReportErrorAndPassAway(EDataSourceStatus status, TString error) {
        TRACING_SPAN_END(TraceCtx_);
        Handler_->OnError(std::move(OriginalQuery_.Project), status, std::move(error));
        PassAway();
    }

    void ReplyWithEmptyResult() {
        TRACING_SPAN_END(TraceCtx_);
        Handler_->OnSuccess(std::make_unique<TReadManyResult>());
        PassAway();
    }

    TReadManyQuery PrepareReadQuery(
        const TVector<TSeriesKey>& keys,
        const THashMap<TSeriesKey, TMetric<ui32>*>& keyToMetrics,
        const NStringPool::TStringPool& metricsStringPool)
    {
        MON_TRACE(
                Yasm,
                "resolved " << keys.size() << " keys: "
                            << DebugOutputForMetrics(
                                       keys.begin(),
                                       keys.end(),
                                       [&](const auto& it) {
                                           return keyToMetrics.at(*it);
                                       },
                                       metricsStringPool));

        TReadManyQuery result;

        FillBaseQueryFields(OriginalQuery_, result);
        result.ResolvedKeys = std::make_unique<TReadManyQuery::TResolvedKeys>();

        NStringPool::TStringPoolBuilder stringPoolBuilder;
        result.ResolvedKeys->MetricKeys.reserve(keys.size());
        for (auto& key: keys) {
            auto& sourceMetric = *keyToMetrics.at(key);
            result.ResolvedKeys->MetricKeys.emplace_back();
            auto& metric = result.ResolvedKeys->MetricKeys.back();
            metric.Name = stringPoolBuilder.Put(metricsStringPool[sourceMetric.Name]);
            metric.Labels.reserve(sourceMetric.Labels.size());
            for (auto& sourceLabel: sourceMetric.Labels) {
                auto labelKey = metricsStringPool[sourceLabel.Key];
                auto labelValue = metricsStringPool[sourceLabel.Value];
                if (labelKey != NLabels::LABEL_PROJECT && labelKey != NLabels::LABEL_SERVICE) {
                    metric.Labels.emplace_back(stringPoolBuilder.Put(labelKey), stringPoolBuilder.Put(labelValue));
                }
            }
        }

        result.ResolvedKeys->CommonLabels.emplace_back(stringPoolBuilder.Put(NLabels::LABEL_SERVICE),
                                                       stringPoolBuilder.Put(SERVICE_VALUE));
        result.MaxTimeSeriesFormat = MaxTimeseriesFormat_;
        result.ResolvedKeys->Strings = stringPoolBuilder.Build();
        result.Operations = OriginalQuery_.Operations;
        return result;
    }

    THashMap<ui32, ui32> GroupByKeys(NStringPool::TStringPoolBuilder& stringPoolBuilder) {
        THashMap<ui32, ui32> groupByKeys;
        const auto& groupBy = OriginalQuery_.Lookup->GroupBy;
        for (ui32 index = 0; index < groupBy.size(); ++index) {
            ui32 labelKey = stringPoolBuilder.Put(groupBy[index]);
            groupByKeys[labelKey] = index;
        }
        return std::move(groupByKeys);
    }

    TDuration RequestResolution_;
    ui32 MaxTimeseriesFormat_;
    TReadManyQuery OriginalQuery_;
    TFindQuery FindQuery_; // emptied after the bootstrap
    IResultHandlerPtr<TReadManyResult> Handler_;
    IDataSourcePtr DataSource_;
    TSpanId TraceCtx_;

    NMonitoring::THistogram* FindTotalCount_;
    NMonitoring::THistogram* FindSize_;
    NMonitoring::TRate* FindTruncated_;
};

} // namespace

std::unique_ptr<IActor> YasmMetricsAggregator(
        TReadManyQuery originalQuery,
        IResultHandlerPtr<TReadManyResult> handler,
        IDataSourcePtr dataSource,
        NMonitoring::TMetricRegistry& registry,
        TSpanId traceCtx)
{
    return std::make_unique<TMetricAggregatingActor>(
            std::move(originalQuery),
            std::move(handler),
            std::move(dataSource),
            registry,
            std::move(traceCtx));
}

} // namespace NSolomon::NDataProxy
