#include "selectors_collapsing_actor.h"
#include "metric_aggregator.h"
#include "yasm_common.h"

#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/yasm/constants/labels.h>

#include <solomon/services/dataproxy/lib/datasource/datasource.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;
using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

class TSelectorCollapsingActor: public TActorBootstrapped<TSelectorCollapsingActor> {
public:
    TSelectorCollapsingActor(
            TReadManyQuery originalQuery,
            IResultHandlerPtr<TReadManyResult> handler,
            IDataSourcePtr dataSource,
            NMonitoring::TMetricRegistry& registry,
            TSpanId traceCtx)
        : OriginalQuery_(std::move(originalQuery))
        , Handler_(std::move(handler))
        , DataSource_(std::move(dataSource))
        , Registry_(registry)
        , TraceCtx_{std::move(traceCtx)}
    {
    }


    void Bootstrap() {
        TLabelValuesQuery query;
        FillBaseQueryFields(OriginalQuery_, query);
        query.Limit = 1'000'000;
        query.Selectors = OriginalQuery_.Lookup->Selectors;

        DataSource_->LabelValues(std::move(query), MakeReplyToHandler<TLabelValuesResult>(SelfId()));
        Become(&TSelectorCollapsingActor::Main);
    }

    STATEFN(Main) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TDataSourceEvents::TSuccess<TLabelValuesResult>, OnSuccess);
            hFunc(TDataSourceEvents::TError, OnError);
        }
    }

    static constexpr char ActorName[] = "SelectorCollapsingActor";

private:
    void OnSuccess(TDataSourceEvents::TSuccess<TLabelValuesResult>::TPtr& ev){
        const auto& result = ev->Get()->Result;
        const auto& strings = result->Strings.Build();
        auto& selectors = OriginalQuery_.Lookup->Selectors;
        const auto& groupBy = OriginalQuery_.Lookup->GroupBy;

        for (const auto& label: result->Labels) {
            auto key = strings[label.Key];
            if (auto it = std::find(groupBy.begin(), groupBy.end(), key); it != groupBy.end()) {
                continue;
            }

            auto it = selectors.Find(key);
            if (it == selectors.end() || it->Pattern() == "*") {
                bool has_aggregation = false;
                for (auto value: label.Values) {
                    if (strings[value] == NYasm::AGGREGATED_MARKER) {
                        has_aggregation = true;
                        break;
                    }
                }

                if (has_aggregation) {
                    OriginalQuery_.Lookup->Selectors.Override(key, NYasm::AGGREGATED_MARKER);
                }
            }
        }

        TString project = OriginalQuery_.Project;
        try {
            Register(YasmMetricsAggregator(std::move(OriginalQuery_), std::move(Handler_), std::move(DataSource_), Registry_, std::move(TraceCtx_)).release());
        } catch (const TInvalidReadManyQuery& e) {
            TRACING_SPAN_END(TraceCtx_);
            Handler_->OnError(std::move(project), EDataSourceStatus::BAD_REQUEST, e.what());
        }
        PassAway();
    }

    void OnError(TDataSourceEvents::TError::TPtr&) {
        Register(YasmMetricsAggregator(std::move(OriginalQuery_), std::move(Handler_), std::move(DataSource_), Registry_, std::move(TraceCtx_)).release());
        PassAway();
    }

    TReadManyQuery OriginalQuery_;
    IResultHandlerPtr<TReadManyResult> Handler_;
    IDataSourcePtr DataSource_;
    NMonitoring::TMetricRegistry& Registry_;
    TSpanId TraceCtx_;
};

} // namespace

std::unique_ptr<IActor> SelectorCollapsingActor(
        TReadManyQuery originalQuery,
        IResultHandlerPtr<TReadManyResult> handler,
        IDataSourcePtr dataSource,
        NMonitoring::TMetricRegistry& registry,
        TSpanId traceCtx)
{
    return std::make_unique<TSelectorCollapsingActor>(
            std::move(originalQuery),
            std::move(handler),
            std::move(dataSource),
            registry,
            std::move(traceCtx));
}

} // namespace NSolomon::NDataProxy
