#include "serializer.h"

#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/libs/cpp/ts_codec/double_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/hist_log_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/hist_ts_codec.h>
#include <solomon/libs/cpp/ts_codec/summary_double_ts_codec.h>

namespace NSolomon::NDataProxy {

using NMonitoring::EMetricType;
using NZoom::NValue::EValueType;
using NZoom::NValue::TValue;
using NZoom::NValue::TValueRef;
using namespace NZoom::NHgram;
using namespace NZoom::NAccumulators;

static constexpr double LAST_INF_UGRAM_BOARD = std::numeric_limits<double>::max();

inline double StockpilePoint(double d) {
    return nextafter(d, LAST_INF_UGRAM_BOARD);
}

static EMetricType ToMetricType(EValueType valueType) {
    switch (valueType) {
        case EValueType::COUNTED_SUM: {
            return EMetricType::DSUMMARY;
        }
        case EValueType::FLOAT: {
            return EMetricType::GAUGE;
        }
        case EValueType::NORMAL_HGRAM: {
            return EMetricType::LOGHIST;
        }
        case EValueType::SMALL_HGRAM: {
            return EMetricType::LOGHIST;
        }
        case EValueType::USER_HGRAM: {
            return EMetricType::HIST;
        }
        default: {
            return EMetricType::UNKNOWN;
        }
    }
}

static EValueType FromMetricType(EMetricType type) {
    switch (type) {
        case EMetricType::LOGHIST: {
            return EValueType::NORMAL_HGRAM;
        }
        case EMetricType::HIST: {
            return EValueType::USER_HGRAM;
        }
        case EMetricType::DSUMMARY: {
            return EValueType::COUNTED_SUM;
        }
        case EMetricType::GAUGE: {
            return EValueType::FLOAT;
        }
        default: {
            return EValueType::NONE;
        }
    }
}


template<typename TEncoder, typename TValueToPoint>
static NTs::TBitBuffer SerializeWithTypes(
        const NZoom::NAccumulators::TCompactAccumulatorsArray& accumulators,
        TInstant startTime,
        TDuration step)
{
    NTs::TBitBuffer buf;
    NTs::TBitWriter writer{&buf};
    TEncoder encoder = TEncoder::Aggr(&writer);

    TInstant timestamp = startTime;
    for (auto offset: xrange(accumulators.Len())) {
        TValueRef value = accumulators.GetValue(offset);
        TValueToPoint valueToPoint(timestamp, step);

        if (value.GetType() != EValueType::NONE) {
            value.Update(valueToPoint);
        }

        encoder.EncodePoint(valueToPoint.GetPoint());
        timestamp += step;
    }

    return buf;
}

void TValueToLogHistPoint::OnStoreSmall(const TVector<double> &values, size_t zeros) {
    THolder<IHgramImpl> hgramNormal = THgramNormal::FromSmall(values, zeros);
    hgramNormal->Store(*this);
}

void TValueToLogHistPoint::OnStoreNormal(const TVector<double> &values, size_t zeros, i16 startPower) {
    Point_.ZeroCount = zeros;
    Point_.StartPower = startPower;
    Point_.Values.reserve(values.size());
    for (auto bucketWeight: values) {
        Point_.Values.emplace_back(bucketWeight);
    }
}

void TValueToLogHistPoint::OnStoreUgram(const TUgramBuckets &buckets) {
    // never called as we dont convert ugrams (stockpile hist points) to LogHist points
    if (buckets.empty()) {
        return;
    }
    THolder<IHgramImpl> normalUgram = THgramNormal::FromUgram(buckets);
    normalUgram->Store(*this);
}

void TValueToHistPoint::OnStoreSmall(const TVector<double> &values, size_t zeros) {
    THolder<IHgramImpl> hgramUgram = THgramUgram::FromSmall(values, zeros);
    hgramUgram->Store(*this);
}

void TValueToHistPoint::OnStoreNormal(const TVector<double> &values, size_t zeros, i16 startPower) {
    THolder<IHgramImpl> hgramUgram = THgramUgram::FromNormal(values, zeros, startPower);
    hgramUgram->Store(*this);
}

void TValueToHistPoint::OnStoreUgram(const TUgramBuckets &buckets) {
    if (buckets.empty()) {
        return;
    }
    Point_.Buckets.clear();
    Point_.Buckets.reserve(buckets.size());

    const TUgramBucket *prevBucket = nullptr;
    for (const auto& bucket: buckets) {
        if (prevBucket == nullptr || prevBucket->UpperBound != bucket.LowerBound) {
            Point_.Buckets.emplace_back(NTs::NValue::THistogram::TBucket{bucket.LowerBound, 0});
        }

        prevBucket = &bucket;
        auto bound =  bucket.UpperBound;
        if (bucket.IsPoint()) {
            bound = StockpilePoint(Point_.Buckets.rbegin()->UpperBound);
        }
        Point_.Buckets.emplace_back(NTs::NValue::THistogram::TBucket{bound, static_cast<ui64>(bucket.Weight)});
    }

    // NOTE(rocco66): see https://st.yandex-team.ru/GOLOVAN-6544
    if (Point_.Buckets.rbegin()->UpperBound != LAST_INF_UGRAM_BOARD) {
        Point_.Buckets.emplace_back(NTs::NValue::THistogram::TBucket{LAST_INF_UGRAM_BOARD, 0});
    }
}

void TValueToSummaryPoint::MulFloat(const double value) {
    Point_.Sum = value;
    Point_.CountValue = 1;
}

void TValueToSummaryPoint::MulCountedSum(const double sum, ui64 count) {
    Point_.Sum = sum;
    Point_.CountValue = count;
}

NTs::TBitBuffer TMetricSerializer::Serialize() {
    switch (Type_) {
        case EMetricType::LOGHIST:
            return SerializeWithTypes<NTs::TLogHistogramTsEncoder, TValueToLogHistPoint>(Accumulators_, StartTime_, Step_);

        case EMetricType::HIST:
            return SerializeWithTypes<NTs::THistogramTsEncoder, TValueToHistPoint>(Accumulators_, StartTime_, Step_);

        case EMetricType::DSUMMARY:
            return SerializeWithTypes<NTs::TSummaryDoubleTsEncoder, TValueToSummaryPoint>(Accumulators_, StartTime_, Step_);

        case EMetricType::GAUGE:
            return SerializeWithTypes<NTs::TDoubleTsEncoder, TValueToGaugePoint>(Accumulators_, StartTime_, Step_);

        case EMetricType::UNKNOWN: {
            Y_ENSURE(Accumulators_.Len() == 0, "Unable to serialize nonempty metrics with UNKNOWN type");
            return NTs::TBitBuffer{};
        }

        default:
            ythrow yexception() << "Unable to serialize metrics of type: " << Type_;
    }
}

void TValueTypeDetector::OnValue(TValueRef value) {
    const EValueType given = FromMetricType(ToMetricType(value.GetType()));
    if (given == EValueType::NONE) {
        return;
    }
    if (ValueType_ == EValueType::NONE) {
        ValueType_ = given;
    } else if (given != ValueType_) {
        // emulate standard hgram evolution from normal to ugram
        if (ValueType_ == EValueType::NORMAL_HGRAM && given == EValueType::USER_HGRAM) {
            ValueType_ = EValueType::USER_HGRAM;
        } else if (ValueType_ == EValueType::USER_HGRAM && given == EValueType::NORMAL_HGRAM) {
            ValueType_ = EValueType::USER_HGRAM;
        } else {
            ythrow yexception() << "can't convert " << given << " to " << ValueType_;
        }
    }
}

EMetricType TValueTypeDetector::GetType() const noexcept {
    return ToMetricType(ValueType_);
}

size_t BucketsSizeAfterSerializing(const TUgramBuckets &buckets) {
    if (buckets.empty()) {
        return 0;
    }
    size_t res = 0;
    const TUgramBucket *prevBucket = nullptr;
    for (const auto& bucket: buckets) {
        if (prevBucket == nullptr || prevBucket->UpperBound != bucket.LowerBound) {
            res++;
        }
        res++;
        prevBucket = &bucket;
    }

    if (prevBucket->UpperBound != LAST_INF_UGRAM_BOARD) {
        res++;
    }
    return res;
}

} // namespace NSolomon::NDataProxy
