#include <solomon/services/dataproxy/lib/datasource/tsdb/tsdb_timeseries.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/dynamic_aggregation.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <infra/yasm/common/points/value/impl.h>
#include <infra/yasm/interfaces/internal/history_api.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace ::NYasm::NInterfaces::NInternal;
using namespace NSolomon::NDataProxy;
using namespace NSolomon::NYasm;
using namespace NSolomon;
using namespace NZoom::NProtobuf;
using namespace NZoom::NValue;
using namespace std::literals::chrono_literals;
using namespace testing;

void AddLabels(
        TLabels<ui32>& labels,
        NStringPool::TStringPoolBuilder& sp,
        const TVector<std::pair<TStringBuf, TStringBuf>>& list)
{
    for (auto [k, v]: list) {
        labels.emplace_back(sp.Put(k), sp.Put(v));
    }
}

std::unique_ptr<TReadManyResult> CreateReadManyResponse(TVector<std::pair<TInstant, TVector<TValue>>>& tsAndValues) {
    auto response = std::make_unique<TReadManyResult>();
    auto& sp = response->Strings;

    TString signalName = "counter-instance_tmmv";

    auto addMetric = [&](TInstant startTime, TVector<NZoom::NValue::TValue>&& values) {
        auto& metric = response->Metrics.emplace_back();
        metric.Meta.Name = sp.Put(signalName);
        AddLabels(metric.Meta.Labels, sp, {{"group", "SAS.000"}, {"signal", signalName}});

        THistoryRequest req{.RequestKey = {"itype=smth"sv}, .SignalName = {signalName}};
        THistoryResponse hr{req, startTime, std::move(values), THistoryAggregatedSeries_EStatusCode_OK};

        metric.TimeSeries = std::make_unique<TTsdbTimeSeries>(
                std::move(hr),
                yandex::solomon::model::MetricType::DSUMMARY,
                NHistDb::NStockpile::GetAggregationType(signalName),
                TDuration::Seconds(5));
        metric.Meta.Type = metric.TimeSeries->Type();
    };

    for (auto&& [ts, values]: tsAndValues) {
        addMetric(ts, std::move(values));
    }

    return response;
}

class TMockDataSource: public IDataSource {
public:
    TMockDataSource(TVector<std::pair<TInstant, TVector<TValue>>>&& tsAndValues)
        : TsAndValues_{std::move(tsAndValues)}
    {}

    bool CanHandle(const TQuery&) const override {
        return true;
    }

    void Find(TFindQuery, IResultHandlerPtr<TFindResult>, ::NSolomon::NTracing::TSpanId) const override {}
    void ResolveOne(TResolveOneQuery, IResultHandlerPtr<TResolveOneResult>, ::NSolomon::NTracing::TSpanId) const override {}
    void ResolveMany(TResolveManyQuery, IResultHandlerPtr<TResolveManyResult>, ::NSolomon::NTracing::TSpanId) const override {}
    void MetricNames(TMetricNamesQuery, IResultHandlerPtr<TMetricNamesResult>, ::NSolomon::NTracing::TSpanId) const override {}
    void LabelKeys(TLabelKeysQuery, IResultHandlerPtr<TLabelKeysResult>, ::NSolomon::NTracing::TSpanId) const override {}
    void LabelValues(TLabelValuesQuery, IResultHandlerPtr<TLabelValuesResult>, ::NSolomon::NTracing::TSpanId) const override {}
    void UniqueLabels(TUniqueLabelsQuery, IResultHandlerPtr<TUniqueLabelsResult>, ::NSolomon::NTracing::TSpanId) const override {}
    void ReadOne(TReadOneQuery, IResultHandlerPtr<TReadOneResult>, ::NSolomon::NTracing::TSpanId) const override {}
    void WaitUntilInitialized() const override {}

    void ReadMany(TReadManyQuery, IResultHandlerPtr<TReadManyResult> handler, ::NSolomon::NTracing::TSpanId) const override {
        handler->OnSuccess(CreateReadManyResponse(TsAndValues_));
    }

private:
    mutable TVector<std::pair<TInstant, TVector<TValue>>> TsAndValues_;
};

class THandler: public IResultHandler<TReadManyResult> {
public:
    void OnSuccess(std::unique_ptr<TReadManyResult> result) override {
        Result = std::move(result);
    }

    void OnError(TString&& project, EDataSourceStatus status, TString&& message) override {
        Cerr << "OnError() << " << "project=" << project << "; status=" << status << "; msg=" << message << Endl;
    }

public:
    std::unique_ptr<TReadManyResult> Result;
};

TEST(Serialization, OfEmptyGaugePoints) {
    auto actorRuntime = TTestActorRuntime::CreateInited();
    actorRuntime->AdvanceCurrentTime(TDuration::Hours(1));

    auto alignedNow = actorRuntime->GetCurrentTime();
    alignedNow = TInstant::Seconds(alignedNow.Seconds() - (alignedNow.Seconds() % 5));
    auto startTs = alignedNow - TDuration::Seconds(30);

    // TValue's copy/move constructors are deleted, so we have to create elements in place
    TVector<TValue> v1, v2, v3;
    TVector<double> ex/*pected*/;
                              v1.emplace_back();    v1.emplace_back(1.0); v1.emplace_back(3.0);  v1.emplace_back(3.0);
    v2.emplace_back(123.456); v2.emplace_back();    v2.emplace_back();    v2.emplace_back(4.0);  v2.emplace_back(4.0);
    v3.emplace_back(234.567); v3.emplace_back();    v3.emplace_back(2.0); v3.emplace_back(5.0);
    // ----------------------------------------------------------------------------------------------------------------
    ex.emplace_back(358.023); ex.emplace_back(NAN); ex.emplace_back(3.0); ex.emplace_back(12.0); //; NAN(SOLOMON-8378)

    TVector<std::pair<TInstant, TVector<TValue>>> tsAndValues;
    tsAndValues.emplace_back(startTs + 5s, std::move(v1));
    tsAndValues.emplace_back(startTs     , std::move(v2));
    tsAndValues.emplace_back(startTs     , std::move(v3));

    TIntrusivePtr<IDataSource> mockDataSource = MakeIntrusive<TMockDataSource>(std::move(tsAndValues));
    auto yasmDynAggrDataSource = YasmDynAggrDataSource(mockDataSource, actorRuntime->GetActorSystem(0));

    TReadManyQuery query;
    query.ResolvedKeys = std::make_unique<TReadManyQuery::TResolvedKeys>();
    query.Lookup = std::make_unique<TReadManyQuery::TLookup>();
    query.Time.From = startTs;
    query.Time.To = query.Time.From + TDuration::Minutes(1);
    query.Operations.emplace_back().mutable_downsampling()->set_grid_millis(5'000);

    auto handler = MakeIntrusive<THandler>();
    yasmDynAggrDataSource->ReadMany(std::move(query), handler);

    auto& result = handler->Result;
    ASSERT_TRUE(result);
    ASSERT_EQ(result->Metrics.size(), 1u);

    auto& ts = result->Metrics[0].TimeSeries;
    ASSERT_GE(ts->PointCount(), ex.size());

    auto it = result->Metrics[0].TimeSeries->Iterator();
    NTs::TVariantPoint point;
    size_t idx{0};

    while (it->Next(&point)) {
        SCOPED_TRACE("idx=" + ToString(idx));

        auto& var = point.Value;
        ASSERT_TRUE(std::holds_alternative<NTs::NValue::TDouble>(var));
        auto& value = std::get<NTs::NValue::TDouble>(var);

        if (idx > ex.size() - 1 || std::isnan(ex[idx])) {
            ASSERT_TRUE(std::isnan(value.Num)) << "expected NaN, but got: " << value.Num;
        } else {
            ASSERT_EQ(value.Num, ex[idx]);
        }

        ++idx;
    }
}
