#include <solomon/services/dataproxy/lib/datasource/ut_utils.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/datasource_actor.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/dynamic_aggregation.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/events.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/metric_aggregator.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/serializer.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/yasm_common.h>
#include <solomon/services/dataproxy/lib/timeseries/compressed.h>
#include <solomon/services/dataproxy/lib/timeseries/type_to_columns.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>
#include <solomon/libs/cpp/stockpile_codec/format.h>
#include <solomon/libs/cpp/yasm/constants/interval.h>
#include <solomon/libs/cpp/yasm/constants/labels.h>

#include <library/cpp/monlib/metrics/histogram_snapshot.h>
#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NSolomon::NTracing;
using namespace NSolomon::NYasm;
using namespace NDataProxy;
using namespace testing;

namespace {

TMap<TStringBuf, TStringBuf> SelectorsKeyToPatternMap(const TSelectors& selectors) {
    TMap<TStringBuf, TStringBuf> result;
    for (const auto& selector: selectors) {
        result[selector.Key()] = selector.Pattern();
    }
    return result;
}

MATCHER_P(FindQueryPartiallyEq, query, "") {
    return query.Project == arg.Project &&
           query.Time.From == arg.Time.From &&
           query.Time.To == arg.Time.To &&
           query.Deadline == arg.Deadline &&
           query.ForceReplicaRead == arg.ForceReplicaRead &&
           SelectorsKeyToPatternMap(query.Selectors) == SelectorsKeyToPatternMap(arg.Selectors);
}

MATCHER_P2(ReadQueryWithBaseFieldsAndMetricsCount, baseQuery, metricsCount, "") {
    return baseQuery.Project == arg.Project &&
           baseQuery.Time.From == arg.Time.From &&
           baseQuery.Time.To == arg.Time.To &&
           baseQuery.Deadline == arg.Deadline &&
           baseQuery.ForceReplicaRead == arg.ForceReplicaRead &&
           static_cast<size_t>(metricsCount) == arg.ResolvedKeys->MetricKeys.size();
}

MATCHER_P2(ReadQueryServiceEq, query, expectedService, "") {
    const auto& service = arg.ResolvedKeys->Strings[arg.ResolvedKeys->CommonLabels[0].Value];
    return query.Project == arg.Project &&
           query.Time.From == arg.Time.From &&
           query.Time.To == arg.Time.To &&
           query.Deadline == arg.Deadline &&
           query.ForceReplicaRead == arg.ForceReplicaRead &&
           service == expectedService;
}

} // namespace

class TMetricAggregatorTest: public Test {
protected:
    void SetUp() override {
        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, true);
        ActorRuntime_->WaitForBootstrap();
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

    template <typename TQueryType>
    TQueryType FillQuery(TStringBuf project, TInstant from, TInstant to, TMaybe<TInstant> deadline = Nothing()) const {
        TQueryType result;
        result.Project = project;
        result.Time.From = from;
        result.Time.To = to;
        result.Deadline = deadline.Defined() ? deadline.GetRef() : ActorRuntime_->GetCurrentTime() + TDuration::Seconds(2);
        return result;
    }

    template <typename TQueryType>
    TQueryType FillQuery(TStringBuf project, TInstant from, TInstant to, TStringBuf selectors, TMaybe<TInstant> deadline = Nothing()) const {
        return FillQueryImpl<TQueryType>(project, from, to, selectors, deadline);
    }

    template <typename TQueryType>
    TQueryType FillQueryImpl(TStringBuf project, TInstant from, TInstant to, TStringBuf selectors, TMaybe<TInstant> deadline) const {
        TQueryType result = FillQuery<TQueryType>(project, from, to, deadline);
        result.Selectors = ParseSelectors(selectors);
        return result;
    }

    template <>
    TReadManyQuery FillQueryImpl<TReadManyQuery>(TStringBuf project, TInstant from, TInstant to, TStringBuf selectors, TMaybe<TInstant> deadline) const {
        TReadManyQuery result = FillQuery<TReadManyQuery>(project, from, to, deadline);
        result.Lookup = std::make_unique<TReadManyQuery::TLookup>();
        result.Lookup->Selectors = ParseSelectors(selectors);
        return result;
    }

    template <typename TQueryType>
    TQueryType AddQuerySelectors(const TQuery& query, TStringBuf newSelectors) const {
        return FillQuery<TQueryType>(query.Project, query.Time.From, query.Time.To, newSelectors, query.Deadline);
    }

    static void ExpectNoCalls(const TIntrusivePtr<TReadManyResultHandlerMock>& handlerMock, const TIntrusivePtr<TBaseDataSourceMock>& dataSourceMock) {
        EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(0);
        EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);
        EXPECT_CALL(*dataSourceMock, Find(_, _, _)).Times(0);
        EXPECT_CALL(*dataSourceMock, ReadMany(_, _, _)).Times(0);
    }

    static void ExpectSuccessfulCalls(const TIntrusivePtr<TReadManyResultHandlerMock>& handlerMock, const TIntrusivePtr<TBaseDataSourceMock>& dataSourceMock,
                                      TFindQuery expectedQuery, std::unique_ptr<TFindResult>& findResult,
                                      TQuery expectedReadManyQueryBase, size_t metricsCount,
                                      std::unique_ptr<TReadManyResult>& datasourceReadManyResult, std::unique_ptr<TReadManyResult>& expectedAggregatedResult) {
        EXPECT_CALL(*dataSourceMock, Find(FindQueryPartiallyEq(std::move(expectedQuery)), _, _)).WillOnce(
            Invoke([&](auto, auto handler, auto) {
                handler->OnSuccess(std::move(findResult));
            })
        );
        EXPECT_CALL(*dataSourceMock, ReadMany(ReadQueryWithBaseFieldsAndMetricsCount(std::move(expectedReadManyQueryBase), metricsCount), _, _)).WillOnce(
            Invoke([&](auto, auto handler, auto) {
                handler->OnSuccess(std::move(datasourceReadManyResult));
            })
        );

        EXPECT_CALL(*handlerMock, OnSuccess(ReadManyResultEq(expectedAggregatedResult.get()))).Times(1);
    }

    void CreateActorAndDispatchEvents(TIntrusivePtr<TReadManyResultHandlerMock> handlerMock, TIntrusivePtr<TBaseDataSourceMock> dataSourceMock, TReadManyQuery&& query) {
        ActorRuntime_->SetLogPriority(ELogComponent::Yasm, NActors::NLog::PRI_TRACE);

        auto dynAggrDs = YasmDynAggrDataSource(std::move(dataSourceMock), ActorRuntime_->GetActorSystem(0));
        auto actor = YasmMetricsAggregator(std::move(query), std::move(handlerMock), std::move(dynAggrDs), Registry_);
        auto actorId = ActorRuntime_->Register(actor.release());

        ActorRuntime_->WaitForBootstrap();
        while (ActorRuntime_->DispatchEvents()) {
        }

        // aggregator actor must die after calling the handler
        EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
    }

    THolder<TTestActorRuntime> ActorRuntime_;
    NMonitoring::TMetricRegistry Registry_;
};

TEST_F(TMetricAggregatorTest, ValidatesRequestBySelectors) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    // expect no calls to handler and datasource on invalid query
    ExpectNoCalls(handlerMock, dataSourceMock);

    auto query = FillQuery<TReadManyQuery>("yasm_yasmagentmetrics", TInstant::Zero(),  TInstant::Zero() + TDuration::Hours(2));
    query.ResolvedKeys = std::make_unique<TReadManyQuery::TResolvedKeys>();
    NStringPool::TStringPoolBuilder strPoolBuilder;
    query.ResolvedKeys->MetricKeys.emplace_back();
    auto& queryMetric = query.ResolvedKeys->MetricKeys.back();
    queryMetric.Name = strPoolBuilder.Put("");
    queryMetric.Labels.emplace_back(strPoolBuilder.Put("service"), strPoolBuilder.Put("yasm"));
    queryMetric.Labels.emplace_back(strPoolBuilder.Put("signal"), strPoolBuilder.Put("counter-instance_tmmv"));

    ASSERT_THROW(YasmMetricsAggregator(std::move(query), handlerMock, dataSourceMock, Registry_),
        TInvalidReadManyQuery);
}

TEST_F(TMetricAggregatorTest, ValidatesRequestTimeInterval) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    // expect no calls to handler and datasource on invalid query
    ExpectNoCalls(handlerMock, dataSourceMock);

    auto query2 = FillQuery<TReadManyQuery>("yasm_yasmagentmetrics", TInstant::Zero() + TDuration::Hours(2),  TInstant::Zero(),
                                            R"({service="yasm", cluster = "group_*", signal="counter-instance_ammx", group="MYT.000|SAS.000"})");
    ASSERT_THROW(CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query2)),
        TInvalidReadManyQuery);
}

TEST_F(TMetricAggregatorTest, RequestTimeIntervalAlignesWithGrid) {
    ui64 unalignedBeg = 1854;
    ui64 grid = 2 * YASM_INTERVAL.Seconds();
    ui64 alignedBeg = unalignedBeg - unalignedBeg % grid;
    ui64 unalignedEnd = alignedBeg + 6 * grid - 2;
    ui64 alignedEnd = (unalignedEnd + grid - 1) / grid * grid;

    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics", TInstant::Seconds(alignedBeg), TInstant::Seconds(alignedEnd));
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000|SAS.000"})");
    query.Time.From = TInstant::Seconds(unalignedBeg);
    query.Time.To = TInstant::Seconds(unalignedEnd);
    query.Lookup->Limit = 100;

    yandex::solomon::math::Operation downsampling;
    downsampling.mutable_downsampling()->Setfill_option(yandex::solomon::math::OperationDownsampling_FillOption_NONE);
    downsampling.mutable_downsampling()->Setgrid_millis(grid * 1000);
    query.Operations.push_back(downsampling);
    auto expectedFindQuery = AddQuerySelectors<TFindQuery>(query,
                                                           R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000|SAS.000"})");
    expectedFindQuery.Time.From = TInstant::Seconds(alignedBeg);
    expectedFindQuery.Time.To = TInstant::Seconds(alignedEnd);
    expectedFindQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "counter-instance_tmmv"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                }, NMonitoring::EMetricType::DSUMMARY
            }
        });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<double>(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "counter-instance_tmmv"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                },
                {1.8, 1.4, 7, 8, 0.1, 4},
                NMonitoring::EMetricType::DSUMMARY
            }
        }, NZoom::NAccumulators::EAccumulatorType::Summ, TInstant::Seconds(alignedBeg), TDuration::Seconds(grid));
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<double, true>(
        {
            {
                {
                    {"signal", "counter-instance_tmmv"},
                },
                {1.8, 1.4, 7, 8, 0.1, 4},
                NMonitoring::EMetricType::GAUGE
            }
        }, NZoom::NAccumulators::EAccumulatorType::Summ, TInstant::Seconds(alignedBeg), TDuration::Seconds(grid));
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedFindQuery, findResult, expectedReadManyQueryBase, 1,
                          datasourceReadManyResult, expectedAggregatedResult);

    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, ExpandsMetagroupsAndPropagatesFindError) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto query = FillQuery<TReadManyQuery>("yasm_yasmagentmetrics", TInstant::Zero(),  TInstant::Zero() + TDuration::Hours(2),
                                           R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000|SAS.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    EXPECT_CALL(*dataSourceMock, Find(FindQueryPartiallyEq(expectedQuery), _, _)).WillOnce(Invoke(
       [&](auto, auto handler, auto) {
           handler->OnError("yasm_yasmagentmetrics", EDataSourceStatus::BACKEND_ERROR, "Cannot find anything");
       }
    ));
    EXPECT_CALL(*dataSourceMock, ReadMany(_, _, _)).Times(0);
    EXPECT_CALL(*handlerMock, OnError(_, EDataSourceStatus::BACKEND_ERROR, _)).Times(1);

    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, FiltersMetricsAndPropagatesReadManyError) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics", TInstant::Zero(), TInstant::Zero() + TDuration::Hours(2));
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000|SAS.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels({
        {
            {
                {"project", "yasm_yasmagentmetrics"},
                {"service", "yasm"},
                {"cluster", "group_0"},
                {"signal", "counter-instance_tmmv"},
                {"group", "MYT.000"},
                {"ctype", "prod"}
            }, NMonitoring::EMetricType::DSUMMARY
        },
        {
            {
                {"project", "yasm_yasmagentmetrics"},
                {"cluster", "group_0"},
                {"signal", "counter-instance_tmmv"},
                {"group", "MYT.000"},
                {"ctype", "test"}
            }, NMonitoring::EMetricType::DSUMMARY
        },
        {
            {
                {"project", "yasm_yasmagentmetrics"},
                {"cluster", "group_0"},
                {"signal", "counter-instance_tmmv"},
                {"group", "MYT.000"},
                {"ctype", NSolomon::NYasm::AGGREGATED_MARKER}
            }, NMonitoring::EMetricType::DSUMMARY
        },
        {
            {
                {"project", "yasm_yasmagentmetrics"},
                {"cluster", "group_0"},
                {"signal", "counter-instance_tmmv"},
                {"group", "SAS.000"},
                {"ctype", "test"}
            }, NMonitoring::EMetricType::DSUMMARY
        },
        {
            {
                {"project", "yasm_yasmagentmetrics"},
                {"cluster", "group_0"},
                {"signal", "counter-instance_tmmv"},
                {"group", "SAS.000"},
                {"ctype", NSolomon::NYasm::AGGREGATED_MARKER}
            }, NMonitoring::EMetricType::DSUMMARY
        }
    });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    EXPECT_CALL(*dataSourceMock, Find(FindQueryPartiallyEq(expectedQuery), _, _)).WillOnce(
        Invoke([&](auto, auto handler, auto) {
            handler->OnSuccess(std::move(findResult));
        })
    );
    EXPECT_CALL(*dataSourceMock, ReadMany(ReadQueryWithBaseFieldsAndMetricsCount(expectedReadManyQueryBase, 2), _, _)).WillOnce(
        Invoke([&](auto, auto handler, auto) {
            handler->OnError("yasm_yasmagentmetrics", EDataSourceStatus::BACKEND_ERROR, "Cannot read anything");
        })
    );
    EXPECT_CALL(*handlerMock, OnError(_, EDataSourceStatus::BACKEND_ERROR, _)).Times(1);

    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, TimesOutOnLongFind) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    const auto timeout = TDuration::Seconds(2);
    auto query = FillQuery<TReadManyQuery>("yasm_yasmagentmetrics", TInstant::Zero(),  TInstant::Zero() + TDuration::Hours(2),
                                           R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000|SAS.000"})",
                                           ActorRuntime_->GetCurrentTime() + timeout);
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels({
        {
            {
                {"project",
                 "yasm_yasmagentmetrics"},
                {"service", "yasm"},
                {"cluster", "group_0"},
                {"signal",
                 "counter-instance_tmmv"},
                {"group", "MYT.000"},
                {"ctype", "prod"}
            }, NMonitoring::EMetricType::DSUMMARY
        },
        {
            {
                {"project", "yasm_yasmagentmetrics"},
                {"cluster", "group_0"},
                {"signal", "counter-instance_tmmv"},
                {"group", "MYT.000"},
                {"ctype", "test"}
            }, NMonitoring::EMetricType::DSUMMARY
        },
        {
            {
                {"project", "yasm_yasmagentmetrics"},
                {"cluster", "group_0"},
                {"signal", "counter-instance_tmmv"},
                {"group", "MYT.000"},
                {"ctype", NSolomon::NYasm::AGGREGATED_MARKER}
            }, NMonitoring::EMetricType::DSUMMARY
        },
        {
            {
                {"project", "yasm_yasmagentmetrics"},
                {"cluster", "group_0"},
                {"signal", "counter-instance_tmmv"},
                {"group", "SAS.000"},
                {"ctype", "test"}
            }, NMonitoring::EMetricType::DSUMMARY
        },
        {
            {
                {"project", "yasm_yasmagentmetrics"},
                {"cluster", "group_0"},
                {"signal", "counter-instance_tmmv"},
                {"group", "SAS.000"},
                {"ctype", NSolomon::NYasm::AGGREGATED_MARKER}
            }, NMonitoring::EMetricType::DSUMMARY
        }
    });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    EXPECT_CALL(*dataSourceMock, Find(FindQueryPartiallyEq(expectedQuery), _, _)).WillOnce(
        Invoke([&](auto, auto handler, auto) {
            ActorRuntime_->AdvanceCurrentTime(timeout + TDuration::Seconds(1));
            handler->OnSuccess(std::move(findResult));
        })
    );
    EXPECT_CALL(*dataSourceMock, ReadMany(_, _, _)).Times(0);
    EXPECT_CALL(*handlerMock, OnError(_, EDataSourceStatus::BACKEND_TIMEOUT, _)).Times(1);

    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, ReadsEmpty) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
            TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 7 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels({
        {
        {
                {"project", "yasm_yasmagentmetrics"},
                {"service", "yasm"},
                {"cluster", "group_0"},
                {"signal", "counter-instance_tmmv"},
                {"group", "MYT.000"},
                {"ctype", "prod"}
            }, NMonitoring::EMetricType::DSUMMARY
        }
    });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<double>(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "counter-instance_tmmv"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                },
                {},
                NMonitoring::EMetricType::DSUMMARY
            }
        }, NZoom::NAccumulators::EAccumulatorType::Summ);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<double, true>(
        {
            {
                {
                    {"signal", "counter-instance_tmmv"},
                },
                {},
                NMonitoring::EMetricType::GAUGE // fallbacks to a type according to aggr rules and info from Metabase
            }
        }, NZoom::NAccumulators::EAccumulatorType::Summ);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, expectedReadManyQueryBase, 1,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, ReadsTimeseries) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
            TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 7 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels({
         {
             {
                 {"project", "yasm_yasmagentmetrics"},
                 {"service", "yasm"},
                 {"cluster", "group_0"},
                 {"signal", "counter-instance_tmmv"},
                 {"group", "MYT.000"},
                 {"ctype", "prod"}
             }, NMonitoring::EMetricType::DSUMMARY
         }
     });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<double>(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "counter-instance_tmmv"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                },
                {1.8, 1.4, 7, 8, 0.1, 4},
                NMonitoring::EMetricType::DSUMMARY
            }
        }, NZoom::NAccumulators::EAccumulatorType::Summ);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<double, true>(
        {
            {
                {
                    {"signal", "counter-instance_tmmv"},
                },
                {1.8, 1.4, 7, 8, 0.1, 4},
                NMonitoring::EMetricType::GAUGE
            }
        }, NZoom::NAccumulators::EAccumulatorType::Summ);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, expectedReadManyQueryBase, 1,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, ReadsDsummary) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
            TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 7 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000|SAS.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "counter-instance_tmmv"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                }, NMonitoring::EMetricType::DSUMMARY
            }
        });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<double>(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "counter-instance_tmmv"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                },
                {1.8, 1.4, 7, 8, 0.1, 4},
                NMonitoring::EMetricType::DSUMMARY
            }
        }, NZoom::NAccumulators::EAccumulatorType::Summ);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<double, true>(
        {
            {
                {
                    {"signal", "counter-instance_tmmv"},
                },
                {1.8, 1.4, 7, 8, 0.1, 4},
                NMonitoring::EMetricType::GAUGE
            }
        }, NZoom::NAccumulators::EAccumulatorType::Summ);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, expectedReadManyQueryBase, 1,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, MergesTimeseries) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
            TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 7 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000|SAS.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "counter-instance_tmmv"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                }, NMonitoring::EMetricType::DSUMMARY
            },
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "counter-instance_tmmv"},
                    {"group", "SAS.000"},
                    {"ctype", "prestable"}
                }, NMonitoring::EMetricType::DSUMMARY
            }
        });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<double>(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "counter-instance_tmmv"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                },
                {1.8, 1.4, 7, 8, 0.1, 4},
                NMonitoring::EMetricType::DSUMMARY
            },
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "counter-instance_tmmv"},
                    {"group", "SAS.000"},
                    {"ctype", "prestable"}
                },
                {0, 1, 2, 3, 4, 5},
                NMonitoring::EMetricType::DSUMMARY
            }
        }, NZoom::NAccumulators::EAccumulatorType::Summ);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<double, true>(
        {
            {
                {
                    {"signal", "counter-instance_tmmv"},
                },
                {1.8, 2.4, 9, 11, 4.1, 9},
                NMonitoring::EMetricType::GAUGE
            }
        }, NZoom::NAccumulators::EAccumulatorType::Summ);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, expectedReadManyQueryBase, 2,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, MergesCountedSumm) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
            TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 2 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="cpu_usage_avvv", group="MYT.000|SAS.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="cpu_usage_avvv", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "cpu_usage_avvv"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                }, NMonitoring::EMetricType::DSUMMARY
            },
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "cpu_usage_avvv"},
                    {"group", "SAS.000"},
                    {"ctype", "prestable"}
                }, NMonitoring::EMetricType::DSUMMARY
            }
        });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    TLabelsList labels1 = {
        {"project", "yasm_yasmagentmetrics"},
        {"service", "yasm"},
        {"cluster", "group_0"},
        {"signal", "cpu_usage_avvv"},
        {"group", "MYT.000"},
        {"ctype", "prod"}
    };
    TVector<NZoom::NValue::TValue> vec1;
    vec1.emplace_back(NZoom::NValue::TValue(2, 45));

    TLabelsList labels2 = {
        {"project", "yasm_yasmagentmetrics"},
        {"service", "yasm"},
        {"cluster", "group_0"},
        {"signal", "cpu_usage_avvv"},
        {"group", "SAS.000"},
        {"ctype", "prestable"}
    };
    TVector<NZoom::NValue::TValue> vec2;
    vec2.emplace_back(NZoom::NValue::TValue(8, 10));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NValue::TValue>, NMonitoring::EMetricType>> labelsWithValues;
    labelsWithValues.emplace_back(labels1, std::move(vec1), NMonitoring::EMetricType::DSUMMARY);
    labelsWithValues.emplace_back(labels2, std::move(vec2), NMonitoring::EMetricType::DSUMMARY);

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<NZoom::NValue::TValue>(
        std::move(labelsWithValues), NZoom::NAccumulators::EAccumulatorType::Average);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    TLabelsList expectedLabels = {{"signal", "cpu_usage_avvv"}};
    TVector<NZoom::NValue::TValue> expectedValues;
    expectedValues.emplace_back(NZoom::NValue::TValue(10, 55));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NValue::TValue>, NMonitoring::EMetricType>> expectedLabelsWithValues;
    expectedLabelsWithValues.emplace_back(expectedLabels, std::move(expectedValues), NMonitoring::EMetricType::DSUMMARY);
    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<NZoom::NValue::TValue, true> (
        std::move(expectedLabelsWithValues), NZoom::NAccumulators::EAccumulatorType::Average);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, expectedReadManyQueryBase, 2,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, MergesLogHist) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto query = FillQuery<TReadManyQuery>("yasm_yasmagentmetrics",
            TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 2 * YASM_INTERVAL,
            R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
            R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "cpu_usage_hgram"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                }, NMonitoring::EMetricType::LOGHIST
            },
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "cpu_usage_hgram"},
                    {"group", "SAS.000"},
                    {"ctype", "prestable"}
                }, NMonitoring::EMetricType::LOGHIST
            }
        });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    TLabelsList labels1 = {
        {"project", "yasm_yasmagentmetrics"},
        {"service", "yasm"},
        {"cluster", "group_0"},
        {"signal", "cpu_usage_hgram"},
        {"group", "MYT.000"},
        {"ctype", "prod"}
    };
    TVector<NZoom::NHgram::THgram> vec1;
    vec1.emplace_back(NZoom::NHgram::THgram::Normal({1, 5, 6}, 5, 0));

    TLabelsList labels2 = {
        {"project", "yasm_yasmagentmetrics"},
        {"service", "yasm"},
        {"cluster", "group_0"},
        {"signal", "cpu_usage_hgram"},
        {"group", "SAS.000"},
        {"ctype", "prestable"}
    };
    TVector<NZoom::NHgram::THgram> vec2;
    vec2.emplace_back(NZoom::NHgram::THgram::Normal({2, 6, 10}, 2, 1));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> labelsWithValues;
    labelsWithValues.emplace_back(labels1, std::move(vec1), NMonitoring::EMetricType::LOGHIST);
    labelsWithValues.emplace_back(labels2, std::move(vec2), NMonitoring::EMetricType::LOGHIST);

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram>(
        std::move(labelsWithValues), NZoom::NAccumulators::EAccumulatorType::Hgram);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    TLabelsList expectedLabels = {{"signal", "cpu_usage_hgram"}};
    TVector<NZoom::NHgram::THgram> expectedValues;
    expectedValues.emplace_back(NZoom::NHgram::THgram::Normal({1, 7, 12, 10}, 7, 0));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> expectedLabelsWithValues;
    expectedLabelsWithValues.emplace_back(expectedLabels, std::move(expectedValues), NMonitoring::EMetricType::LOGHIST);
    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram, true> (
        std::move(expectedLabelsWithValues), NZoom::NAccumulators::EAccumulatorType::Hgram);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, query, 2,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, NotDuplicatesInfBucketInHist) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
            TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 2 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "cpu_usage_hgram"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                }, NMonitoring::EMetricType::HIST
            }
        });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    TLabelsList labels1 = {
        {"project", "yasm_yasmagentmetrics"},
        {"service", "yasm"},
        {"cluster", "group_0"},
        {"signal", "cpu_usage_hgram"},
        {"group", "MYT.000"},
        {"ctype", "prod"}
    };
    TVector<NZoom::NHgram::THgram> vec1;
    vec1.emplace_back(NZoom::NHgram::THgram::Ugram({{0, 1, 2}, {1, std::numeric_limits<double>::max(), 0}}));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> labelsWithValues;
    labelsWithValues.emplace_back(labels1, std::move(vec1), NMonitoring::EMetricType::HIST);

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram>(
        std::move(labelsWithValues), NZoom::NAccumulators::EAccumulatorType::Hgram);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    TLabelsList expectedLabels = {{"signal", "cpu_usage_hgram"}};
    TVector<NZoom::NHgram::THgram> expectedValues;
    expectedValues.emplace_back(NZoom::NHgram::THgram::Ugram({{0, 1, 2}}));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> expectedLabelsWithValues;
    expectedLabelsWithValues.emplace_back(expectedLabels, std::move(expectedValues), NMonitoring::EMetricType::HIST);
    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram, true> (
        std::move(expectedLabelsWithValues), NZoom::NAccumulators::EAccumulatorType::Hgram);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, expectedReadManyQueryBase, 1,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, MergesHist) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
            TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 2 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "cpu_usage_hgram"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                }, NMonitoring::EMetricType::HIST
            },
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "cpu_usage_hgram"},
                    {"group", "SAS.000"},
                    {"ctype", "prestable"}
                }, NMonitoring::EMetricType::HIST
            }
        });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    TLabelsList labels1 = {
        {"project", "yasm_yasmagentmetrics"},
        {"service", "yasm"},
        {"cluster", "group_0"},
        {"signal", "cpu_usage_hgram"},
        {"group", "MYT.000"},
        {"ctype", "prod"}
    };
    TVector<NZoom::NHgram::THgram> vec1;
    vec1.emplace_back(NZoom::NHgram::THgram::Ugram({{0, 1, 2}, {1, 2, 5}}));

    TLabelsList labels2 = {
        {"project", "yasm_yasmagentmetrics"},
        {"service", "yasm"},
        {"cluster", "group_0"},
        {"signal", "cpu_usage_hgram"},
        {"group", "SAS.000"},
        {"ctype", "prestable"}
    };
    TVector<NZoom::NHgram::THgram> vec2;
    vec2.emplace_back(NZoom::NHgram::THgram::Ugram({{0, 2, 2}}));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> labelsWithValues;
    labelsWithValues.emplace_back(labels1, std::move(vec1), NMonitoring::EMetricType::HIST);
    labelsWithValues.emplace_back(labels2, std::move(vec2), NMonitoring::EMetricType::HIST);

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram>(
        std::move(labelsWithValues), NZoom::NAccumulators::EAccumulatorType::Hgram);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    TLabelsList expectedLabels = {{"signal", "cpu_usage_hgram"}};
    TVector<NZoom::NHgram::THgram> expectedValues;
    expectedValues.emplace_back(NZoom::NHgram::THgram::Ugram({{0, 1, 3}, {1, 2, 6}}));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> expectedLabelsWithValues;
    expectedLabelsWithValues.emplace_back(expectedLabels, std::move(expectedValues), NMonitoring::EMetricType::HIST);
    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram, true>(
        std::move(expectedLabelsWithValues), NZoom::NAccumulators::EAccumulatorType::Hgram);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, expectedReadManyQueryBase, 2,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, CompressHist) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
                                                       TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 2 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
            {
                    {
                            {
                                    {"project", "yasm_yasmagentmetrics"},
                                    {"service", "yasm"},
                                    {"cluster", "group_0"},
                                    {"signal", "cpu_usage_hgram"},
                                    {"group", "MYT.000"},
                                    {"ctype", "prod"}
                            }, NMonitoring::EMetricType::HIST
                    },
                    {
                            {
                                    {"project", "yasm_yasmagentmetrics"},
                                    {"service", "yasm"},
                                    {"cluster", "group_0"},
                                    {"signal", "cpu_usage_hgram"},
                                    {"group", "SAS.000"},
                                    {"ctype", "prestable"}
                            }, NMonitoring::EMetricType::HIST
                    }
            });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    TLabelsList labels = {
            {"project", "yasm_yasmagentmetrics"},
            {"service", "yasm"},
            {"cluster", "group_0"},
            {"signal", "cpu_usage_hgram"},
            {"group", "MYT.000"},
            {"ctype", "prod"}
    };
    TVector<NZoom::NHgram::THgram> vec;
    NZoom::NHgram::TUgramBuckets buckets;
    for (ui32 i = 0; i < NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT + 5; ++i) {
        buckets.push_back({(double)i, (double)i + 1, 2});
    }
    NZoom::NHgram::TUgramBuckets expectedBuckets;
    expectedBuckets.push_back({0, 8, 16});
    for (ui32 i = 8; i < NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT + 5; ++i) {
        expectedBuckets.push_back({(double)i, (double)i + 1, 2});
    }
    size_t expectedBucketsSizeAfterSerializing = BucketsSizeAfterSerializing(expectedBuckets);
    vec.emplace_back(NZoom::NHgram::THgram::Ugram(std::move(buckets)));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> labelsWithValues;
    labelsWithValues.emplace_back(labels, std::move(vec), NMonitoring::EMetricType::HIST);

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram>(
            std::move(labelsWithValues), NZoom::NAccumulators::EAccumulatorType::Hgram);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    TLabelsList expectedLabels = {{"signal", "cpu_usage_hgram"}};
    TVector<NZoom::NHgram::THgram> expectedValues;
    expectedValues.emplace_back(NZoom::NHgram::THgram::Ugram(std::move(expectedBuckets)));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> expectedLabelsWithValues;
    expectedLabelsWithValues.emplace_back(expectedLabels, std::move(expectedValues), NMonitoring::EMetricType::HIST);
    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram, true> (
            std::move(expectedLabelsWithValues), NZoom::NAccumulators::EAccumulatorType::Hgram);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    for (auto& metric: expectedAggregatedResult->Metrics) {
        NTs::TVariantPoint point;
        for (auto it = metric.TimeSeries->Iterator(); it->Next(&point); ) {
            auto value = point.Get<NTs::NValue::THistogram>();
            EXPECT_EQ(value.Buckets.size(), expectedBucketsSizeAfterSerializing);
        }
    }

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, expectedReadManyQueryBase, 2,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, CompressHistIncludingTwoBuckets) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
                                                       TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 2 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
            {
                    {
                            {
                                    {"project", "yasm_yasmagentmetrics"},
                                    {"service", "yasm"},
                                    {"cluster", "group_0"},
                                    {"signal", "cpu_usage_hgram"},
                                    {"group", "MYT.000"},
                                    {"ctype", "prod"}
                            }, NMonitoring::EMetricType::HIST
                    },
                    {
                            {
                                    {"project", "yasm_yasmagentmetrics"},
                                    {"service", "yasm"},
                                    {"cluster", "group_0"},
                                    {"signal", "cpu_usage_hgram"},
                                    {"group", "SAS.000"},
                                    {"ctype", "prestable"}
                            }, NMonitoring::EMetricType::HIST
                    }
            });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    TLabelsList labels = {
            {"project", "yasm_yasmagentmetrics"},
            {"service", "yasm"},
            {"cluster", "group_0"},
            {"signal", "cpu_usage_hgram"},
            {"group", "MYT.000"},
            {"ctype", "prod"}
    };
    TVector<NZoom::NHgram::THgram> vec;
    NZoom::NHgram::TUgramBuckets buckets;
    for (ui32 i = 0; i < NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT - 1; ++i) {
        buckets.push_back({(double)i, (double)i + 1, 2});
    }
    NZoom::NHgram::TUgramBuckets expectedBuckets;
    expectedBuckets.push_back({0, 2, 4});
    for (ui32 i = 2; i < NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT - 1; ++i) {
        expectedBuckets.push_back({(double)i, (double)i + 1, 2});
    }
    size_t expectedBucketsSizeAfterSerializing = BucketsSizeAfterSerializing(expectedBuckets);
    vec.emplace_back(NZoom::NHgram::THgram::Ugram(std::move(buckets)));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> labelsWithValues;
    labelsWithValues.emplace_back(labels, std::move(vec), NMonitoring::EMetricType::HIST);

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram>(
            std::move(labelsWithValues), NZoom::NAccumulators::EAccumulatorType::Hgram);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    TLabelsList expectedLabels = {{"signal", "cpu_usage_hgram"}};
    TVector<NZoom::NHgram::THgram> expectedValues;
    expectedValues.emplace_back(NZoom::NHgram::THgram::Ugram(std::move(expectedBuckets)));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> expectedLabelsWithValues;
    expectedLabelsWithValues.emplace_back(expectedLabels, std::move(expectedValues), NMonitoring::EMetricType::HIST);
    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram, true> (
            std::move(expectedLabelsWithValues), NZoom::NAccumulators::EAccumulatorType::Hgram);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    for (auto& metric: expectedAggregatedResult->Metrics) {
        NTs::TVariantPoint point;
        for (auto it = metric.TimeSeries->Iterator(); it->Next(&point); ) {
            auto value = point.Get<NTs::NValue::THistogram>();
            EXPECT_EQ(value.Buckets.size(), expectedBucketsSizeAfterSerializing);
        }
    }

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, expectedReadManyQueryBase, 2,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, DontCompressHist) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
                                                       TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 2 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
            {
                    {
                            {
                                    {"project", "yasm_yasmagentmetrics"},
                                    {"service", "yasm"},
                                    {"cluster", "group_0"},
                                    {"signal", "cpu_usage_hgram"},
                                    {"group", "MYT.000"},
                                    {"ctype", "prod"}
                            }, NMonitoring::EMetricType::HIST
                    },
                    {
                            {
                                    {"project", "yasm_yasmagentmetrics"},
                                    {"service", "yasm"},
                                    {"cluster", "group_0"},
                                    {"signal", "cpu_usage_hgram"},
                                    {"group", "SAS.000"},
                                    {"ctype", "prestable"}
                            }, NMonitoring::EMetricType::HIST
                    }
            });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    TLabelsList labels = {
            {"project", "yasm_yasmagentmetrics"},
            {"service", "yasm"},
            {"cluster", "group_0"},
            {"signal", "cpu_usage_hgram"},
            {"group", "MYT.000"},
            {"ctype", "prod"}
    };
    TVector<NZoom::NHgram::THgram> vec;
    NZoom::NHgram::TUgramBuckets buckets;
    for (ui32 i = 0; i < NMonitoring::HISTOGRAM_MAX_BUCKETS_COUNT - 2; ++i) {
        buckets.push_back({(double)i, (double)i + 1, 2});
    }
    NZoom::NHgram::TUgramBuckets expectedBuckets = buckets;
    size_t expectedBucketsSizeAfterSerializing = BucketsSizeAfterSerializing(expectedBuckets);
    vec.emplace_back(NZoom::NHgram::THgram::Ugram(std::move(buckets)));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> labelsWithValues;
    labelsWithValues.emplace_back(labels, std::move(vec), NMonitoring::EMetricType::HIST);

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram>(
            std::move(labelsWithValues), NZoom::NAccumulators::EAccumulatorType::Hgram);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    TLabelsList expectedLabels = {{"signal", "cpu_usage_hgram"}};
    TVector<NZoom::NHgram::THgram> expectedValues;
    expectedValues.emplace_back(NZoom::NHgram::THgram::Ugram(std::move(expectedBuckets)));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> expectedLabelsWithValues;
    expectedLabelsWithValues.emplace_back(expectedLabels, std::move(expectedValues), NMonitoring::EMetricType::HIST);
    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram, true> (
            std::move(expectedLabelsWithValues), NZoom::NAccumulators::EAccumulatorType::Hgram);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    for (auto& metric: expectedAggregatedResult->Metrics) {
        NTs::TVariantPoint point;
        for (auto it = metric.TimeSeries->Iterator(); it->Next(&point); ) {
            auto value = point.Get<NTs::NValue::THistogram>();
            EXPECT_EQ(value.Buckets.size(), expectedBucketsSizeAfterSerializing);
        }
    }

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, expectedReadManyQueryBase, 2,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, MergesLogHistWithHistToHist) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
            TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 2 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "cpu_usage_hgram"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                }, NMonitoring::EMetricType::LOGHIST
            },
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "cpu_usage_hgram"},
                    {"group", "SAS.000"},
                    {"ctype", "prestable"}
                }, NMonitoring::EMetricType::HIST
            }
        });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    TLabelsList labels1 = {
        {"project", "yasm_yasmagentmetrics"},
        {"service", "yasm"},
        {"cluster", "group_0"},
        {"signal", "cpu_usage_hgram"},
        {"group", "MYT.000"},
        {"ctype", "prod"}
    };
    TVector<NZoom::NHgram::THgram> vec1;
    vec1.emplace_back(NZoom::NHgram::THgram::Normal({1, 5, 6}, 5, 0));

    TLabelsList labels2 = {
        {"project", "yasm_yasmagentmetrics"},
        {"service", "yasm"},
        {"cluster", "group_0"},
        {"signal", "cpu_usage_hgram"},
        {"group", "SAS.000"},
        {"ctype", "prestable"}
    };
    TVector<NZoom::NHgram::THgram> vec2;
    vec2.emplace_back(NZoom::NHgram::THgram::Ugram({{0, 1.5, 3}}));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> labelsWithValues;
    labelsWithValues.emplace_back(labels1, std::move(vec1), NMonitoring::EMetricType::LOGHIST);
    labelsWithValues.emplace_back(labels2, std::move(vec2), NMonitoring::EMetricType::HIST);

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram>(
        std::move(labelsWithValues), NZoom::NAccumulators::EAccumulatorType::Hgram);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    TLabelsList expectedLabels = {{"signal", "cpu_usage_hgram"}};
    TVector<NZoom::NHgram::THgram> expectedValues;
    expectedValues.emplace_back(NZoom::NHgram::THgram::Ugram({
        {0, 0, 5},
        {0, 1, 2},
        {1, 1.5, 2},
        {1.5, 2.25, 5},
        {2.25, 3.375, 6},
    }));

    TVector<std::tuple<TLabelsList, TVector<NZoom::NHgram::THgram>, NMonitoring::EMetricType>> expectedLabelsWithValues;
    expectedLabelsWithValues.emplace_back(expectedLabels, std::move(expectedValues), NMonitoring::EMetricType::HIST);
    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<NZoom::NHgram::THgram, true>(
            std::move(expectedLabelsWithValues),
            NZoom::NAccumulators::EAccumulatorType::Hgram);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, expectedReadManyQueryBase, 2,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, ClusterAndService) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
            TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 7 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="cpu_usage_hgram", group="MYT.000|SAS.000"})");
    query.Lookup->Limit = 100;
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
        {
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "cpu_usage_hgram"},
                    {"group", "MYT.000"},
                    {"ctype", "prod"}
                }, NMonitoring::EMetricType::LOGHIST
            },
            {
                {
                    {"project", "yasm_yasmagentmetrics"},
                    {"service", "yasm"},
                    {"cluster", "group_0"},
                    {"signal", "cpu_usage_hgram"},
                    {"group", "SAS.000"},
                    {"ctype", "prestable"}
                }, NMonitoring::EMetricType::HIST
            }
        });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    EXPECT_CALL(*dataSourceMock, Find(FindQueryPartiallyEq(expectedQuery), _, _)).WillOnce(
        Invoke([&](auto, auto handler, auto) {
          handler->OnSuccess(std::move(findResult));
        })
    );

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();
    EXPECT_CALL(*dataSourceMock, ReadMany(ReadQueryServiceEq(expectedReadManyQueryBase, "yasm"), _, _)).WillOnce(
        Invoke([&](auto, auto handler, auto) {
          handler->OnSuccess(std::move(datasourceReadManyResult));
        })
    );
    EXPECT_CALL(*handlerMock, OnSuccess).Times(1);

    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, ReadsGroupByLabels) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
                                                       TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 7 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="counter-instance_tmmv", group="MYT.000"})");
    query.Lookup->Limit = 100;
    query.Lookup->GroupBy = {"geo", "service"};
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
            {
              {
                      {
                              {"project", "yasm_yasmagentmetrics"},
                              {"signal", "counter-instance_tmmv"},
                              {"geo", "vla"},
                              {"service", "yasm"},
                              {"cluster", "group_0"},
                              {"group", "MYT.000"},
                              {"ctype", "prod"}
                      }, NMonitoring::EMetricType::DSUMMARY
              },
              {
                      {
                              {"project", "yasm_yasmagentmetrics"},
                              {"signal", "counter-instance_tmmv"},
                              {"geo", "mon"},
                              {"service", "yasm"},
                              {"cluster", "group_0"},
                              {"group", "MYT.000"},
                              {"ctype", "prod"}
                      }, NMonitoring::EMetricType::DSUMMARY
              }
        });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<double>(
            {
                    {
                            {
                                    {"project", "yasm_yasmagentmetrics"},
                                    {"signal", "counter-instance_tmmv"},
                                    {"geo", "vla"},
                                    {"service", "yasm"},
                                    {"cluster", "group_0"},
                                    {"group", "MYT.000"},
                                    {"ctype", "prod"}
                            },
                            {1.8, 1.4, 7, 8, 0.1, 4},
                            NMonitoring::EMetricType::DSUMMARY
                    },
                    {
                            {
                                    {"project", "yasm_yasmagentmetrics"},
                                    {"signal", "counter-instance_tmmv"},
                                    {"geo", "mon"},
                                    {"service", "yasm"},
                                    {"cluster", "group_0"},
                                    {"group", "MYT.000"},
                                    {"ctype", "prod"}
                            },
                            {1.8, 1.4, 7, 8, 0.2, 5},
                            NMonitoring::EMetricType::DSUMMARY
                    }
            }, NZoom::NAccumulators::EAccumulatorType::Summ);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<double, true>(
            {
                {
                    {
                          {"signal", "counter-instance_tmmv"},
                          {"geo", "vla"},
                          {"service", "yasm"}
                    },
                    {1.8, 1.4, 7, 8, 0.1, 4},
                    NMonitoring::EMetricType::GAUGE
                },
                {
                    {
                            {"signal", "counter-instance_tmmv"},
                            {"geo", "mon"},
                            {"service", "yasm"}
                    },
                    {1.8, 1.4, 7, 8, 0.2, 5},
                    NMonitoring::EMetricType::GAUGE
                }
            },
            NZoom::NAccumulators::EAccumulatorType::Summ);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, expectedReadManyQueryBase, 2,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}

TEST_F(TMetricAggregatorTest, GroupBySignal) {
    auto dataSourceMock = MakeIntrusive<TBaseDataSourceMock>();
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();
    auto expectedReadManyQueryBase = FillQuery<TQuery>("yasm_yasmagentmetrics",
                                                       TInstant::Zero() + YASM_INTERVAL, TInstant::Zero() + 7 * YASM_INTERVAL);
    auto query = AddQuerySelectors<TReadManyQuery>(expectedReadManyQueryBase,
                                                   R"({service="yasm", cluster = "group_*", signal="counter-instance*_tmmv", group="MYT.000"})");
    auto expectedQuery = AddQuerySelectors<TFindQuery>(query,
                                                       R"({service="yasm", cluster = "group_*", signal="counter-instance*_tmmv", group="MYT.000"})");
    query.Lookup->Limit = 100;
    query.Lookup->GroupBy = {"signal"};
    expectedQuery.Limit = 100;

    std::unique_ptr<TFindResult> findResult = std::make_unique<TFindResult>();
    auto [metrics, metricStrPool] = MakeMetricsFromLabels(
            {
              {
                      {
                              {"project", "yasm_yasmagentmetrics"},
                              {"signal", "counter-instance1_tmmv"},
                              {"key", "value1"},
                              {"service", "yasm"},
                              {"cluster", "group_0"},
                              {"group", "MYT.000"},
                              {"ctype", "prod"}
                      }, NMonitoring::EMetricType::DSUMMARY
              },
              {
                      {
                              {"project", "yasm_yasmagentmetrics"},
                              {"signal", "counter-instance1_tmmv"},
                              {"key", "value2"},
                              {"service", "yasm"},
                              {"cluster", "group_0"},
                              {"group", "MYT.000"},
                              {"ctype", "prod"}
                      }, NMonitoring::EMetricType::DSUMMARY
              },
              {
                      {
                              {"project", "yasm_yasmagentmetrics"},
                              {"signal", "counter-instance2_tmmv"},
                              {"key", "value1"},
                              {"service", "yasm"},
                              {"cluster", "group_0"},
                              {"group", "MYT.000"},
                              {"ctype", "prod"}
                      }, NMonitoring::EMetricType::DSUMMARY
              },
              {
                      {
                              {"project", "yasm_yasmagentmetrics"},
                              {"signal", "counter-instance2_tmmv"},
                              {"key", "value2"},
                              {"service", "yasm"},
                              {"cluster", "group_0"},
                              {"group", "MYT.000"},
                              {"ctype", "prod"}
                      }, NMonitoring::EMetricType::DSUMMARY
              }
        });
    findResult->Strings = std::move(metricStrPool);
    findResult->Metrics = std::move(metrics);
    findResult->TotalCount = findResult->Metrics.size();

    std::unique_ptr<TReadManyResult> datasourceReadManyResult = std::make_unique<TReadManyResult>();

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<double>(
            {
                    {
                            {
                                    {"project", "yasm_yasmagentmetrics"},
                                    {"signal", "counter-instance1_tmmv"},
                                    {"key", "value1"},
                                    {"service", "yasm"},
                                    {"cluster", "group_0"},
                                    {"group", "MYT.000"},
                                    {"ctype", "prod"}
                            },
                            {1.8, 1.4, 7, 8, 0.1, 4},
                            NMonitoring::EMetricType::DSUMMARY
                    },
                    {
                            {
                                    {"project", "yasm_yasmagentmetrics"},
                                    {"signal", "counter-instance1_tmmv"},
                                    {"key", "value2"},
                                    {"service", "yasm"},
                                    {"cluster", "group_0"},
                                    {"group", "MYT.000"},
                                    {"ctype", "prod"}
                            },
                            {1.8, 1.4, 7, 8, 0.1, 4},
                            NMonitoring::EMetricType::DSUMMARY
                    },
                    {
                            {
                                    {"project", "yasm_yasmagentmetrics"},
                                    {"signal", "counter-instance2_tmmv"},
                                    {"key", "value1"},
                                    {"service", "yasm"},
                                    {"cluster", "group_0"},
                                    {"group", "MYT.000"},
                                    {"ctype", "prod"}
                            },
                            {1.8, 1.4, 7, 8, 0.2, 5},
                            NMonitoring::EMetricType::DSUMMARY
                    },
                    {
                            {
                                    {"project", "yasm_yasmagentmetrics"},
                                    {"signal", "counter-instance2_tmmv"},
                                    {"key", "value2"},
                                    {"service", "yasm"},
                                    {"cluster", "group_0"},
                                    {"group", "MYT.000"},
                                    {"ctype", "prod"}
                            },
                            {1.8, 1.4, 7, 8, 0.2, 5},
                            NMonitoring::EMetricType::DSUMMARY
                    }
            }, NZoom::NAccumulators::EAccumulatorType::Summ);
    datasourceReadManyResult->Strings = std::move(readManyStrPool);
    datasourceReadManyResult->Metrics = std::move(readManyMetrics);

    std::unique_ptr<TReadManyResult> expectedAggregatedResult = std::make_unique<TReadManyResult>();
    auto [aggregatedMetrics, aggregatedStrPool] = MakeMetricsWithTimeseries<double, true>(
            {
                {
                    {
                          {"signal", "counter-instance1_tmmv"},
                    },
                    {3.6, 2.8, 14, 16, 0.2, 8},
                    NMonitoring::EMetricType::GAUGE
                },
                {
                    {
                            {"signal", "counter-instance2_tmmv"},
                    },
                    {3.6, 2.8, 14, 16, 0.4, 10},
                    NMonitoring::EMetricType::GAUGE
                }
            },
            NZoom::NAccumulators::EAccumulatorType::Summ);
    expectedAggregatedResult->Strings = std::move(aggregatedStrPool);
    expectedAggregatedResult->Metrics = std::move(aggregatedMetrics);

    ExpectSuccessfulCalls(handlerMock, dataSourceMock, expectedQuery, findResult, expectedReadManyQueryBase, 4,
                          datasourceReadManyResult, expectedAggregatedResult);
    CreateActorAndDispatchEvents(handlerMock, dataSourceMock, std::move(query));
}
