#include <solomon/services/dataproxy/lib/datasource/yasm/selectors.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NSolomon::NDataProxy;
using namespace NSolomon::NYasm;

THashMap<TString, TVector<TString>> CreateMapping() {
    THashMap<TString, TVector<TString>> map;
    map["ASEARCH"] = {"IVA.005", "IVA.007"};
    map["METAGROUP"] = {"GROUP"};

    return map;
}

class TTestShardConfig: public IShardConfig {
private:
    ui64 GetShardCount(TStringBuf itype) const override {
        if (itype == "yasm"sv) {
            return 1;
        } else {
            return 22;
        }
    }
};

class TTransformationTest: public ::testing::Test {
public:
    TTransformationTest()
        : SelectorsTransformer{std::make_shared<TTestShardConfig>()}
    {
    }

    TYasmSelectorsTransformer SelectorsTransformer;
};

TEST_F(TTransformationTest, GroupWithoutSignal) {
    TSelectors selectors;
    selectors.Add("hosts", "IVA.005");
    selectors.Add("geo", "msk");

    auto hosts = THostsLabel::FromSelectors(selectors, [](auto) { return false; });

    SelectorsTransformer.FillHostOrGroup(selectors, hosts);
    SelectorsTransformer.FillClusterService(selectors, "yasm", hosts);

    ASSERT_TRUE(selectors.Has("service"));
    EXPECT_EQ(selectors.Find("service")->Pattern(), "yasm");

    EXPECT_FALSE(selectors.Has("hosts"));
    ASSERT_TRUE(selectors.Has("group"));
    EXPECT_TRUE(selectors.Find("group")->Type() == EMatcherType::EXACT);
    EXPECT_EQ(selectors.Find("group")->Pattern(), "IVA.005");

    ASSERT_TRUE(selectors.Has("geo"));
    EXPECT_EQ(selectors.Find("geo")->Pattern(), "msk");

    ASSERT_TRUE(selectors.Has("cluster"));
    EXPECT_EQ(selectors.Find("cluster")->Pattern(), "group_*");
}

TEST_F(TTransformationTest, BothGroupAndSignal) {
    TSelectors selectors;
    selectors.Add("hosts", "GROUP"); // I wanted to test IVA.005, but this won't work because of the bug: see SOLOMON-7974
    selectors.Add("signal", "counter_mmmm");
    selectors.Add("geo", "msk");

    auto hosts = THostsLabel::FromSelectors(selectors, [](auto) { return false; });

    SelectorsTransformer.FillHostOrGroup(selectors, hosts);
    SelectorsTransformer.FillClusterService(selectors, "yasm", hosts);

    ASSERT_TRUE(selectors.Has("service"));
    EXPECT_EQ(selectors.Find("service")->Pattern(), "yasm");

    EXPECT_FALSE(selectors.Has("hosts"));
    ASSERT_TRUE(selectors.Has("group"));
    EXPECT_TRUE(selectors.Find("group")->Type() == EMatcherType::EXACT);
    EXPECT_EQ(selectors.Find("group")->Pattern(), "GROUP");

    ASSERT_TRUE(selectors.Has("geo"));
    EXPECT_EQ(selectors.Find("geo")->Pattern(), "msk");

    ASSERT_TRUE(selectors.Has("signal"));
    EXPECT_EQ(selectors.Find("signal")->Pattern(), "counter_mmmm");

    ASSERT_TRUE(selectors.Has("cluster"));
    EXPECT_EQ(selectors.Find("cluster")->Pattern(), "group_0");
}

TEST_F(TTransformationTest, HostWithoutSignal) {
    TSelectors selectors;
    selectors.Add("hosts", "sas-005.search.yandex.net");
    selectors.Add("geo", "msk");

    auto hosts = THostsLabel::FromSelectors(selectors, [](auto) { return false; });

    SelectorsTransformer.FillHostOrGroup(selectors, hosts);
    SelectorsTransformer.FillClusterService(selectors, "yasm", hosts);

    ASSERT_TRUE(selectors.Has("service"));
    EXPECT_EQ(selectors.Find("service")->Pattern(), "yasm");

    EXPECT_FALSE(selectors.Has("hosts"));
    EXPECT_FALSE(selectors.Has("group"));

    ASSERT_TRUE(selectors.Has("host"));
    EXPECT_TRUE(selectors.Find("host")->Type() == EMatcherType::EXACT);
    EXPECT_EQ(selectors.Find("host")->Pattern(), "sas-005.search.yandex.net");

    ASSERT_TRUE(selectors.Has("geo"));
    EXPECT_EQ(selectors.Find("geo")->Pattern(), "msk");

    ASSERT_TRUE(selectors.Has("cluster"));
    EXPECT_EQ(selectors.Find("cluster")->Pattern(), "host_*");
}

TEST_F(TTransformationTest, HostWithSignal) {
    TSelectors selectors;
    selectors.Add("hosts", "sas-005.search.yandex.net");
    selectors.Add("signal", "counter_mmmm");
    selectors.Add("geo", "msk");

    auto hosts = THostsLabel::FromSelectors(selectors, [](auto) { return false; });

    SelectorsTransformer.FillHostOrGroup(selectors, hosts);
    SelectorsTransformer.FillClusterService(selectors, "yasm", hosts);

    ASSERT_TRUE(selectors.Has("service"));
    EXPECT_EQ(selectors.Find("service")->Pattern(), "yasm");

    EXPECT_FALSE(selectors.Has("hosts"));
    EXPECT_FALSE(selectors.Has("group"));

    ASSERT_TRUE(selectors.Has("host"));
    EXPECT_TRUE(selectors.Find("host")->Type() == EMatcherType::EXACT);
    EXPECT_EQ(selectors.Find("host")->Pattern(), "sas-005.search.yandex.net");

    ASSERT_TRUE(selectors.Has("geo"));
    EXPECT_EQ(selectors.Find("geo")->Pattern(), "msk");

    ASSERT_TRUE(selectors.Has("signal"));
    EXPECT_EQ(selectors.Find("signal")->Pattern(), "counter_mmmm");

    ASSERT_TRUE(selectors.Has("cluster"));
    EXPECT_EQ(selectors.Find("cluster")->Pattern(), "host_0");
}
