#include "yasm.h"
#include "yasm_handler.h"
#include "events.h"

#include <solomon/libs/cpp/trace/trace.h>
#include <solomon/libs/cpp/yasm/constants/client_id.h>
#include <solomon/libs/cpp/yasm/constants/labels.h>
#include <solomon/services/dataproxy/lib/datasource/datasource.h>
#include <solomon/services/dataproxy/lib/datasource/top/top.h>

#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>

#include <infra/yasm/common/groups/metagroup_groups.h>
#include <infra/yasm/stockpile_client/common/base_types.h>

#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_set.h>

using namespace NActors;
using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

bool IsSolomonAware(const TLabelValuesQuery& query) {
    if (auto it = query.Selectors.Find(NLabels::LABEL_CLUSTER); it != query.Selectors.end() && it->Pattern() != "*") {
        return true;
    }

    if (auto it = query.Selectors.Find(NLabels::LABEL_SERVICE); it != query.Selectors.end() && it->Pattern() != "*") {
        return true;
    }

    return false;
}

class THostToHostsHandler: public IResultHandler<TReadManyResult> {
public:
    explicit THostToHostsHandler(IResultHandlerPtr<TReadManyResult> handler)
        : OriginalHandler_{std::move(handler)}
    {
    }

private:
    void OnSuccess(std::unique_ptr<TReadManyResult> res) override {
        auto hosts = res->Strings.Put(NLabels::LABEL_HOSTS);
        auto host = res->Strings.Put(NLabels::LABEL_HOST);

        for (auto& metric: res->Metrics) {
            for (auto& [key, value]: metric.Meta.Labels) {
                if (key == host) {
                    key = hosts;
                }
            }
        }

        OriginalHandler_->OnSuccess(std::move(res));
    }

    void OnError(TString&& project, EDataSourceStatus status, TString&& message) override {
        OriginalHandler_->OnError(std::move(project), status, std::move(message));
    }

private:
    IResultHandlerPtr<TReadManyResult> OriginalHandler_;
};

using ::NYasm::NGroups::TMetagroupGroupsConfig;

class TYasmDataSource: public IDataSource {
public:
    TYasmDataSource(
            TActorRuntime& runtime,
            TActorId yasmDataSourceActor,
            IDataSourcePtr baseDataSource)
        : Runtime_(runtime)
        , YasmDataSourceActor_(yasmDataSourceActor)
        , BaseDataSource_(std::move(baseDataSource))
    {
    }

private:
    bool CanHandle(const TQuery& query) const override {
        if (query.ClientId.StartsWith(NYasm::YASM_COLLECTOR_CLIENT_ID_PREFIX)) {
            return false;
        }

        return (query.Project.StartsWith(NHistDb::NStockpile::STOCKPILE_YASM_PROJECTS_PREFIX)) &&
               BaseDataSource_->CanHandle(query);
    }

    void ReadMany(TReadManyQuery query, IResultHandlerPtr<TReadManyResult> handler, TSpanId traceCtx) const override {
        if (query.Lookup) {
            const auto& selectors = query.Lookup->Selectors;
            auto signalIt = selectors.Find(NYasm::LABEL_SIGNAL);
            if (signalIt == selectors.end()) {
                TRACING_SPAN_END(traceCtx);
                handler->OnError(std::move(query.Project), EDataSourceStatus::BAD_REQUEST, "signal must be specified");
                return;
            }

            auto hostsIt = selectors.Find(NLabels::LABEL_HOSTS);
            if (hostsIt != selectors.end() && !hostsIt->IsExact()) {
                query.Lookup->Selectors.Add(NLabels::LABEL_HOST, hostsIt->Pattern(), hostsIt->Negative());
                query.Lookup->Selectors.Remove(NLabels::LABEL_HOSTS);
                handler = MakeIntrusive<THostToHostsHandler>(std::move(handler));
            }

            for (const auto& selector: selectors) {
                if (!selector.IsExact() && selector.Key() != NLabels::LABEL_CLUSTER && selector.Key() != NLabels::LABEL_SERVICE) {
                    auto& groupBy = query.Lookup->GroupBy;
                    if (std::find(groupBy.begin(), groupBy.end(), selector.Key()) == groupBy.end()) {
                        query.Lookup->GroupBy.emplace_back(selector.Key());
                    }
                }
            }

            auto hostIt = selectors.Find(NLabels::LABEL_HOST);
            auto groupIt = selectors.Find(NLabels::LABEL_GROUP);
            if (hostIt != selectors.end() && groupIt != selectors.end() && hostIt->Pattern() != NYasm::AGGREGATED_MARKER) {
                TRACING_SPAN_END(traceCtx);
                handler->OnError(
                        std::move(query.Project),
                        EDataSourceStatus::BAD_REQUEST,
                        "host and group cannot be specified both at the same time");
                return;
            }
        }

        if (auto top = GetTopOperation(query)) {
            handler = MakeTopHandler(std::move(handler), *top);

            // don't propagate top to stockpile
            EraseIf(query.Operations, [](const auto& op) {
                return op.has_top();
            });
        }

        if (query.ResolvedKeys) {
            // requesting by resolved keys does not require any yasm-specific processing
            BaseDataSource_->ReadMany(std::move(query), std::move(handler), std::move(traceCtx));
        } else {
            Runtime_.ActorSystem().Send(new IEventHandle{
                    YasmDataSourceActor_, {},
                    new TYasmDataSourceEvents::TRequest{std::move(query), std::move(handler)},
                    0, 0, nullptr,
                    std::move(traceCtx)
            });
        }
    }

    void Find(TFindQuery query, IResultHandlerPtr<TFindResult> handler, TSpanId traceCtx) const override {
        Runtime_.ActorSystem().Send(new IEventHandle{
                YasmDataSourceActor_, {},
                new TYasmDataSourceEvents::TRequest{std::move(query), std::move(handler)},
                0, 0, nullptr,
                std::move(traceCtx)
        });
    }

    void ResolveOne(TResolveOneQuery query, IResultHandlerPtr<TResolveOneResult> handler, TSpanId traceCtx) const override {
        // TODO: SelectorsTransformer->Transform(query.Selectors);
        BaseDataSource_->ResolveOne(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void ResolveMany(TResolveManyQuery query, IResultHandlerPtr<TResolveManyResult> handler, TSpanId traceCtx) const override {
        // TODO: SelectorsTransformer->Transform(query.Selectors);
        BaseDataSource_->ResolveMany(std::move(query), std::move(handler), std::move(traceCtx));
    }

    void MetricNames(TMetricNamesQuery query, IResultHandlerPtr<TMetricNamesResult> handler, TSpanId traceCtx) const override {
        Runtime_.ActorSystem().Send(new IEventHandle{
                YasmDataSourceActor_, {},
                new TYasmDataSourceEvents::TRequest{std::move(query), std::move(handler)},
                0, 0, nullptr,
                std::move(traceCtx)
        });
    }

    void LabelKeys(TLabelKeysQuery query, IResultHandlerPtr<TLabelKeysResult> handler, TSpanId traceCtx) const override {
        auto h = YasmHandler(std::move(handler));
        Runtime_.ActorSystem().Send(new IEventHandle{
                YasmDataSourceActor_, {},
                new TYasmDataSourceEvents::TRequest{std::move(query), std::move(h)},
                0, 0, nullptr,
                std::move(traceCtx)
        });
    }

    void LabelValues(TLabelValuesQuery query, IResultHandlerPtr<TLabelValuesResult> handler, TSpanId traceCtx) const override {
        if (!IsSolomonAware(query)) {
            handler = YasmHandler(std::move(handler));
        }

        Runtime_.ActorSystem().Send(new IEventHandle{
                YasmDataSourceActor_, {},
                new TYasmDataSourceEvents::TRequest{std::move(query), std::move(handler)},
                0, 0, nullptr,
                std::move(traceCtx)
        });
    }

    void UniqueLabels(TUniqueLabelsQuery query, IResultHandlerPtr<TUniqueLabelsResult> handler, TSpanId traceCtx) const override {
        Runtime_.ActorSystem().Send(new IEventHandle{
                YasmDataSourceActor_, {},
                new TYasmDataSourceEvents::TRequest{std::move(query), std::move(handler)},
                0, 0, nullptr,
                std::move(traceCtx)
        });
    }

    void ReadOne(TReadOneQuery query, IResultHandlerPtr<TReadOneResult> handler, TSpanId traceCtx) const override {
        Runtime_.ActorSystem().Send(new IEventHandle{
                YasmDataSourceActor_, /* sender */ {},
                new TYasmDataSourceEvents::TRequest{std::move(query), std::move(handler)},
                0, 0, nullptr,
                std::move(traceCtx)
        });
    }

    void WaitUntilInitialized() const override {
        BaseDataSource_->WaitUntilInitialized();
    }

private:
    TActorRuntime& Runtime_;
    TActorId YasmDataSourceActor_;
    IDataSourcePtr BaseDataSource_;
};

} // namespace

TIntrusivePtr<IDataSource> YasmDataSource(
        TActorRuntime& runtime,
        NActors::TActorId yasmDataSourceActor,
        IDataSourcePtr yasmDataSource)
{
    return MakeIntrusive<TYasmDataSource>(
            runtime,
            yasmDataSourceActor,
            std::move(yasmDataSource));
}

} // namespace NSolomon::NDataProxy
