#include "yasm_common.h"
#include <util/generic/algorithm.h>

namespace NSolomon::NDataProxy {
namespace {

TInstant FloorToGrid(TInstant ts, TDuration grid) {
    ui64 tsValue = ts.GetValue();
    return TInstant::FromValue(tsValue - (tsValue % grid.GetValue()));
}

TInstant CeilToGrid(TInstant ts, TDuration grid) {
    ui64 gridValue = grid.GetValue();
    ui64 tsValue = ts.GetValue();
    return TInstant::FromValue(((tsValue + gridValue - 1) / gridValue) * gridValue);
}

} // namespace

TTimeRange ExtractAndRoundTimeRange(const TReadManyQuery& query, TDuration resolution) {
    TTimeRange result{
        .From = FloorToGrid(query.Time.From, resolution),
        .To = CeilToGrid(query.Time.To, resolution),
    };
    DP_ENSURE(result.From < result.To,
              "aligned time range is invalid, from (" << result.From << ") >= to (" << result.To
              << "), grid is " << resolution);

    return result;
}

void FillBaseQueryFields(const TQuery& source, TQuery& target) {
    target.Project = source.Project;
    target.Time = source.Time;
    target.ForceReplicaRead = source.ForceReplicaRead;
    target.Deadline = source.Deadline;
    target.ShortTermStorage = source.ShortTermStorage;
}

TDuration ExtractAndAdjustResolution(TReadManyQuery& query) {
    auto downsamplingOpIt = FindIf(query.Operations, [](const auto& op) {
        return op.type_case() == yandex::solomon::math::Operation::kDownsampling;
    });

    auto minResolution = query.IsYasmSts() ? NYasm::YASM_STS_INTERVAL : NYasm::YASM_INTERVAL;
    if (downsamplingOpIt != query.Operations.end()) {
        ui64 downsamplingMillis = downsamplingOpIt->downsampling().grid_millis();
        if (downsamplingMillis < minResolution.MilliSeconds()) {
            downsamplingOpIt->mutable_downsampling()->set_grid_millis(minResolution.MilliSeconds());
            return minResolution;
        }

        return TDuration::MilliSeconds(downsamplingMillis);
    }
    return minResolution;
}

} // namespace NSolomon::NDataProxy
