#pragma once

#include <solomon/services/dataproxy/lib/cluster_id/cluster_id.h>
#include <solomon/services/dataproxy/lib/event_slots.h>
#include <solomon/services/dataproxy/lib/shard/shard_info.h>

#include <solomon/libs/cpp/actors/fwd.h>
#include <solomon/libs/cpp/clients/memstore/rpc.h>
#include <solomon/libs/cpp/actors/events/events.h>

#include <library/cpp/actors/core/event_local.h>

namespace NSolomon::NDataProxy {

class TMemStoreShardEvents: private TEventSlot<EEventSpace::DataProxy, ES_MEMSTORE_SHARD> {
    enum {
        FindReq = SpaceBegin,
        FindResp,
        LabelKeysReq,
        LabelKeysResp,
        LabelValuesReq,
        LabelValuesResp,
        UniqueLabelsReq,
        UniqueLabelsResp,
        ReadOneReq,
        ReadOneResp,
        ReadManyReq,
        ReadManyResp,
        Error,
        UpdateLocation,
        End,
    };
    static_assert(End < SpaceEnd, "too many event types");

    template <typename TMessage, ui32 Type>
    struct TRequest: public NActors::TEventLocal<TRequest<TMessage, Type>, Type> {
        TMessage Message;
        TInstant Deadline;

        TRequest(TMessage message, TInstant deadline) noexcept
            : Message{std::move(message)}
            , Deadline{deadline}
        {
        }
    };

    template <typename TMessage, ui32 Type>
    struct TResponse: public NActors::TEventLocal<TResponse<TMessage, Type>, Type> {
        TShardId ShardId;
        TClusterId ClusterId;
        TMessage Message;

        TResponse(TShardId shardId, TClusterId clusterId, TMessage message) noexcept
            : ShardId{shardId}
            , ClusterId{clusterId}
            , Message{std::move(message)}
        {
        }
    };

public:
    /**
     * Data requests and responses.
     */
    using TFindReq = TRequest<yandex::monitoring::memstore::FindRequest, FindReq>;
    using TFindResp = TResponse<yandex::monitoring::memstore::FindResponse, FindResp>;

    using TLabelKeysReq = TRequest<yandex::monitoring::memstore::LabelKeysRequest, LabelKeysReq>;
    using TLabelKeysResp = TResponse<yandex::monitoring::memstore::LabelKeysResponse, LabelKeysResp>;

    using TLabelValuesReq = TRequest<yandex::monitoring::memstore::LabelValuesRequest, LabelValuesReq>;
    using TLabelValuesResp = TResponse<yandex::monitoring::memstore::LabelValuesResponse, LabelValuesResp>;

    using TUniqueLabelsReq = TRequest<yandex::monitoring::memstore::UniqueLabelsRequest, UniqueLabelsReq>;
    using TUniqueLabelsResp = TResponse<yandex::monitoring::memstore::UniqueLabelsResponse, UniqueLabelsResp>;

    using TReadOneReq = TRequest<yandex::monitoring::memstore::ReadOneRequest, ReadOneReq>;
    using TReadOneResp = TResponse<yandex::monitoring::memstore::ReadOneResponse, ReadOneResp>;

    using TReadManyReq = TRequest<yandex::monitoring::memstore::ReadManyRequest, ReadManyReq>;
    using TReadManyResp = TResponse<yandex::monitoring::memstore::ReadManyResponse, ReadManyResp>;

    /**
     * Generic error resulting event.
     */
    struct TError: public NActors::TEventLocal<TError, Error> {
        TShardId ShardId;
        TClusterId ClusterId;
        grpc::StatusCode StatusCode;
        TString Message;

        TError(TShardId shardId, TClusterId clusterId, grpc::StatusCode statusCode, TString message) noexcept
            : ShardId{shardId}
            , ClusterId{clusterId}
            , StatusCode{statusCode}
            , Message{std::move(message)}
        {
        }
    };

    /**
     * Update shard location after it was moved from one node to another.
     * This event will be processed only if it was sent by shard's parent (cluster actor).
     */
    struct TUpdateLocation: public NActors::TEventLocal<TUpdateLocation, UpdateLocation> {
        TString Address;

        explicit TUpdateLocation(TString address) noexcept
            : Address{std::move(address)}
        {
        }
    };
};

std::unique_ptr<NActors::IActor> MemStoreShard(
    TClusterId clusterId,
    TShardId shardId,
    TString shardLocation,
    std::shared_ptr<NMemStore::IMemStoreClusterRpc> rpc);

} // namespace NSolomon::NDataProxy
