#include "fake_node.h"

#include <solomon/services/dataproxy/lib/memstore/cluster.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NActors;
using namespace NSolomon;
using namespace NDataProxy;

namespace {

TActorId FindShard(const std::vector<TMemStoreShard>& shards, TShardId id) {
    auto it = std::find_if(shards.begin(), shards.end(), [id](const auto& pair) {
        return pair.Id == id;
    });
    return it == shards.end() ? TActorId{} : it->ActorId;
}

} // namespace

class TMemStoreClusterTest: public ::testing::Test {
protected:
    void SetUp() override {
        ActorRuntime = TTestActorRuntime::CreateInited(1, false, true);
        ActorRuntime->WaitForBootstrap();
        EdgeId = ActorRuntime->AllocateEdgeActor();
    }

    void TearDown() override {
        ActorRuntime.Reset();
    }

    template <typename TEvent>
    void Send(TActorId actorId, THolder<TEvent> event) {
        ActorRuntime->Send(actorId, EdgeId, std::move(event));
    }

    template <typename TEvent>
    typename TEvent::TPtr Receive() {
        return ActorRuntime->GrabEdgeEvent<TEvent>(EdgeId, TDuration::Seconds(3));
    }

public:
    THolder<TTestActorRuntime> ActorRuntime;
    TActorId EdgeId;
    std::shared_ptr<NMemStore::IMemStoreClusterRpc> Rpc;
};

TShardSelector ParseShardSelector(TStringBuf str) {
    TSelectors selectors = ParseSelectors(str);
    auto projectIt = selectors.Find("project");
    Y_ENSURE(projectIt != selectors.end(), "no project selector in " << str);
    Y_ENSURE(projectIt->IsExact(), "project selector must be exact, but was: " << *projectIt);

    auto clusterIt = selectors.Find("cluster");
    auto serviceIt = selectors.Find("service");

    return {
        TString{projectIt->Pattern()},
        clusterIt == selectors.end() ? TSelector{"cluster", AnyMatcher(), false} : *clusterIt,
        serviceIt == selectors.end() ? TSelector{"service", AnyMatcher(), false} : *serviceIt,
    };
}

TEST_F(TMemStoreClusterTest, FindShards) {
    TFakeMemStoreNode nodeA{"node-a"};
    nodeA.AddShard(1, "solomon", "production", "fetcher");
    nodeA.AddShard(2, "solomon", "production", "coremon");
    nodeA.AddShard(3, "solomon", "production", "gateway");

    TFakeMemStoreNode nodeB{"node-b"};
    nodeB.AddShard(4, "solomon", "prestable", "fetcher");
    nodeB.AddShard(5, "solomon", "prestable", "coremon");
    nodeB.AddShard(6, "solomon", "prestable", "gateway");

    TFakeMemStoreNode nodeC{"node-c"};
    nodeC.AddShard(7, "solomon", "testing", "fetcher");
    nodeC.AddShard(8, "solomon", "testing", "coremon");
    nodeC.AddShard(9, "solomon", "testing", "gateway");

    TClusterId clusterId{EDc::Myt, EReplica::R0};
    auto cluster = MemStoreCluster({
        .ClusterId = clusterId,
        .Addresses = {"node-a", "node-b"},
        .Rpc = MakeFakeRpc({&nodeA, &nodeB, &nodeC})
    });

    auto clusterActorId = ActorRuntime->Register(cluster.release());
    ActorRuntime->WaitForBootstrap();
    ActorRuntime->AdvanceCurrentTime(TDuration::Minutes(1));

    {
        Send(clusterActorId, MakeHolder<TMemStoreClusterEvents::TFindShardsReq>(
                ParseShardSelector("{project=solomon, cluster=production}")));
        auto resp = Receive<TMemStoreClusterEvents::TFindShardsResp>();

        ASSERT_TRUE(resp);
        EXPECT_EQ(clusterId, resp->Get()->ClusterId);

        ASSERT_EQ(3u, resp->Get()->Shards.size());
        EXPECT_NE(TActorId{}, FindShard(resp->Get()->Shards, 1));
        EXPECT_NE(TActorId{}, FindShard(resp->Get()->Shards, 2));
        EXPECT_NE(TActorId{}, FindShard(resp->Get()->Shards, 3));
    }
    {
        Send(clusterActorId, MakeHolder<TMemStoreClusterEvents::TFindShardsReq>(
                ParseShardSelector("{project=solomon, service=fetcher}")));
        auto resp = Receive<TMemStoreClusterEvents::TFindShardsResp>();

        ASSERT_TRUE(resp);
        EXPECT_EQ(clusterId, resp->Get()->ClusterId);

        ASSERT_EQ(2u, resp->Get()->Shards.size());
        EXPECT_NE(TActorId{}, FindShard(resp->Get()->Shards, 1));
        EXPECT_NE(TActorId{}, FindShard(resp->Get()->Shards, 4));
    }
    {
        Send(clusterActorId, MakeHolder<TMemStoreClusterEvents::TFindShardsReq>(
                ParseShardSelector("{project=solomon, service=fetcher, cluster=staging}")));
        auto resp = Receive<TMemStoreClusterEvents::TFindShardsResp>();

        ASSERT_TRUE(resp);
        EXPECT_EQ(clusterId, resp->Get()->ClusterId);
        ASSERT_TRUE(resp->Get()->Shards.empty());
    }
}
